# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    Age,
    Range,
    Timing,
    BackboneElement,
    CodeableReference,
    Annotation,
    DomainResource,
)


class ProcedurePerformer(BackboneElement):
    """
    Indicates who or what performed the procedure and how they were involved.
    """

    function: Optional[CodeableConcept] = Field(
        description="Type of performance",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Who performed the procedure",
        default=None,
    )
    onBehalfOf: Optional[Reference] = Field(
        description="Organization the device or practitioner was acting for",
        default=None,
    )
    period: Optional[Period] = Field(
        description="When the performer performed the procedure",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "onBehalfOf",
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ProcedureFocalDevice(BackboneElement):
    """
    A device that is implanted, removed or otherwise manipulated (calibration, battery replacement, fitting a prosthesis, attaching a wound-vac, etc.) as a focal portion of the Procedure.
    """

    action: Optional[CodeableConcept] = Field(
        description="Kind of change to device",
        default=None,
    )
    manipulated: Optional[Reference] = Field(
        description="Device that was changed",
        default=None,
    )

    @field_validator(
        *(
            "manipulated",
            "action",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Procedure(DomainResource):
    """
    An action that is or was performed on or for a patient, practitioner, device, organization, or location. For example, this can be a physical intervention on a patient like an operation, or less invasive like long term services, counseling, or hypnotherapy.  This can be a quality or safety inspection for a location, organization, or device.  This can be an accreditation procedure on a practitioner for licensing.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Procedure"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Identifiers for this procedure",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="A request for this procedure",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of referenced event",
        default=None,
    )
    status: Optional[Code] = Field(
        description="preparation | in-progress | not-done | on-hold | stopped | completed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableConcept] = Field(
        description="Reason for current status",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Classification of the procedure",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Identification of the procedure",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Individual or entity the procedure was performed on",
        default=None,
    )
    focus: Optional[Reference] = Field(
        description="Who is the target of the procedure when it is not the subject of record only",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="The Encounter during which this Procedure was created",
        default=None,
    )
    occurrenceDateTime: Optional[DateTime] = Field(
        description="When the procedure occurred or is occurring",
        default=None,
    )
    occurrenceDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDateTime extensions",
        default=None,
        alias="_occurrenceDateTime",
    )
    occurrencePeriod: Optional[Period] = Field(
        description="When the procedure occurred or is occurring",
        default=None,
    )
    occurrenceString: Optional[String] = Field(
        description="When the procedure occurred or is occurring",
        default=None,
    )
    occurrenceString_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceString extensions",
        default=None,
        alias="_occurrenceString",
    )
    occurrenceAge: Optional[Age] = Field(
        description="When the procedure occurred or is occurring",
        default=None,
    )
    occurrenceRange: Optional[Range] = Field(
        description="When the procedure occurred or is occurring",
        default=None,
    )
    occurrenceTiming: Optional[Timing] = Field(
        description="When the procedure occurred or is occurring",
        default=None,
    )
    recorded: Optional[DateTime] = Field(
        description="When the procedure was first captured in the subject\u0027s record",
        default=None,
    )
    recorded_ext: Optional[Element] = Field(
        description="Placeholder element for recorded extensions",
        default=None,
        alias="_recorded",
    )
    recorder: Optional[Reference] = Field(
        description="Who recorded the procedure",
        default=None,
    )
    reportedBoolean: Optional[Boolean] = Field(
        description="Reported rather than primary record",
        default=None,
    )
    reportedBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for reportedBoolean extensions",
        default=None,
        alias="_reportedBoolean",
    )
    reportedReference: Optional[Reference] = Field(
        description="Reported rather than primary record",
        default=None,
    )
    performer: Optional[List[ProcedurePerformer]] = Field(
        description="Who performed the procedure and what they did",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where the procedure happened",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="The justification that the procedure was performed",
        default=None,
    )
    bodySite: Optional[List[CodeableConcept]] = Field(
        description="Target body sites",
        default=None,
    )
    outcome: Optional[CodeableConcept] = Field(
        description="The result of procedure",
        default=None,
    )
    report: Optional[List[Reference]] = Field(
        description="Any report resulting from the procedure",
        default=None,
    )
    complication: Optional[List[CodeableReference]] = Field(
        description="Complication following the procedure",
        default=None,
    )
    followUp: Optional[List[CodeableConcept]] = Field(
        description="Instructions for follow up",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional information about the procedure",
        default=None,
    )
    focalDevice: Optional[List[ProcedureFocalDevice]] = Field(
        description="Manipulated, implanted, or removed device",
        default=None,
    )
    used: Optional[List[CodeableReference]] = Field(
        description="Items used during procedure",
        default=None,
    )
    supportingInfo: Optional[List[Reference]] = Field(
        description="Extra information relevant to the procedure",
        default=None,
    )
    resourceType: Literal["Procedure"] = Field(
        description=None,
        default="Procedure",
    )

    @property
    def occurrence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurrence",
        )

    @property
    def reported(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="reported",
        )

    @field_validator(
        *(
            "supportingInfo",
            "used",
            "focalDevice",
            "note",
            "followUp",
            "complication",
            "report",
            "outcome",
            "bodySite",
            "reason",
            "location",
            "performer",
            "recorder",
            "recorded",
            "encounter",
            "focus",
            "subject",
            "code",
            "category",
            "statusReason",
            "status",
            "partOf",
            "basedOn",
            "instantiatesUri",
            "instantiatesCanonical",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("performer",), mode="after", check_fields=None)
    @classmethod
    def FHIR_prc_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="onBehalfOf.exists() and actor.resolve().exists() implies actor.resolve().where($this is Practitioner or $this is PractitionerRole).empty()",
            human="Procedure.performer.onBehalfOf can only be populated when performer.actor isn't Practitioner or PractitionerRole",
            key="prc-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurrence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period, String, Age, Range, Timing],
            field_name_base="occurrence",
            required=False,
        )

    @model_validator(mode="after")
    def reported_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, Reference],
            field_name_base="reported",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
