# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Canonical,
    DateTime,
    Markdown,
    Date,
    Integer,
    Id,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    CodeableConcept,
    Reference,
    ContactDetail,
    UsageContext,
    Period,
    RelatedArtifact,
    BackboneElement,
    Quantity,
    Range,
    Ratio,
    Duration,
    TriggerDefinition,
    Expression,
    DataRequirement,
    Age,
    Timing,
    CodeableReference,
    DomainResource,
)


class PlanDefinitionGoalTarget(BackboneElement):
    """
    Indicates what should be done and within what timeframe.
    """

    measure: Optional[CodeableConcept] = Field(
        description="The parameter whose value is to be tracked",
        default=None,
    )
    detailQuantity: Optional[Quantity] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailRange: Optional[Range] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailCodeableConcept: Optional[CodeableConcept] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailString: Optional[String] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailString_ext: Optional[Element] = Field(
        description="Placeholder element for detailString extensions",
        default=None,
        alias="_detailString",
    )
    detailBoolean: Optional[Boolean] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for detailBoolean extensions",
        default=None,
        alias="_detailBoolean",
    )
    detailInteger: Optional[Integer] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailInteger_ext: Optional[Element] = Field(
        description="Placeholder element for detailInteger extensions",
        default=None,
        alias="_detailInteger",
    )
    detailRatio: Optional[Ratio] = Field(
        description="The target value to be achieved",
        default=None,
    )
    due: Optional[Duration] = Field(
        description="Reach goal within",
        default=None,
    )

    @property
    def detail(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="detail",
        )

    @field_validator(
        *(
            "due",
            "measure",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def detail_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Quantity,
                Range,
                CodeableConcept,
                String,
                Boolean,
                Integer,
                Ratio,
            ],
            field_name_base="detail",
            required=False,
        )


class PlanDefinitionGoal(BackboneElement):
    """
    A goal describes an expected outcome that activities within the plan are intended to achieve. For example, weight loss, restoring an activity of daily living, obtaining herd immunity via immunization, meeting a process improvement objective, meeting the acceptance criteria for a test as specified by a quality specification, etc.
    """

    category: Optional[CodeableConcept] = Field(
        description="E.g. Treatment, dietary, behavioral",
        default=None,
    )
    description: Optional[CodeableConcept] = Field(
        description="Code or text describing the goal",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="high-priority | medium-priority | low-priority",
        default=None,
    )
    start: Optional[CodeableConcept] = Field(
        description="When goal pursuit begins",
        default=None,
    )
    addresses: Optional[List[CodeableConcept]] = Field(
        description="What does the goal address",
        default=None,
    )
    documentation: Optional[List[RelatedArtifact]] = Field(
        description="Supporting documentation for the goal",
        default=None,
    )
    target: Optional[List[PlanDefinitionGoalTarget]] = Field(
        description="Target outcome for the goal",
        default=None,
    )

    @field_validator(
        *(
            "target",
            "documentation",
            "addresses",
            "start",
            "priority",
            "description",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActorOption(BackboneElement):
    """
    The characteristics of the candidates that could serve as the actor.
    """

    type: Optional[Code] = Field(
        description="careteam | device | group | healthcareservice | location | organization | patient | practitioner | practitionerrole | relatedperson",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    typeCanonical: Optional[Canonical] = Field(
        description="Who or what can participate",
        default=None,
    )
    typeCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for typeCanonical extensions",
        default=None,
        alias="_typeCanonical",
    )
    typeReference: Optional[Reference] = Field(
        description="Who or what can participate",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="E.g. Nurse, Surgeon, Parent",
        default=None,
    )

    @field_validator(
        *(
            "role",
            "typeReference",
            "typeCanonical",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActor(BackboneElement):
    """
    Actors represent the individuals or groups involved in the execution of the defined set of activities.
    """

    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Describes the actor",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    option: Optional[List[PlanDefinitionActorOption]] = Field(
        description="Who or what can be this actor",
        default=None,
    )

    @field_validator(
        *(
            "option",
            "description",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionCondition(BackboneElement):
    """
    An expression that describes applicability criteria or start/stop conditions for the action.
    """

    kind: Optional[Code] = Field(
        description="applicability | start | stop",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    expression: Optional[Expression] = Field(
        description="Boolean-valued expression",
        default=None,
    )

    @field_validator(
        *(
            "expression",
            "kind",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionInput(BackboneElement):
    """
    Defines input data requirements for the action.
    """

    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    requirement: Optional[DataRequirement] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData: Optional[Id] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData_ext: Optional[Element] = Field(
        description="Placeholder element for relatedData extensions",
        default=None,
        alias="_relatedData",
    )

    @field_validator(
        *(
            "relatedData",
            "requirement",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionOutput(BackboneElement):
    """
    Defines the outputs of the action, if any.
    """

    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    requirement: Optional[DataRequirement] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData: Optional[String] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData_ext: Optional[Element] = Field(
        description="Placeholder element for relatedData extensions",
        default=None,
        alias="_relatedData",
    )

    @field_validator(
        *(
            "relatedData",
            "requirement",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionRelatedAction(BackboneElement):
    """
    A relationship to another action such as "before" or "30-60 minutes after start of".
    """

    targetId: Optional[Id] = Field(
        description="What action is this related to",
        default=None,
    )
    targetId_ext: Optional[Element] = Field(
        description="Placeholder element for targetId extensions",
        default=None,
        alias="_targetId",
    )
    relationship: Optional[Code] = Field(
        description="before | before-start | before-end | concurrent | concurrent-with-start | concurrent-with-end | after | after-start | after-end",
        default=None,
    )
    relationship_ext: Optional[Element] = Field(
        description="Placeholder element for relationship extensions",
        default=None,
        alias="_relationship",
    )
    endRelationship: Optional[Code] = Field(
        description="before | before-start | before-end | concurrent | concurrent-with-start | concurrent-with-end | after | after-start | after-end",
        default=None,
    )
    endRelationship_ext: Optional[Element] = Field(
        description="Placeholder element for endRelationship extensions",
        default=None,
        alias="_endRelationship",
    )
    offsetDuration: Optional[Duration] = Field(
        description="Time offset for the relationship",
        default=None,
    )
    offsetRange: Optional[Range] = Field(
        description="Time offset for the relationship",
        default=None,
    )

    @property
    def offset(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="offset",
        )

    @field_validator(
        *(
            "endRelationship",
            "relationship",
            "targetId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def offset_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Duration, Range],
            field_name_base="offset",
            required=False,
        )


class PlanDefinitionActionParticipant(BackboneElement):
    """
    Indicates who should participate in performing the action described.
    """

    actorId: Optional[String] = Field(
        description="What actor",
        default=None,
    )
    actorId_ext: Optional[Element] = Field(
        description="Placeholder element for actorId extensions",
        default=None,
        alias="_actorId",
    )
    type: Optional[Code] = Field(
        description="careteam | device | group | healthcareservice | location | organization | patient | practitioner | practitionerrole | relatedperson",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    typeCanonical: Optional[Canonical] = Field(
        description="Who or what can participate",
        default=None,
    )
    typeCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for typeCanonical extensions",
        default=None,
        alias="_typeCanonical",
    )
    typeReference: Optional[Reference] = Field(
        description="Who or what can participate",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="E.g. Nurse, Surgeon, Parent",
        default=None,
    )
    function: Optional[CodeableConcept] = Field(
        description="E.g. Author, Reviewer, Witness, etc",
        default=None,
    )

    @field_validator(
        *(
            "function",
            "role",
            "typeReference",
            "typeCanonical",
            "type",
            "actorId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionDynamicValue(BackboneElement):
    """
    Customizations that should be applied to the statically defined resource. For example, if the dosage of a medication must be computed based on the patient's weight, a customization would be used to specify an expression that calculated the weight, and the path on the resource that would contain the result.
    """

    path: Optional[String] = Field(
        description="The path to the element to be set dynamically",
        default=None,
    )
    path_ext: Optional[Element] = Field(
        description="Placeholder element for path extensions",
        default=None,
        alias="_path",
    )
    expression: Optional[Expression] = Field(
        description="An expression that provides the dynamic value for the customization",
        default=None,
    )

    @field_validator(
        *(
            "expression",
            "path",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionAction(BackboneElement):
    """
    An action or group of actions to be taken as part of the plan. For example, in clinical care, an action would be to prescribe a particular indicated medication, or perform a particular test as appropriate. In pharmaceutical quality, an action would be the test that needs to be performed on a drug product as defined in the quality specification.
    """

    linkId: Optional[String] = Field(
        description="Unique id for the action in the PlanDefinition",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    prefix: Optional[String] = Field(
        description="User-visible prefix for the action (e.g. 1. or A.)",
        default=None,
    )
    prefix_ext: Optional[Element] = Field(
        description="Placeholder element for prefix extensions",
        default=None,
        alias="_prefix",
    )
    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Brief description of the action",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    textEquivalent: Optional[Markdown] = Field(
        description="Static text equivalent of the action, used if the dynamic aspects cannot be interpreted by the receiving system",
        default=None,
    )
    textEquivalent_ext: Optional[Element] = Field(
        description="Placeholder element for textEquivalent extensions",
        default=None,
        alias="_textEquivalent",
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    code: Optional[CodeableConcept] = Field(
        description="Code representing the meaning of the action or sub-actions",
        default=None,
    )
    reason: Optional[List[CodeableConcept]] = Field(
        description="Why the action should be performed",
        default=None,
    )
    documentation: Optional[List[RelatedArtifact]] = Field(
        description="Supporting documentation for the intended performer of the action",
        default=None,
    )
    goalId: Optional[List[Id]] = Field(
        description="What goals this action supports",
        default=None,
    )
    goalId_ext: Optional[Element] = Field(
        description="Placeholder element for goalId extensions",
        default=None,
        alias="_goalId",
    )
    subjectCodeableConcept: Optional[CodeableConcept] = Field(
        description="Type of individual the action is focused on",
        default=None,
    )
    subjectReference: Optional[Reference] = Field(
        description="Type of individual the action is focused on",
        default=None,
    )
    subjectCanonical: Optional[Canonical] = Field(
        description="Type of individual the action is focused on",
        default=None,
    )
    subjectCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for subjectCanonical extensions",
        default=None,
        alias="_subjectCanonical",
    )
    trigger: Optional[List[TriggerDefinition]] = Field(
        description="When the action should be triggered",
        default=None,
    )
    condition: Optional[List[PlanDefinitionActionCondition]] = Field(
        description="Whether or not the action is applicable",
        default=None,
    )
    input: Optional[List[PlanDefinitionActionInput]] = Field(
        description="Input data requirements",
        default=None,
    )
    output: Optional[List[PlanDefinitionActionOutput]] = Field(
        description="Output data definition",
        default=None,
    )
    relatedAction: Optional[List[PlanDefinitionActionRelatedAction]] = Field(
        description="Relationship to another action",
        default=None,
    )
    timingAge: Optional[Age] = Field(
        description="When the action should take place",
        default=None,
    )
    timingDuration: Optional[Duration] = Field(
        description="When the action should take place",
        default=None,
    )
    timingRange: Optional[Range] = Field(
        description="When the action should take place",
        default=None,
    )
    timingTiming: Optional[Timing] = Field(
        description="When the action should take place",
        default=None,
    )
    location: Optional[CodeableReference] = Field(
        description="Where it should happen",
        default=None,
    )
    participant: Optional[List[PlanDefinitionActionParticipant]] = Field(
        description="Who should participate in the action",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="create | update | remove | fire-event",
        default=None,
    )
    groupingBehavior: Optional[Code] = Field(
        description="visual-group | logical-group | sentence-group",
        default=None,
    )
    groupingBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for groupingBehavior extensions",
        default=None,
        alias="_groupingBehavior",
    )
    selectionBehavior: Optional[Code] = Field(
        description="any | all | all-or-none | exactly-one | at-most-one | one-or-more",
        default=None,
    )
    selectionBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for selectionBehavior extensions",
        default=None,
        alias="_selectionBehavior",
    )
    requiredBehavior: Optional[Code] = Field(
        description="must | could | must-unless-documented",
        default=None,
    )
    requiredBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for requiredBehavior extensions",
        default=None,
        alias="_requiredBehavior",
    )
    precheckBehavior: Optional[Code] = Field(
        description="yes | no",
        default=None,
    )
    precheckBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for precheckBehavior extensions",
        default=None,
        alias="_precheckBehavior",
    )
    cardinalityBehavior: Optional[Code] = Field(
        description="single | multiple",
        default=None,
    )
    cardinalityBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for cardinalityBehavior extensions",
        default=None,
        alias="_cardinalityBehavior",
    )
    definitionCanonical: Optional[Canonical] = Field(
        description="Description of the activity to be performed",
        default=None,
    )
    definitionCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for definitionCanonical extensions",
        default=None,
        alias="_definitionCanonical",
    )
    definitionUri: Optional[Uri] = Field(
        description="Description of the activity to be performed",
        default=None,
    )
    definitionUri_ext: Optional[Element] = Field(
        description="Placeholder element for definitionUri extensions",
        default=None,
        alias="_definitionUri",
    )
    transform: Optional[Canonical] = Field(
        description="Transform to apply the template",
        default=None,
    )
    transform_ext: Optional[Element] = Field(
        description="Placeholder element for transform extensions",
        default=None,
        alias="_transform",
    )
    dynamicValue: Optional[List[PlanDefinitionActionDynamicValue]] = Field(
        description="Dynamic aspects of the definition",
        default=None,
    )
    action: Optional[List["PlanDefinitionAction"]] = Field(
        description="A sub-action",
        default=None,
    )

    @property
    def subject(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="subject",
        )

    @property
    def timing(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="timing",
        )

    @property
    def definition(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="definition",
        )

    @field_validator(
        *(
            "action",
            "dynamicValue",
            "transform",
            "cardinalityBehavior",
            "precheckBehavior",
            "requiredBehavior",
            "selectionBehavior",
            "groupingBehavior",
            "type",
            "participant",
            "location",
            "relatedAction",
            "output",
            "input",
            "condition",
            "trigger",
            "goalId",
            "documentation",
            "reason",
            "code",
            "priority",
            "textEquivalent",
            "description",
            "title",
            "prefix",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("input",), mode="after", check_fields=None)
    @classmethod
    def FHIR_pld_0_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="requirement.exists() xor relatedData.exists()",
            human="Input data elements must have a requirement or a relatedData, but not both",
            key="pld-0",
            severity="error",
        )

    @field_validator(*("output",), mode="after", check_fields=None)
    @classmethod
    def FHIR_pld_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="requirement.exists() xor relatedData.exists()",
            human="Output data element must have a requirement or a relatedData, but not both",
            key="pld-1",
            severity="error",
        )

    @model_validator(mode="after")
    def subject_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference, Canonical],
            field_name_base="subject",
            required=False,
        )

    @model_validator(mode="after")
    def timing_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Age, Duration, Range, Timing],
            field_name_base="timing",
            required=False,
        )

    @model_validator(mode="after")
    def definition_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Canonical, Uri],
            field_name_base="definition",
            required=False,
        )


class PlanDefinition(DomainResource):
    """
    This resource allows for the definition of various types of plans as a sharable, consumable, and executable artifact. The resource is general enough to support the description of a broad range of clinical and non-clinical artifacts such as clinical decision support rules, order sets, protocols, and drug quality specifications.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/PlanDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this plan definition, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the plan definition",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the plan definition",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this plan definition (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this plan definition (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    subtitle: Optional[String] = Field(
        description="Subordinate title of the plan definition",
        default=None,
    )
    subtitle_ext: Optional[Element] = Field(
        description="Placeholder element for subtitle extensions",
        default=None,
        alias="_subtitle",
    )
    type: Optional[CodeableConcept] = Field(
        description="order-set | clinical-protocol | eca-rule | workflow-definition",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    subjectCodeableConcept: Optional[CodeableConcept] = Field(
        description="Type of individual the plan definition is focused on",
        default=None,
    )
    subjectReference: Optional[Reference] = Field(
        description="Type of individual the plan definition is focused on",
        default=None,
    )
    subjectCanonical: Optional[Canonical] = Field(
        description="Type of individual the plan definition is focused on",
        default=None,
    )
    subjectCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for subjectCanonical extensions",
        default=None,
        alias="_subjectCanonical",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the plan definition",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for plan definition (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this plan definition is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    usage: Optional[Markdown] = Field(
        description="Describes the clinical usage of the plan",
        default=None,
    )
    usage_ext: Optional[Element] = Field(
        description="Placeholder element for usage extensions",
        default=None,
        alias="_usage",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    approvalDate: Optional[Date] = Field(
        description="When the plan definition was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the plan definition was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the plan definition is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="E.g. Education, Treatment, Assessment",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the content",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the content",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the content",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the content",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations",
        default=None,
    )
    library: Optional[List[Canonical]] = Field(
        description="Logic used by the plan definition",
        default=None,
    )
    library_ext: Optional[Element] = Field(
        description="Placeholder element for library extensions",
        default=None,
        alias="_library",
    )
    goal: Optional[List[PlanDefinitionGoal]] = Field(
        description="What the plan is trying to accomplish",
        default=None,
    )
    actor: Optional[List[PlanDefinitionActor]] = Field(
        description="Actors within the plan",
        default=None,
    )
    action: Optional[List[PlanDefinitionAction]] = Field(
        description="Action defined by the plan",
        default=None,
    )
    asNeededBoolean: Optional[Boolean] = Field(
        description="Preconditions for service",
        default=None,
    )
    asNeededBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for asNeededBoolean extensions",
        default=None,
        alias="_asNeededBoolean",
    )
    asNeededCodeableConcept: Optional[CodeableConcept] = Field(
        description="Preconditions for service",
        default=None,
    )
    resourceType: Literal["PlanDefinition"] = Field(
        description=None,
        default="PlanDefinition",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @property
    def subject(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="subject",
        )

    @property
    def asNeeded(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="asNeeded",
        )

    @field_validator(
        *(
            "action",
            "actor",
            "goal",
            "library",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyrightLabel",
            "copyright",
            "usage",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "type",
            "subtitle",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def subject_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference, Canonical],
            field_name_base="subject",
            required=False,
        )

    @model_validator(mode="after")
    def asNeeded_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, CodeableConcept],
            field_name_base="asNeeded",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_pld_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="%context.repeat(action).where((goalId in %context.goal.id).not()).exists().not()",
            human="goalid should reference the id of a goal definition",
            key="pld-3",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_pld_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="%context.repeat(action).relatedAction.where((targetId in %context.repeat(action).id).not()).exists().not()",
            human="targetId should reference the id of an action",
            key="pld-4",
            severity="warning",
        )
