# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
    Boolean,
    Integer,
    Date,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Quantity,
    BackboneElement,
    MarketingStatus,
    ProductShelfLife,
    Attachment,
    CodeableReference,
    DomainResource,
)


class PackagedProductDefinitionLegalStatusOfSupply(BackboneElement):
    """
    The legal status of supply of the packaged item as classified by the regulator.
    """

    code: Optional[CodeableConcept] = Field(
        description="The actual status of supply. In what situation this package type may be supplied for use",
        default=None,
    )
    jurisdiction: Optional[CodeableConcept] = Field(
        description="The place where the legal status of supply applies",
        default=None,
    )

    @field_validator(
        *(
            "jurisdiction",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PackagedProductDefinitionPackagingProperty(BackboneElement):
    """
    General characteristics of this item.
    """

    type: Optional[CodeableConcept] = Field(
        description="A code expressing the type of characteristic",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate: Optional[Date] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="A value for the characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Quantity, Date, Boolean, Attachment],
            field_name_base="value",
            required=False,
        )


class PackagedProductDefinitionPackagingContainedItem(BackboneElement):
    """
    The item(s) within the packaging.
    """

    item: Optional[CodeableReference] = Field(
        description="The actual item(s) of medication, as manufactured, or a device, or other medically related item (food, biologicals, raw materials, medical fluids, gases etc.), as contained in the package",
        default=None,
    )
    amount: Optional[Quantity] = Field(
        description="The number of this type of item within this packaging or for continuous items such as liquids it is the quantity (for example 25ml). See also PackagedProductDefinition.containedItemQuantity (especially the long definition)",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "item",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PackagedProductDefinitionPackaging(BackboneElement):
    """
    A packaging item, as a container for medically related items, possibly with other packaging items within, or a packaging component, such as bottle cap (which is not a device or a medication manufactured item).
    """

    identifier: Optional[List[Identifier]] = Field(
        description="An identifier that is specific to this particular part of the packaging. Including possibly a Data Carrier Identifier",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The physical type of the container of the items",
        default=None,
    )
    componentPart: Optional[Boolean] = Field(
        description="Is this a part of the packaging (e.g. a cap or bottle stopper), rather than the packaging itself (e.g. a bottle or vial)",
        default=None,
    )
    componentPart_ext: Optional[Element] = Field(
        description="Placeholder element for componentPart extensions",
        default=None,
        alias="_componentPart",
    )
    quantity: Optional[Integer] = Field(
        description="The quantity of this level of packaging in the package that contains it (with the outermost level being 1)",
        default=None,
    )
    quantity_ext: Optional[Element] = Field(
        description="Placeholder element for quantity extensions",
        default=None,
        alias="_quantity",
    )
    material: Optional[List[CodeableConcept]] = Field(
        description="Material type of the package item",
        default=None,
    )
    alternateMaterial: Optional[List[CodeableConcept]] = Field(
        description="A possible alternate material for this part of the packaging, that is allowed to be used instead of the usual material",
        default=None,
    )
    shelfLifeStorage: Optional[List[ProductShelfLife]] = Field(
        description="Shelf Life and storage information",
        default=None,
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="Manufacturer of this packaging item (multiple means these are all potential manufacturers)",
        default=None,
    )
    property_: Optional[List[PackagedProductDefinitionPackagingProperty]] = Field(
        description="General characteristics of this item",
        default=None,
    )
    containedItem: Optional[List[PackagedProductDefinitionPackagingContainedItem]] = (
        Field(
            description="The item(s) within the packaging",
            default=None,
        )
    )
    packaging: Optional[List["PackagedProductDefinitionPackaging"]] = Field(
        description="Allows containers (and parts of containers) within containers, still as a part of single packaged product",
        default=None,
    )

    @field_validator(
        *(
            "packaging",
            "containedItem",
            "property_",
            "manufacturer",
            "shelfLifeStorage",
            "alternateMaterial",
            "material",
            "quantity",
            "componentPart",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PackagedProductDefinitionCharacteristic(BackboneElement):
    """
    Allows the key features to be recorded, such as "hospital pack", "nurse prescribable", "calendar pack".
    """

    type: Optional[CodeableConcept] = Field(
        description="A code expressing the type of characteristic",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate: Optional[Date] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="A value for the characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Quantity, Date, Boolean, Attachment],
            field_name_base="value",
            required=False,
        )


class PackagedProductDefinition(DomainResource):
    """
    A medically related item or items, in a container or package.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/PackagedProductDefinition"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="A unique identifier for this package as whole - not for the content of the package",
        default=None,
    )
    name: Optional[String] = Field(
        description="A name for this package. Typically as listed in a drug formulary, catalogue, inventory etc",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[CodeableConcept] = Field(
        description="A high level category e.g. medicinal product, raw material, shipping container etc",
        default=None,
    )
    packageFor: Optional[List[Reference]] = Field(
        description="The product that this is a pack for",
        default=None,
    )
    status: Optional[CodeableConcept] = Field(
        description="The status within the lifecycle of this item. High level - not intended to duplicate details elsewhere e.g. legal status, or authorization/marketing status",
        default=None,
    )
    statusDate: Optional[DateTime] = Field(
        description="The date at which the given status became applicable",
        default=None,
    )
    statusDate_ext: Optional[Element] = Field(
        description="Placeholder element for statusDate extensions",
        default=None,
        alias="_statusDate",
    )
    containedItemQuantity: Optional[List[Quantity]] = Field(
        description="A total of the complete count of contained items of a particular type/form, independent of sub-packaging or organization. This can be considered as the pack size. See also packaging.containedItem.amount (especially the long definition)",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Textual description. Note that this is not the name of the package or product",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    legalStatusOfSupply: Optional[
        List[PackagedProductDefinitionLegalStatusOfSupply]
    ] = Field(
        description="The legal status of supply of the packaged item as classified by the regulator",
        default=None,
    )
    marketingStatus: Optional[List[MarketingStatus]] = Field(
        description="Allows specifying that an item is on the market for sale, or that it is not available, and the dates and locations associated",
        default=None,
    )
    copackagedIndicator: Optional[Boolean] = Field(
        description="Identifies if the drug product is supplied with another item such as a diluent or adjuvant",
        default=None,
    )
    copackagedIndicator_ext: Optional[Element] = Field(
        description="Placeholder element for copackagedIndicator extensions",
        default=None,
        alias="_copackagedIndicator",
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="Manufacturer of this package type (multiple means these are all possible manufacturers)",
        default=None,
    )
    attachedDocument: Optional[List[Reference]] = Field(
        description="Additional information or supporting documentation about the packaged product",
        default=None,
    )
    packaging: Optional[PackagedProductDefinitionPackaging] = Field(
        description="A packaging item, as a container for medically related items, possibly with other packaging items within, or a packaging component, such as bottle cap",
        default=None,
    )
    characteristic: Optional[List[PackagedProductDefinitionCharacteristic]] = Field(
        description='Allows the key features to be recorded, such as "hospital pack", "nurse prescribable"',
        default=None,
    )
    resourceType: Literal["PackagedProductDefinition"] = Field(
        description=None,
        default="PackagedProductDefinition",
    )

    @field_validator(
        *(
            "characteristic",
            "packaging",
            "attachedDocument",
            "manufacturer",
            "copackagedIndicator",
            "marketingStatus",
            "legalStatusOfSupply",
            "description",
            "containedItemQuantity",
            "statusDate",
            "status",
            "packageFor",
            "type",
            "name",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
