# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Base64Binary,
    Boolean,
    DateTime,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    CodeableConcept,
    Reference,
    BackboneElement,
    CodeableReference,
    Ratio,
    Quantity,
    Attachment,
    Identifier,
    Annotation,
    DomainResource,
)


class NutritionProductNutrient(BackboneElement):
    """
    The product's nutritional information expressed by the nutrients.
    """

    item: Optional[CodeableReference] = Field(
        description="The (relevant) nutrients in the product",
        default=None,
    )
    amount: Optional[List[Ratio]] = Field(
        description="The amount of nutrient expressed in one or more units: X per pack / per serving / per dose",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "item",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionProductIngredient(BackboneElement):
    """
    Ingredients contained in this product.
    """

    item: Optional[CodeableReference] = Field(
        description="The ingredient contained in the product",
        default=None,
    )
    amount: Optional[List[Ratio]] = Field(
        description="The amount of ingredient that is in the product",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "item",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionProductCharacteristic(BackboneElement):
    """
    Specifies descriptive properties of the nutrition product.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code specifying the type of characteristic",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="The value of the characteristic",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="The value of the characteristic",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="The value of the characteristic",
        default=None,
    )
    valueBase64Binary: Optional[Base64Binary] = Field(
        description="The value of the characteristic",
        default=None,
    )
    valueBase64Binary_ext: Optional[Element] = Field(
        description="Placeholder element for valueBase64Binary extensions",
        default=None,
        alias="_valueBase64Binary",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="The value of the characteristic",
        default=None,
    )
    valueBoolean: Optional[Boolean] = Field(
        description="The value of the characteristic",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                CodeableConcept,
                String,
                Quantity,
                Base64Binary,
                Attachment,
                Boolean,
            ],
            field_name_base="value",
            required=True,
        )


class NutritionProductInstance(BackboneElement):
    """
    Conveys instance-level information about this product item. One or several physical, countable instances or occurrences of the product.
    """

    quantity: Optional[Quantity] = Field(
        description="The amount of items or instances",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="The identifier for the physical instance, typically a serial number or manufacturer number",
        default=None,
    )
    name: Optional[String] = Field(
        description="The name for the specific product",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    lotNumber: Optional[String] = Field(
        description="The identification of the batch or lot of the product",
        default=None,
    )
    lotNumber_ext: Optional[Element] = Field(
        description="Placeholder element for lotNumber extensions",
        default=None,
        alias="_lotNumber",
    )
    expiry: Optional[DateTime] = Field(
        description="The expiry date or date and time for the product",
        default=None,
    )
    expiry_ext: Optional[Element] = Field(
        description="Placeholder element for expiry extensions",
        default=None,
        alias="_expiry",
    )
    useBy: Optional[DateTime] = Field(
        description="The date until which the product is expected to be good for consumption",
        default=None,
    )
    useBy_ext: Optional[Element] = Field(
        description="Placeholder element for useBy extensions",
        default=None,
        alias="_useBy",
    )
    biologicalSourceEvent: Optional[Identifier] = Field(
        description="An identifier that supports traceability to the event during which material in this product from one or more biological entities was obtained or pooled",
        default=None,
    )

    @field_validator(
        *(
            "biologicalSourceEvent",
            "useBy",
            "expiry",
            "lotNumber",
            "name",
            "identifier",
            "quantity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionProduct(DomainResource):
    """
    A food or supplement that is consumed by patients.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/NutritionProduct"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="A code that can identify the detailed nutrients and ingredients in a specific food product",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | inactive | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Broad product groups or categories used to classify the product, such as Legume and Legume Products, Beverages, or Beef Products",
        default=None,
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="Manufacturer, representative or officially responsible for the product",
        default=None,
    )
    nutrient: Optional[List[NutritionProductNutrient]] = Field(
        description="The product\u0027s nutritional information expressed by the nutrients",
        default=None,
    )
    ingredient: Optional[List[NutritionProductIngredient]] = Field(
        description="Ingredients contained in this product",
        default=None,
    )
    knownAllergen: Optional[List[CodeableReference]] = Field(
        description="Known or suspected allergens that are a part of this product",
        default=None,
    )
    characteristic: Optional[List[NutritionProductCharacteristic]] = Field(
        description="Specifies descriptive properties of the nutrition product",
        default=None,
    )
    instance: Optional[List[NutritionProductInstance]] = Field(
        description="One or several physical instances or occurrences of the nutrition product",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments made about the product",
        default=None,
    )
    resourceType: Literal["NutritionProduct"] = Field(
        description=None,
        default="NutritionProduct",
    )

    @field_validator(
        *(
            "note",
            "instance",
            "characteristic",
            "knownAllergen",
            "ingredient",
            "nutrient",
            "manufacturer",
            "category",
            "status",
            "code",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
