# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    BackboneElement,
    CodeableReference,
    Timing,
    Quantity,
    Annotation,
    DomainResource,
)


class NutritionIntakeConsumedItem(BackboneElement):
    """
    What food or fluid product or item was consumed.
    """

    type: Optional[CodeableConcept] = Field(
        description="The type of food or fluid product",
        default=None,
    )
    nutritionProduct: Optional[CodeableReference] = Field(
        description="Code that identifies the food or fluid product that was consumed",
        default=None,
    )
    schedule: Optional[Timing] = Field(
        description="Scheduled frequency of consumption",
        default=None,
    )
    amount: Optional[Quantity] = Field(
        description="Quantity of the specified food",
        default=None,
    )
    rate: Optional[Quantity] = Field(
        description="Rate at which enteral feeding was administered",
        default=None,
    )
    notConsumed: Optional[Boolean] = Field(
        description="Flag to indicate if the food or fluid item was refused or otherwise not consumed",
        default=None,
    )
    notConsumed_ext: Optional[Element] = Field(
        description="Placeholder element for notConsumed extensions",
        default=None,
        alias="_notConsumed",
    )
    notConsumedReason: Optional[CodeableConcept] = Field(
        description="Reason food or fluid was not consumed",
        default=None,
    )

    @field_validator(
        *(
            "notConsumedReason",
            "notConsumed",
            "rate",
            "amount",
            "schedule",
            "nutritionProduct",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionIntakeIngredientLabel(BackboneElement):
    """
    Total nutrient amounts for the whole meal, product, serving, etc.
    """

    nutrient: Optional[CodeableReference] = Field(
        description="Total nutrient consumed",
        default=None,
    )
    amount: Optional[Quantity] = Field(
        description="Total amount of nutrient consumed",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "nutrient",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionIntakePerformer(BackboneElement):
    """
    Who performed the intake and how they were involved.
    """

    function: Optional[CodeableConcept] = Field(
        description="Type of performer",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Who performed the intake",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionIntake(DomainResource):
    """
    A record of food or fluid that is being consumed by a patient.  A NutritionIntake may indicate that the patient may be consuming the food or fluid now or has consumed the food or fluid in the past.  The source of this information can be the patient, significant other (such as a family member or spouse), or a clinician.  A common scenario where this information is captured is during the history taking process during a patient visit or stay or through an app that tracks food or fluids consumed.   The consumption information may come from sources such as the patient's memory, from a nutrition label,  or from a clinician documenting observed intake.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/NutritionIntake"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External identifier",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Fulfils plan, proposal or order",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of referenced event",
        default=None,
    )
    status: Optional[Code] = Field(
        description="preparation | in-progress | not-done | on-hold | stopped | completed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[List[CodeableConcept]] = Field(
        description="Reason for current status",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Code representing an overall type of nutrition intake",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who is/was consuming the food or fluid",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter associated with NutritionIntake",
        default=None,
    )
    occurrenceDateTime: Optional[DateTime] = Field(
        description="The date/time or interval when the food or fluid is/was consumed",
        default=None,
    )
    occurrenceDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDateTime extensions",
        default=None,
        alias="_occurrenceDateTime",
    )
    occurrencePeriod: Optional[Period] = Field(
        description="The date/time or interval when the food or fluid is/was consumed",
        default=None,
    )
    recorded: Optional[DateTime] = Field(
        description="When the intake was recorded",
        default=None,
    )
    recorded_ext: Optional[Element] = Field(
        description="Placeholder element for recorded extensions",
        default=None,
        alias="_recorded",
    )
    reportedBoolean: Optional[Boolean] = Field(
        description="Person or organization that provided the information about the consumption of this food or fluid",
        default=None,
    )
    reportedBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for reportedBoolean extensions",
        default=None,
        alias="_reportedBoolean",
    )
    reportedReference: Optional[Reference] = Field(
        description="Person or organization that provided the information about the consumption of this food or fluid",
        default=None,
    )
    consumedItem: Optional[List[NutritionIntakeConsumedItem]] = Field(
        description="What food or fluid product or item was consumed",
        default=None,
    )
    ingredientLabel: Optional[List[NutritionIntakeIngredientLabel]] = Field(
        description="Total nutrient for the whole meal, product, serving",
        default=None,
    )
    performer: Optional[List[NutritionIntakePerformer]] = Field(
        description="Who was performed in the intake",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where the intake occurred",
        default=None,
    )
    derivedFrom: Optional[List[Reference]] = Field(
        description="Additional supporting information",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Reason for why the food or fluid is /was consumed",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Further information about the consumption",
        default=None,
    )
    resourceType: Literal["NutritionIntake"] = Field(
        description=None,
        default="NutritionIntake",
    )

    @property
    def occurrence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurrence",
        )

    @property
    def reported(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="reported",
        )

    @field_validator(
        *(
            "note",
            "reason",
            "derivedFrom",
            "location",
            "performer",
            "ingredientLabel",
            "consumedItem",
            "recorded",
            "encounter",
            "subject",
            "code",
            "statusReason",
            "status",
            "partOf",
            "basedOn",
            "instantiatesUri",
            "instantiatesCanonical",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurrence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="occurrence",
            required=False,
        )

    @model_validator(mode="after")
    def reported_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, Reference],
            field_name_base="reported",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
