# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    CodeableConcept,
    UsageContext,
    Period,
    RelatedArtifact,
    BackboneElement,
    DomainResource,
)


class NamingSystemUniqueId(BackboneElement):
    """
    Indicates how the system may be identified when referenced in electronic exchange.
    """

    type: Optional[Code] = Field(
        description="oid | uuid | uri | iri-stem | v2csmnemonic | other",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    value: Optional[String] = Field(
        description="The unique identifier",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    preferred: Optional[Boolean] = Field(
        description="Is this the id that should be used for this type",
        default=None,
    )
    preferred_ext: Optional[Element] = Field(
        description="Placeholder element for preferred extensions",
        default=None,
        alias="_preferred",
    )
    comment: Optional[String] = Field(
        description="Notes about identifier usage",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )
    period: Optional[Period] = Field(
        description="When is identifier valid?",
        default=None,
    )
    authoritative: Optional[Boolean] = Field(
        description="Whether the identifier is authoritative",
        default=None,
    )
    authoritative_ext: Optional[Element] = Field(
        description="Placeholder element for authoritative extensions",
        default=None,
        alias="_authoritative",
    )

    @field_validator(
        *(
            "authoritative",
            "period",
            "comment",
            "preferred",
            "value",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NamingSystem(DomainResource):
    """
    A curated namespace that issues unique symbols within that namespace for the identification of concepts, people, devices, etc.  Represents a "System" used within the Identifier and Coding data types.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/NamingSystem"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this naming system, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the naming system (business identifier)",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the naming system",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this naming system (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Title for this naming system (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    kind: Optional[Code] = Field(
        description="codesystem | identifier | root",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    responsible: Optional[String] = Field(
        description="Who maintains system namespace?",
        default=None,
    )
    responsible_ext: Optional[Element] = Field(
        description="Placeholder element for responsible extensions",
        default=None,
        alias="_responsible",
    )
    type: Optional[CodeableConcept] = Field(
        description="e.g. driver,  provider,  patient, bank etc",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the naming system",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for naming system (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this naming system is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    approvalDate: Optional[Date] = Field(
        description="When the NamingSystem was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the NamingSystem was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the NamingSystem is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="E.g. Education, Treatment, Assessment, etc",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the CodeSystem",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the NamingSystem",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the NamingSystem",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the NamingSystem",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations, etc",
        default=None,
    )
    usage: Optional[String] = Field(
        description="How/where is it used",
        default=None,
    )
    usage_ext: Optional[Element] = Field(
        description="Placeholder element for usage extensions",
        default=None,
        alias="_usage",
    )
    uniqueId: Optional[List[NamingSystemUniqueId]] = Field(
        description="Unique identifiers used for system",
        default=None,
    )
    resourceType: Literal["NamingSystem"] = Field(
        description=None,
        default="NamingSystem",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "uniqueId",
            "usage",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "type",
            "responsible",
            "contact",
            "publisher",
            "date",
            "experimental",
            "kind",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_nsd_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="kind != 'root' or uniqueId.all(type != 'uuid')",
            human="Root systems cannot have uuid identifiers",
            key="nsd-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_nsd_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="uniqueId.where(preferred = true).select(type).isDistinct()",
            human="Can't have more than one preferred identifier for a type",
            key="nsd-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_nsd_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="uniqueId.where(authoritative = 'true').select(type.toString() & period.start.toString() & period.end.toString()).isDistinct()",
            human="Can't have more than one authoritative identifier for a type/period combination (only one authoritative identifier allowed at any given point of time)",
            key="nsd-3",
            severity="error",
        )
