# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, Integer

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    Attachment,
    BackboneElement,
    CodeableConcept,
    Range,
    DomainResource,
)


class MolecularSequenceRelativeStartingSequence(BackboneElement):
    """
    A sequence that is used as a starting sequence to describe variants that are present in a sequence analyzed.
    """

    genomeAssembly: Optional[CodeableConcept] = Field(
        description="The genome assembly used for starting sequence, e.g. GRCh38",
        default=None,
    )
    chromosome: Optional[CodeableConcept] = Field(
        description="Chromosome Identifier",
        default=None,
    )
    sequenceCodeableConcept: Optional[CodeableConcept] = Field(
        description="The reference sequence that represents the starting sequence",
        default=None,
    )
    sequenceString: Optional[String] = Field(
        description="The reference sequence that represents the starting sequence",
        default=None,
    )
    sequenceString_ext: Optional[Element] = Field(
        description="Placeholder element for sequenceString extensions",
        default=None,
        alias="_sequenceString",
    )
    sequenceReference: Optional[Reference] = Field(
        description="The reference sequence that represents the starting sequence",
        default=None,
    )
    windowStart: Optional[Integer] = Field(
        description="Start position of the window on the starting sequence",
        default=None,
    )
    windowStart_ext: Optional[Element] = Field(
        description="Placeholder element for windowStart extensions",
        default=None,
        alias="_windowStart",
    )
    windowEnd: Optional[Integer] = Field(
        description="End position of the window on the starting sequence",
        default=None,
    )
    windowEnd_ext: Optional[Element] = Field(
        description="Placeholder element for windowEnd extensions",
        default=None,
        alias="_windowEnd",
    )
    orientation: Optional[Code] = Field(
        description="sense | antisense",
        default=None,
    )
    orientation_ext: Optional[Element] = Field(
        description="Placeholder element for orientation extensions",
        default=None,
        alias="_orientation",
    )
    strand: Optional[Code] = Field(
        description="watson | crick",
        default=None,
    )
    strand_ext: Optional[Element] = Field(
        description="Placeholder element for strand extensions",
        default=None,
        alias="_strand",
    )

    @property
    def sequence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="sequence",
        )

    @field_validator(
        *(
            "strand",
            "orientation",
            "windowEnd",
            "windowStart",
            "chromosome",
            "genomeAssembly",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def sequence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, String, Reference],
            field_name_base="sequence",
            required=False,
        )


class MolecularSequenceRelativeEdit(BackboneElement):
    """
    Changes in sequence from the starting sequence.
    """

    start: Optional[Integer] = Field(
        description="Start position of the edit on the starting sequence",
        default=None,
    )
    start_ext: Optional[Element] = Field(
        description="Placeholder element for start extensions",
        default=None,
        alias="_start",
    )
    end: Optional[Integer] = Field(
        description="End position of the edit on the starting sequence",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )
    replacementSequence: Optional[String] = Field(
        description="Allele that was observed",
        default=None,
    )
    replacementSequence_ext: Optional[Element] = Field(
        description="Placeholder element for replacementSequence extensions",
        default=None,
        alias="_replacementSequence",
    )
    replacedSequence: Optional[String] = Field(
        description="Allele in the starting sequence",
        default=None,
    )
    replacedSequence_ext: Optional[Element] = Field(
        description="Placeholder element for replacedSequence extensions",
        default=None,
        alias="_replacedSequence",
    )

    @field_validator(
        *(
            "replacedSequence",
            "replacementSequence",
            "end",
            "start",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceRelative(BackboneElement):
    """
    A sequence defined relative to another sequence.
    """

    coordinateSystem: Optional[CodeableConcept] = Field(
        description="Ways of identifying nucleotides or amino acids within a sequence",
        default=None,
    )
    ordinalPosition: Optional[Integer] = Field(
        description="Indicates the order in which the sequence should be considered when putting multiple \u0027relative\u0027 elements together",
        default=None,
    )
    ordinalPosition_ext: Optional[Element] = Field(
        description="Placeholder element for ordinalPosition extensions",
        default=None,
        alias="_ordinalPosition",
    )
    sequenceRange: Optional[Range] = Field(
        description="Indicates the nucleotide range in the composed sequence when multiple \u0027relative\u0027 elements are used together",
        default=None,
    )
    startingSequence: Optional[MolecularSequenceRelativeStartingSequence] = Field(
        description="A sequence used as starting sequence",
        default=None,
    )
    edit: Optional[List[MolecularSequenceRelativeEdit]] = Field(
        description="Changes in sequence from the starting sequence",
        default=None,
    )

    @field_validator(
        *(
            "edit",
            "startingSequence",
            "sequenceRange",
            "ordinalPosition",
            "coordinateSystem",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("startingSequence",), mode="after", check_fields=None)
    @classmethod
    def FHIR_msq_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="chromosome.exists() = genomeAssembly.exists()",
            human="Both genomeAssembly and chromosome must be both contained if either one of them is contained",
            key="msq-5",
            severity="error",
        )

    @field_validator(*("startingSequence",), mode="after", check_fields=None)
    @classmethod
    def FHIR_msq_6_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="genomeAssembly.exists() xor sequence.exists()",
            human="Have and only have one of the following elements in startingSequence: 1. genomeAssembly; 2 sequence",
            key="msq-6",
            severity="error",
        )


class MolecularSequence(DomainResource):
    """
    Representation of a molecular sequence.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MolecularSequence"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Unique ID for this particular sequence",
        default=None,
    )
    type: Optional[Code] = Field(
        description="aa | dna | rna",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    subject: Optional[Reference] = Field(
        description="Subject this sequence is associated too",
        default=None,
    )
    focus: Optional[List[Reference]] = Field(
        description="What the molecular sequence is about, when it is not about the subject of record",
        default=None,
    )
    specimen: Optional[Reference] = Field(
        description="Specimen used for sequencing",
        default=None,
    )
    device: Optional[Reference] = Field(
        description="The method for sequencing",
        default=None,
    )
    performer: Optional[Reference] = Field(
        description="Who should be responsible for test result",
        default=None,
    )
    literal: Optional[String] = Field(
        description="Sequence that was observed",
        default=None,
    )
    literal_ext: Optional[Element] = Field(
        description="Placeholder element for literal extensions",
        default=None,
        alias="_literal",
    )
    formatted: Optional[List[Attachment]] = Field(
        description="Embedded file or a link (URL) which contains content to represent the sequence",
        default=None,
    )
    relative: Optional[List[MolecularSequenceRelative]] = Field(
        description="A sequence defined relative to another sequence",
        default=None,
    )
    resourceType: Literal["MolecularSequence"] = Field(
        description=None,
        default="MolecularSequence",
    )

    @field_validator(
        *(
            "relative",
            "formatted",
            "literal",
            "performer",
            "device",
            "specimen",
            "focus",
            "subject",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
