# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
    Integer,
    Date,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    MarketingStatus,
    Reference,
    CodeableReference,
    BackboneElement,
    Coding,
    Period,
    Quantity,
    Attachment,
    DomainResource,
)


class MedicinalProductDefinitionContact(BackboneElement):
    """
    A product specific contact, person (in a role), or an organization.
    """

    type: Optional[CodeableConcept] = Field(
        description="Allows the contact to be classified, for example QPPV, Pharmacovigilance Enquiry Information",
        default=None,
    )
    contact: Optional[Reference] = Field(
        description="A product specific contact, person (in a role), or an organization",
        default=None,
    )

    @field_validator(
        *(
            "contact",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductDefinitionNamePart(BackboneElement):
    """
    Coding words or phrases of the name.
    """

    part: Optional[String] = Field(
        description="A fragment of a product name",
        default=None,
    )
    part_ext: Optional[Element] = Field(
        description="Placeholder element for part extensions",
        default=None,
        alias="_part",
    )
    type: Optional[CodeableConcept] = Field(
        description="Identifying type for this part of the name (e.g. strength part)",
        default=None,
    )

    @field_validator(
        *(
            "type",
            "part",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductDefinitionNameUsage(BackboneElement):
    """
    Country and jurisdiction where the name applies, and associated language.
    """

    country: Optional[CodeableConcept] = Field(
        description="Country code for where this name applies",
        default=None,
    )
    jurisdiction: Optional[CodeableConcept] = Field(
        description="Jurisdiction code for where this name applies",
        default=None,
    )
    language: Optional[CodeableConcept] = Field(
        description="Language code for this name",
        default=None,
    )

    @field_validator(
        *(
            "language",
            "jurisdiction",
            "country",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductDefinitionName(BackboneElement):
    """
    The product's name, including full name and possibly coded parts.
    """

    productName: Optional[String] = Field(
        description="The full product name",
        default=None,
    )
    productName_ext: Optional[Element] = Field(
        description="Placeholder element for productName extensions",
        default=None,
        alias="_productName",
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of product name, such as rINN, BAN, Proprietary, Non-Proprietary",
        default=None,
    )
    part: Optional[List[MedicinalProductDefinitionNamePart]] = Field(
        description="Coding words or phrases of the name",
        default=None,
    )
    usage: Optional[List[MedicinalProductDefinitionNameUsage]] = Field(
        description="Country and jurisdiction where the name applies",
        default=None,
    )

    @field_validator(
        *(
            "usage",
            "part",
            "type",
            "productName",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductDefinitionCrossReference(BackboneElement):
    """
    Reference to another product, e.g. for linking authorised to investigational product, or a virtual product.
    """

    product: Optional[CodeableReference] = Field(
        description="Reference to another product, e.g. for linking authorised to investigational product",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The type of relationship, for instance branded to generic or virtual to actual product",
        default=None,
    )

    @field_validator(
        *(
            "type",
            "product",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductDefinitionOperation(BackboneElement):
    """
    A manufacturing or administrative process or step associated with (or performed on) the medicinal product.
    """

    type: Optional[CodeableReference] = Field(
        description="The type of manufacturing operation e.g. manufacturing itself, re-packaging",
        default=None,
    )
    effectiveDate: Optional[Period] = Field(
        description="Date range of applicability",
        default=None,
    )
    organization: Optional[List[Reference]] = Field(
        description="The organization responsible for the particular process, e.g. the manufacturer or importer",
        default=None,
    )
    confidentialityIndicator: Optional[CodeableConcept] = Field(
        description="Specifies whether this process is considered proprietary or confidential",
        default=None,
    )

    @field_validator(
        *(
            "confidentialityIndicator",
            "organization",
            "effectiveDate",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductDefinitionCharacteristic(BackboneElement):
    """
    Allows the key product features to be recorded, such as "sugar free", "modified release", "parallel import".
    """

    type: Optional[CodeableConcept] = Field(
        description="A code expressing the type of characteristic",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueMarkdown: Optional[Markdown] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for valueMarkdown extensions",
        default=None,
        alias="_valueMarkdown",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueInteger: Optional[Integer] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDate: Optional[Date] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="A value for the characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                CodeableConcept,
                Markdown,
                Quantity,
                Integer,
                Date,
                Boolean,
                Attachment,
            ],
            field_name_base="value",
            required=False,
        )


class MedicinalProductDefinition(DomainResource):
    """
    Detailed definition of a medicinal product, typically for uses other than direct patient care (e.g. regulatory use, drug catalogs, to support prescribing, adverse events management etc.).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/MedicinalProductDefinition"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for this product. Could be an MPID",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Regulatory type, e.g. Investigational or Authorized",
        default=None,
    )
    domain: Optional[CodeableConcept] = Field(
        description="If this medicine applies to human or veterinary uses",
        default=None,
    )
    version: Optional[String] = Field(
        description="A business identifier relating to a specific version of the product",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    status: Optional[CodeableConcept] = Field(
        description="The status within the lifecycle of this product record",
        default=None,
    )
    statusDate: Optional[DateTime] = Field(
        description="The date at which the given status became applicable",
        default=None,
    )
    statusDate_ext: Optional[Element] = Field(
        description="Placeholder element for statusDate extensions",
        default=None,
        alias="_statusDate",
    )
    description: Optional[Markdown] = Field(
        description="General description of this product",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    combinedPharmaceuticalDoseForm: Optional[CodeableConcept] = Field(
        description="The dose form for a single part product, or combined form of a multiple part product",
        default=None,
    )
    route: Optional[List[CodeableConcept]] = Field(
        description="The path by which the product is taken into or makes contact with the body",
        default=None,
    )
    indication: Optional[Markdown] = Field(
        description="Description of indication(s) for this product, used when structured indications are not required",
        default=None,
    )
    indication_ext: Optional[Element] = Field(
        description="Placeholder element for indication extensions",
        default=None,
        alias="_indication",
    )
    legalStatusOfSupply: Optional[CodeableConcept] = Field(
        description="The legal status of supply of the medicinal product as classified by the regulator",
        default=None,
    )
    additionalMonitoringIndicator: Optional[CodeableConcept] = Field(
        description="Whether the Medicinal Product is subject to additional monitoring for regulatory reasons",
        default=None,
    )
    specialMeasures: Optional[List[CodeableConcept]] = Field(
        description="Whether the Medicinal Product is subject to special measures for regulatory reasons",
        default=None,
    )
    pediatricUseIndicator: Optional[CodeableConcept] = Field(
        description="If authorised for use in children",
        default=None,
    )
    classification: Optional[List[CodeableConcept]] = Field(
        description="Allows the product to be classified by various systems",
        default=None,
    )
    marketingStatus: Optional[List[MarketingStatus]] = Field(
        description="Marketing status of the medicinal product, in contrast to marketing authorization",
        default=None,
    )
    packagedMedicinalProduct: Optional[List[CodeableConcept]] = Field(
        description="Package type for the product",
        default=None,
    )
    comprisedOf: Optional[List[Reference]] = Field(
        description="Types of medicinal manufactured items and/or devices that this product consists of, such as tablets, capsule, or syringes",
        default=None,
    )
    ingredient: Optional[List[CodeableConcept]] = Field(
        description="The ingredients of this medicinal product - when not detailed in other resources",
        default=None,
    )
    impurity: Optional[List[CodeableReference]] = Field(
        description="Any component of the drug product which is not the chemical entity defined as the drug substance, or an excipient in the drug product",
        default=None,
    )
    attachedDocument: Optional[List[Reference]] = Field(
        description="Additional documentation about the medicinal product",
        default=None,
    )
    masterFile: Optional[List[Reference]] = Field(
        description="A master file for the medicinal product (e.g. Pharmacovigilance System Master File)",
        default=None,
    )
    contact: Optional[List[MedicinalProductDefinitionContact]] = Field(
        description="A product specific contact, person (in a role), or an organization",
        default=None,
    )
    clinicalTrial: Optional[List[Reference]] = Field(
        description="Clinical trials or studies that this product is involved in",
        default=None,
    )
    code: Optional[List[Coding]] = Field(
        description="A code that this product is known by, within some formal terminology",
        default=None,
    )
    name: Optional[List[MedicinalProductDefinitionName]] = Field(
        description="The product\u0027s name, including full name and possibly coded parts",
        default=None,
    )
    crossReference: Optional[List[MedicinalProductDefinitionCrossReference]] = Field(
        description="Reference to another product, e.g. for linking authorised to investigational product",
        default=None,
    )
    operation: Optional[List[MedicinalProductDefinitionOperation]] = Field(
        description="A manufacturing or administrative process for the medicinal product",
        default=None,
    )
    characteristic: Optional[List[MedicinalProductDefinitionCharacteristic]] = Field(
        description='Key product features such as "sugar free", "modified release"',
        default=None,
    )
    resourceType: Literal["MedicinalProductDefinition"] = Field(
        description=None,
        default="MedicinalProductDefinition",
    )

    @field_validator(
        *(
            "characteristic",
            "operation",
            "crossReference",
            "name",
            "code",
            "clinicalTrial",
            "contact",
            "masterFile",
            "attachedDocument",
            "impurity",
            "ingredient",
            "comprisedOf",
            "packagedMedicinalProduct",
            "marketingStatus",
            "classification",
            "pediatricUseIndicator",
            "specialMeasures",
            "additionalMonitoringIndicator",
            "legalStatusOfSupply",
            "indication",
            "route",
            "combinedPharmaceuticalDoseForm",
            "description",
            "statusDate",
            "status",
            "version",
            "domain",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
