# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    CodeableReference,
    Period,
    Timing,
    Annotation,
    Dosage,
    BackboneElement,
    DomainResource,
)


class MedicationStatementAdherence(BackboneElement):
    """
    Indicates whether the medication is or is not being consumed or administered.
    """

    code: Optional[CodeableConcept] = Field(
        description="Type of adherence",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Details of the reason for the current use of the medication",
        default=None,
    )

    @field_validator(
        *(
            "reason",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationStatement(DomainResource):
    """
        A record of a medication that is being consumed by a patient.   A MedicationStatement may indicate that the patient may be taking the medication now or has taken the medication in the past or will be taking the medication in the future.  The source of this information can be the patient, significant other (such as a family member or spouse), or a clinician.  A common scenario where this information is captured is during the history taking process during a patient visit or stay.   The medication information may come from sources such as the patient's memory, from a prescription bottle,  or from a list of medications the patient, clinician or other party maintains.

    The primary difference between a medicationstatement and a medicationadministration is that the medication administration has complete administration information and is based on actual administration information from the person who administered the medication.  A medicationstatement is often, if not always, less specific.  There is no required date/time when the medication was administered, in fact we only know that a source has reported the patient is taking this medication, where details such as time, quantity, or rate or even medication product may be incomplete or missing or less precise.  As stated earlier, the Medication Statement information may come from the patient's memory, from a prescription bottle or from a list of medications the patient, clinician or other party maintains.  Medication administration is more formal and is not missing detailed information.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MedicationStatement"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External identifier",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of referenced event",
        default=None,
    )
    status: Optional[Code] = Field(
        description="recorded | entered-in-error | draft",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Type of medication statement",
        default=None,
    )
    medication: Optional[CodeableReference] = Field(
        description="What medication was taken",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who is/was taking  the medication",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter associated with MedicationStatement",
        default=None,
    )
    effectiveDateTime: Optional[DateTime] = Field(
        description="The date/time or interval when the medication is/was/will be taken",
        default=None,
    )
    effectiveDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for effectiveDateTime extensions",
        default=None,
        alias="_effectiveDateTime",
    )
    effectivePeriod: Optional[Period] = Field(
        description="The date/time or interval when the medication is/was/will be taken",
        default=None,
    )
    effectiveTiming: Optional[Timing] = Field(
        description="The date/time or interval when the medication is/was/will be taken",
        default=None,
    )
    dateAsserted: Optional[DateTime] = Field(
        description="When the usage was asserted?",
        default=None,
    )
    dateAsserted_ext: Optional[Element] = Field(
        description="Placeholder element for dateAsserted extensions",
        default=None,
        alias="_dateAsserted",
    )
    informationSource: Optional[List[Reference]] = Field(
        description="Person or organization that provided the information about the taking of this medication",
        default=None,
    )
    derivedFrom: Optional[List[Reference]] = Field(
        description="Link to information used to derive the MedicationStatement",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Reason for why the medication is being/was taken",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Further information about the usage",
        default=None,
    )
    relatedClinicalInformation: Optional[List[Reference]] = Field(
        description="Link to information relevant to the usage of a medication",
        default=None,
    )
    renderedDosageInstruction: Optional[Markdown] = Field(
        description="Full representation of the dosage instructions",
        default=None,
    )
    renderedDosageInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for renderedDosageInstruction extensions",
        default=None,
        alias="_renderedDosageInstruction",
    )
    dosage: Optional[List[Dosage]] = Field(
        description="Details of how medication is/was taken or should be taken",
        default=None,
    )
    adherence: Optional[MedicationStatementAdherence] = Field(
        description="Indicates whether the medication is or is not being consumed or administered",
        default=None,
    )
    resourceType: Literal["MedicationStatement"] = Field(
        description=None,
        default="MedicationStatement",
    )

    @property
    def effective(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="effective",
        )

    @field_validator(
        *(
            "adherence",
            "dosage",
            "renderedDosageInstruction",
            "relatedClinicalInformation",
            "note",
            "reason",
            "derivedFrom",
            "informationSource",
            "dateAsserted",
            "encounter",
            "subject",
            "medication",
            "category",
            "status",
            "partOf",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def effective_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period, Timing],
            field_name_base="effective",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
