# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Boolean,
    Markdown,
    UnsignedInt,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    CodeableReference,
    Annotation,
    Period,
    Dosage,
    BackboneElement,
    Quantity,
    Duration,
    DomainResource,
)


class MedicationRequestDispenseRequestInitialFill(BackboneElement):
    """
    Indicates the quantity or duration for the first dispense of the medication.
    """

    quantity: Optional[Quantity] = Field(
        description="First fill quantity",
        default=None,
    )
    duration: Optional[Duration] = Field(
        description="First fill duration",
        default=None,
    )

    @field_validator(
        *(
            "duration",
            "quantity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationRequestDispenseRequest(BackboneElement):
    """
    Indicates the specific details for the dispense or medication supply part of a medication request (also known as a Medication Prescription or Medication Order).  Note that this information is not always sent with the order.  There may be in some settings (e.g. hospitals) institutional or system support for completing the dispense details in the pharmacy department.
    """

    initialFill: Optional[MedicationRequestDispenseRequestInitialFill] = Field(
        description="First fill details",
        default=None,
    )
    dispenseInterval: Optional[Duration] = Field(
        description="Minimum period of time between dispenses",
        default=None,
    )
    validityPeriod: Optional[Period] = Field(
        description="Time period supply is authorized for",
        default=None,
    )
    numberOfRepeatsAllowed: Optional[UnsignedInt] = Field(
        description="Number of refills authorized",
        default=None,
    )
    numberOfRepeatsAllowed_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfRepeatsAllowed extensions",
        default=None,
        alias="_numberOfRepeatsAllowed",
    )
    quantity: Optional[Quantity] = Field(
        description="Amount of medication to supply per dispense",
        default=None,
    )
    expectedSupplyDuration: Optional[Duration] = Field(
        description="Number of days supply per dispense",
        default=None,
    )
    dispenser: Optional[Reference] = Field(
        description="Intended performer of dispense",
        default=None,
    )
    dispenserInstruction: Optional[List[Annotation]] = Field(
        description="Additional information for the dispenser",
        default=None,
    )
    doseAdministrationAid: Optional[CodeableConcept] = Field(
        description="Type of adherence packaging to use for the dispense",
        default=None,
    )

    @field_validator(
        *(
            "doseAdministrationAid",
            "dispenserInstruction",
            "dispenser",
            "expectedSupplyDuration",
            "quantity",
            "numberOfRepeatsAllowed",
            "validityPeriod",
            "dispenseInterval",
            "initialFill",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationRequestSubstitution(BackboneElement):
    """
    Indicates whether or not substitution can or should be part of the dispense. In some cases, substitution must happen, in other cases substitution must not happen. This block explains the prescriber's intent. If nothing is specified substitution may be done.
    """

    allowedBoolean: Optional[Boolean] = Field(
        description="Whether substitution is allowed or not",
        default=None,
    )
    allowedBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for allowedBoolean extensions",
        default=None,
        alias="_allowedBoolean",
    )
    allowedCodeableConcept: Optional[CodeableConcept] = Field(
        description="Whether substitution is allowed or not",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Why should (not) substitution be made",
        default=None,
    )

    @property
    def allowed(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="allowed",
        )

    @field_validator(
        *("reason", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def allowed_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, CodeableConcept],
            field_name_base="allowed",
            required=True,
        )


class MedicationRequest(DomainResource):
    """
    An order or request for both supply of the medication and the instructions for administration of the medication to a patient. The resource is called "MedicationRequest" rather than "MedicationPrescription" or "MedicationOrder" to generalize the use across inpatient and outpatient settings, including care plans, etc., and to harmonize with workflow patterns.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MedicationRequest"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External ids for this request",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="A plan or request that is fulfilled in whole or in part by this medication request",
        default=None,
    )
    priorPrescription: Optional[Reference] = Field(
        description="Reference to an order/prescription that is being replaced by this MedicationRequest",
        default=None,
    )
    groupIdentifier: Optional[Identifier] = Field(
        description="Composite request this is part of",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | on-hold | ended | stopped | completed | cancelled | entered-in-error | draft | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableConcept] = Field(
        description="Reason for current status",
        default=None,
    )
    statusChanged: Optional[DateTime] = Field(
        description="When the status was changed",
        default=None,
    )
    statusChanged_ext: Optional[Element] = Field(
        description="Placeholder element for statusChanged extensions",
        default=None,
        alias="_statusChanged",
    )
    intent: Optional[Code] = Field(
        description="proposal | plan | order | original-order | reflex-order | filler-order | instance-order | option",
        default=None,
    )
    intent_ext: Optional[Element] = Field(
        description="Placeholder element for intent extensions",
        default=None,
        alias="_intent",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Grouping or category of medication request",
        default=None,
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    doNotPerform: Optional[Boolean] = Field(
        description="True if patient is to stop taking or not to start taking the medication",
        default=None,
    )
    doNotPerform_ext: Optional[Element] = Field(
        description="Placeholder element for doNotPerform extensions",
        default=None,
        alias="_doNotPerform",
    )
    medication: Optional[CodeableReference] = Field(
        description="Medication to be taken",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Individual or group for whom the medication has been requested",
        default=None,
    )
    informationSource: Optional[List[Reference]] = Field(
        description="The person or organization who provided the information about this request, if the source is someone other than the requestor",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter created as part of encounter/admission/stay",
        default=None,
    )
    supportingInformation: Optional[List[Reference]] = Field(
        description="Information to support fulfilling of the medication",
        default=None,
    )
    authoredOn: Optional[DateTime] = Field(
        description="When request was initially authored",
        default=None,
    )
    authoredOn_ext: Optional[Element] = Field(
        description="Placeholder element for authoredOn extensions",
        default=None,
        alias="_authoredOn",
    )
    requester: Optional[Reference] = Field(
        description="Who/What requested the Request",
        default=None,
    )
    reported: Optional[Boolean] = Field(
        description="Reported rather than primary record",
        default=None,
    )
    reported_ext: Optional[Element] = Field(
        description="Placeholder element for reported extensions",
        default=None,
        alias="_reported",
    )
    performerType: Optional[CodeableConcept] = Field(
        description="Desired kind of performer of the medication administration",
        default=None,
    )
    performer: Optional[List[Reference]] = Field(
        description="Intended performer of administration",
        default=None,
    )
    device: Optional[List[CodeableReference]] = Field(
        description="Intended type of device for the administration",
        default=None,
    )
    recorder: Optional[Reference] = Field(
        description="Person who entered the request",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Reason or indication for ordering or not ordering the medication",
        default=None,
    )
    courseOfTherapyType: Optional[CodeableConcept] = Field(
        description="Overall pattern of medication administration",
        default=None,
    )
    insurance: Optional[List[Reference]] = Field(
        description="Associated insurance coverage",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Information about the prescription",
        default=None,
    )
    renderedDosageInstruction: Optional[Markdown] = Field(
        description="Full representation of the dosage instructions",
        default=None,
    )
    renderedDosageInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for renderedDosageInstruction extensions",
        default=None,
        alias="_renderedDosageInstruction",
    )
    effectiveDosePeriod: Optional[Period] = Field(
        description="Period over which the medication is to be taken",
        default=None,
    )
    dosageInstruction: Optional[List[Dosage]] = Field(
        description="Specific instructions for how the medication should be taken",
        default=None,
    )
    dispenseRequest: Optional[MedicationRequestDispenseRequest] = Field(
        description="Medication supply authorization",
        default=None,
    )
    substitution: Optional[MedicationRequestSubstitution] = Field(
        description="Any restrictions on medication substitution",
        default=None,
    )
    eventHistory: Optional[List[Reference]] = Field(
        description="A list of events of interest in the lifecycle",
        default=None,
    )
    resourceType: Literal["MedicationRequest"] = Field(
        description=None,
        default="MedicationRequest",
    )

    @field_validator(
        *(
            "eventHistory",
            "substitution",
            "dispenseRequest",
            "dosageInstruction",
            "effectiveDosePeriod",
            "renderedDosageInstruction",
            "note",
            "insurance",
            "courseOfTherapyType",
            "reason",
            "recorder",
            "device",
            "performer",
            "performerType",
            "reported",
            "requester",
            "authoredOn",
            "supportingInformation",
            "encounter",
            "informationSource",
            "subject",
            "medication",
            "doNotPerform",
            "priority",
            "category",
            "intent",
            "statusChanged",
            "statusReason",
            "status",
            "groupIdentifier",
            "priorPrescription",
            "basedOn",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
