# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Markdown,
    Boolean,
    Base64Binary,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    Period,
    Money,
    CodeableReference,
    Dosage,
    Quantity,
    Range,
    Annotation,
    Duration,
    Ratio,
    Attachment,
    DomainResource,
)


class MedicationKnowledgeRelatedMedicationKnowledge(BackboneElement):
    """
    Associated or related medications. For example, if the medication is a branded product (e.g. Crestor), this is the Therapeutic Moeity (e.g. Rosuvastatin) or if this is a generic medication (e.g. Rosuvastatin), this would link to a branded product (e.g. Crestor.
    """

    type: Optional[CodeableConcept] = Field(
        description="Category of medicationKnowledge",
        default=None,
    )
    reference: Optional[List[Reference]] = Field(
        description="Associated documentation about the associated medication knowledge",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeMonograph(BackboneElement):
    """
    Associated documentation about the medication.
    """

    type: Optional[CodeableConcept] = Field(
        description="The category of medication document",
        default=None,
    )
    source: Optional[Reference] = Field(
        description="Associated documentation about the medication",
        default=None,
    )

    @field_validator(
        *(
            "source",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeCost(BackboneElement):
    """
    The price of the medication.
    """

    effectiveDate: Optional[List[Period]] = Field(
        description="The date range for which the cost is effective",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The category of the cost information",
        default=None,
    )
    source: Optional[String] = Field(
        description="The source or owner for the price information",
        default=None,
    )
    source_ext: Optional[Element] = Field(
        description="Placeholder element for source extensions",
        default=None,
        alias="_source",
    )
    costMoney: Optional[Money] = Field(
        description="The price or category of the cost of the medication",
        default=None,
    )
    costCodeableConcept: Optional[CodeableConcept] = Field(
        description="The price or category of the cost of the medication",
        default=None,
    )

    @property
    def cost(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="cost",
        )

    @field_validator(
        *(
            "source",
            "type",
            "effectiveDate",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def cost_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Money, CodeableConcept],
            field_name_base="cost",
            required=True,
        )


class MedicationKnowledgeMonitoringProgram(BackboneElement):
    """
    The program under which the medication is reviewed.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of program under which the medication is monitored",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name of the reviewing program",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )

    @field_validator(
        *(
            "name",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeIndicationGuidelineDosingGuidelineDosage(BackboneElement):
    """
    Dosage for the medication for the specific guidelines.
    """

    type: Optional[CodeableConcept] = Field(
        description="Category of dosage for a medication",
        default=None,
    )
    dosage: Optional[List[Dosage]] = Field(
        description="Dosage for the medication for the specific guidelines",
        default=None,
    )

    @field_validator(
        *(
            "dosage",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeIndicationGuidelineDosingGuidelinePatientCharacteristic(
    BackboneElement
):
    """
    Characteristics of the patient that are relevant to the administration guidelines (for example, height, weight, gender, etc.).
    """

    type: Optional[CodeableConcept] = Field(
        description="Categorization of specific characteristic that is relevant to the administration guideline",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="The specific characteristic",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="The specific characteristic",
        default=None,
    )
    valueRange: Optional[Range] = Field(
        description="The specific characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Quantity, Range],
            field_name_base="value",
            required=False,
        )


class MedicationKnowledgeIndicationGuidelineDosingGuideline(BackboneElement):
    """
    The guidelines for the dosage of the medication for the indication.
    """

    treatmentIntent: Optional[CodeableConcept] = Field(
        description="Intention of the treatment",
        default=None,
    )
    dosage: Optional[
        List[MedicationKnowledgeIndicationGuidelineDosingGuidelineDosage]
    ] = Field(
        description="Dosage for the medication for the specific guidelines",
        default=None,
    )
    administrationTreatment: Optional[CodeableConcept] = Field(
        description="Type of treatment the guideline applies to",
        default=None,
    )
    patientCharacteristic: Optional[
        List[MedicationKnowledgeIndicationGuidelineDosingGuidelinePatientCharacteristic]
    ] = Field(
        description="Characteristics of the patient that are relevant to the administration guidelines",
        default=None,
    )

    @field_validator(
        *(
            "patientCharacteristic",
            "administrationTreatment",
            "dosage",
            "treatmentIntent",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeIndicationGuideline(BackboneElement):
    """
    Guidelines or protocols that are applicable for the administration of the medication based on indication.
    """

    indication: Optional[List[CodeableReference]] = Field(
        description="Indication for use that applies to the specific administration guideline",
        default=None,
    )
    dosingGuideline: Optional[
        List[MedicationKnowledgeIndicationGuidelineDosingGuideline]
    ] = Field(
        description="Guidelines for dosage of the medication",
        default=None,
    )

    @field_validator(
        *(
            "dosingGuideline",
            "indication",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeMedicineClassification(BackboneElement):
    """
    Categorization of the medication within a formulary or classification system.
    """

    type: Optional[CodeableConcept] = Field(
        description="The type of category for the medication (for example, therapeutic classification, therapeutic sub-classification)",
        default=None,
    )
    sourceString: Optional[String] = Field(
        description="The source of the classification",
        default=None,
    )
    sourceString_ext: Optional[Element] = Field(
        description="Placeholder element for sourceString extensions",
        default=None,
        alias="_sourceString",
    )
    sourceUri: Optional[Uri] = Field(
        description="The source of the classification",
        default=None,
    )
    sourceUri_ext: Optional[Element] = Field(
        description="Placeholder element for sourceUri extensions",
        default=None,
        alias="_sourceUri",
    )
    classification: Optional[List[CodeableConcept]] = Field(
        description="Specific category assigned to the medication",
        default=None,
    )

    @property
    def source(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="source",
        )

    @field_validator(
        *(
            "classification",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def source_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Uri],
            field_name_base="source",
            required=False,
        )


class MedicationKnowledgePackagingCost(BackboneElement):
    """
    The cost of the packaged medication.
    """

    effectiveDate: Optional[List[Period]] = Field(
        description="The date range for which the cost is effective",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The category of the cost information",
        default=None,
    )
    source: Optional[String] = Field(
        description="The source or owner for the price information",
        default=None,
    )
    source_ext: Optional[Element] = Field(
        description="Placeholder element for source extensions",
        default=None,
        alias="_source",
    )
    costMoney: Optional[Money] = Field(
        description="The price or category of the cost of the medication",
        default=None,
    )
    costCodeableConcept: Optional[CodeableConcept] = Field(
        description="The price or category of the cost of the medication",
        default=None,
    )

    @property
    def cost(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="cost",
        )

    @field_validator(
        *(
            "source",
            "type",
            "effectiveDate",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def cost_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Money, CodeableConcept],
            field_name_base="cost",
            required=True,
        )


class MedicationKnowledgePackaging(BackboneElement):
    """
    Information that only applies to packages (not products).
    """

    cost: Optional[List[MedicationKnowledgePackagingCost]] = Field(
        description="Cost of the packaged medication",
        default=None,
    )
    packagedProduct: Optional[Reference] = Field(
        description="The packaged medication that is being priced",
        default=None,
    )

    @field_validator(
        *(
            "packagedProduct",
            "cost",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeStorageGuidelineEnvironmentalSetting(BackboneElement):
    """
    Describes a setting/value on the environment for the adequate storage of the medication and other substances.  Environment settings may involve temperature, humidity, or exposure to light.
    """

    type: Optional[CodeableConcept] = Field(
        description="Categorization of the setting",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Value of the setting",
        default=None,
    )
    valueRange: Optional[Range] = Field(
        description="Value of the setting",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Value of the setting",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Quantity, Range, CodeableConcept],
            field_name_base="value",
            required=True,
        )


class MedicationKnowledgeStorageGuideline(BackboneElement):
    """
    Information on how the medication should be stored, for example, refrigeration temperatures and length of stability at a given temperature.
    """

    reference: Optional[Uri] = Field(
        description="Reference to additional information",
        default=None,
    )
    reference_ext: Optional[Element] = Field(
        description="Placeholder element for reference extensions",
        default=None,
        alias="_reference",
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional storage notes",
        default=None,
    )
    stabilityDuration: Optional[Duration] = Field(
        description="Duration remains stable",
        default=None,
    )
    environmentalSetting: Optional[
        List[MedicationKnowledgeStorageGuidelineEnvironmentalSetting]
    ] = Field(
        description="Setting or value of environment for adequate storage",
        default=None,
    )

    @field_validator(
        *(
            "environmentalSetting",
            "stabilityDuration",
            "note",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeRegulatorySubstitution(BackboneElement):
    """
    Specifies if changes are allowed when dispensing a medication from a regulatory perspective.
    """

    type: Optional[CodeableConcept] = Field(
        description="Specifies the type of substitution allowed",
        default=None,
    )
    allowed: Optional[Boolean] = Field(
        description="Specifies if regulation allows for changes in the medication when dispensing",
        default=None,
    )
    allowed_ext: Optional[Element] = Field(
        description="Placeholder element for allowed extensions",
        default=None,
        alias="_allowed",
    )

    @field_validator(
        *(
            "allowed",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeRegulatoryMaxDispense(BackboneElement):
    """
    The maximum number of units of the medication that can be dispensed in a period.
    """

    quantity: Optional[Quantity] = Field(
        description="The maximum number of units of the medication that can be dispensed",
        default=None,
    )
    period: Optional[Duration] = Field(
        description="The period that applies to the maximum number of units",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "quantity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeRegulatory(BackboneElement):
    """
    Regulatory information about a medication.
    """

    regulatoryAuthority: Optional[Reference] = Field(
        description="Specifies the authority of the regulation",
        default=None,
    )
    substitution: Optional[List[MedicationKnowledgeRegulatorySubstitution]] = Field(
        description="Specifies if changes are allowed when dispensing a medication from a regulatory perspective",
        default=None,
    )
    schedule: Optional[List[CodeableConcept]] = Field(
        description="Specifies the schedule of a medication in jurisdiction",
        default=None,
    )
    maxDispense: Optional[MedicationKnowledgeRegulatoryMaxDispense] = Field(
        description="The maximum number of units of the medication that can be dispensed in a period",
        default=None,
    )

    @field_validator(
        *(
            "maxDispense",
            "schedule",
            "substitution",
            "regulatoryAuthority",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeDefinitionalIngredient(BackboneElement):
    """
    Identifies a particular constituent of interest in the product.
    """

    item: Optional[CodeableReference] = Field(
        description="Substances contained in the medication",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="A code that defines the type of ingredient, active, base, etc",
        default=None,
    )
    strengthRatio: Optional[Ratio] = Field(
        description="Quantity of ingredient present",
        default=None,
    )
    strengthCodeableConcept: Optional[CodeableConcept] = Field(
        description="Quantity of ingredient present",
        default=None,
    )
    strengthQuantity: Optional[Quantity] = Field(
        description="Quantity of ingredient present",
        default=None,
    )

    @property
    def strength(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="strength",
        )

    @field_validator(
        *(
            "type",
            "item",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def strength_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Ratio, CodeableConcept, Quantity],
            field_name_base="strength",
            required=False,
        )


class MedicationKnowledgeDefinitionalDrugCharacteristic(BackboneElement):
    """
    Specifies descriptive properties of the medicine, such as color, shape, imprints, etc.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code specifying the type of characteristic of medication",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueBase64Binary: Optional[Base64Binary] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueBase64Binary_ext: Optional[Element] = Field(
        description="Placeholder element for valueBase64Binary extensions",
        default=None,
        alias="_valueBase64Binary",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="Description of the characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, String, Quantity, Base64Binary, Attachment],
            field_name_base="value",
            required=False,
        )


class MedicationKnowledgeDefinitional(BackboneElement):
    """
    Along with the link to a Medicinal Product Definition resource, this information provides common definitional elements that are needed to understand the specific medication that is being described.
    """

    definition: Optional[List[Reference]] = Field(
        description="Definitional resources that provide more information about this medication",
        default=None,
    )
    doseForm: Optional[CodeableConcept] = Field(
        description="powder | tablets | capsule +",
        default=None,
    )
    intendedRoute: Optional[List[CodeableConcept]] = Field(
        description="The intended or approved route of administration",
        default=None,
    )
    ingredient: Optional[List[MedicationKnowledgeDefinitionalIngredient]] = Field(
        description="Active or inactive ingredient",
        default=None,
    )
    drugCharacteristic: Optional[
        List[MedicationKnowledgeDefinitionalDrugCharacteristic]
    ] = Field(
        description="Specifies descriptive properties of the medicine",
        default=None,
    )

    @field_validator(
        *(
            "drugCharacteristic",
            "ingredient",
            "intendedRoute",
            "doseForm",
            "definition",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledge(DomainResource):
    """
    Information about a medication that is used to support knowledge.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MedicationKnowledge"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for this medication",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Code that identifies this medication",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | entered-in-error | inactive",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    author: Optional[Reference] = Field(
        description="Creator or owner of the knowledge or information about the medication",
        default=None,
    )
    intendedJurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Codes that identify the different jurisdictions for which the information of this resource was created",
        default=None,
    )
    name: Optional[List[String]] = Field(
        description="A name associated with the medication being described",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    relatedMedicationKnowledge: Optional[
        List[MedicationKnowledgeRelatedMedicationKnowledge]
    ] = Field(
        description="Associated or related medication information",
        default=None,
    )
    associatedMedication: Optional[List[Reference]] = Field(
        description="The set of medication resources that are associated with this medication",
        default=None,
    )
    productType: Optional[List[CodeableConcept]] = Field(
        description="Category of the medication or product",
        default=None,
    )
    monograph: Optional[List[MedicationKnowledgeMonograph]] = Field(
        description="Associated documentation about the medication",
        default=None,
    )
    preparationInstruction: Optional[Markdown] = Field(
        description="The instructions for preparing the medication",
        default=None,
    )
    preparationInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for preparationInstruction extensions",
        default=None,
        alias="_preparationInstruction",
    )
    cost: Optional[List[MedicationKnowledgeCost]] = Field(
        description="The pricing of the medication",
        default=None,
    )
    monitoringProgram: Optional[List[MedicationKnowledgeMonitoringProgram]] = Field(
        description="Program under which a medication is reviewed",
        default=None,
    )
    indicationGuideline: Optional[List[MedicationKnowledgeIndicationGuideline]] = Field(
        description="Guidelines or protocols for administration of the medication for an indication",
        default=None,
    )
    medicineClassification: Optional[
        List[MedicationKnowledgeMedicineClassification]
    ] = Field(
        description="Categorization of the medication within a formulary or classification system",
        default=None,
    )
    packaging: Optional[List[MedicationKnowledgePackaging]] = Field(
        description="Details about packaged medications",
        default=None,
    )
    clinicalUseIssue: Optional[List[Reference]] = Field(
        description="Potential clinical issue with or between medication(s)",
        default=None,
    )
    storageGuideline: Optional[List[MedicationKnowledgeStorageGuideline]] = Field(
        description="How the medication should be stored",
        default=None,
    )
    regulatory: Optional[List[MedicationKnowledgeRegulatory]] = Field(
        description="Regulatory information about a medication",
        default=None,
    )
    definitional: Optional[MedicationKnowledgeDefinitional] = Field(
        description="Minimal definition information about the medication",
        default=None,
    )
    resourceType: Literal["MedicationKnowledge"] = Field(
        description=None,
        default="MedicationKnowledge",
    )

    @field_validator(
        *(
            "definitional",
            "regulatory",
            "storageGuideline",
            "clinicalUseIssue",
            "packaging",
            "medicineClassification",
            "indicationGuideline",
            "monitoringProgram",
            "cost",
            "preparationInstruction",
            "monograph",
            "productType",
            "associatedMedication",
            "relatedMedicationKnowledge",
            "name",
            "intendedJurisdiction",
            "author",
            "status",
            "code",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )
