# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    CodeableConcept,
    Reference,
    ContactDetail,
    UsageContext,
    Period,
    RelatedArtifact,
    BackboneElement,
    Expression,
    DomainResource,
)


class MeasureTerm(BackboneElement):
    """
    Provides a description of an individual term used within the measure.
    """

    code: Optional[CodeableConcept] = Field(
        description="What term?",
        default=None,
    )
    definition: Optional[Markdown] = Field(
        description="Meaning of the term",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )

    @field_validator(
        *(
            "definition",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MeasureGroupPopulation(BackboneElement):
    """
    A population criteria for the measure.
    """

    linkId: Optional[String] = Field(
        description="Unique id for population in measure",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    code: Optional[CodeableConcept] = Field(
        description="initial-population | numerator | numerator-exclusion | denominator | denominator-exclusion | denominator-exception | measure-population | measure-population-exclusion | measure-observation",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="The human readable description of this population criteria",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="The criteria that defines this population",
        default=None,
    )
    groupDefinition: Optional[Reference] = Field(
        description="A group resource that defines this population",
        default=None,
    )
    inputPopulationId: Optional[String] = Field(
        description="Which population",
        default=None,
    )
    inputPopulationId_ext: Optional[Element] = Field(
        description="Placeholder element for inputPopulationId extensions",
        default=None,
        alias="_inputPopulationId",
    )
    aggregateMethod: Optional[CodeableConcept] = Field(
        description="Aggregation method for a measure score (e.g. sum, average, median, minimum, maximum, count)",
        default=None,
    )

    @field_validator(
        *(
            "aggregateMethod",
            "inputPopulationId",
            "groupDefinition",
            "criteria",
            "description",
            "code",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("linkId",), mode="after", check_fields=None)
    @classmethod
    def FHIR_mea_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="$this.length() <= 255",
            human="Link ids should be 255 characters or less",
            key="mea-3",
            severity="warning",
        )


class MeasureGroupStratifierComponent(BackboneElement):
    """
    A component of the stratifier criteria for the measure report, specified as either the name of a valid CQL expression defined within a referenced library or a valid FHIR Resource Path.
    """

    linkId: Optional[String] = Field(
        description="Unique id for stratifier component in measure",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    code: Optional[CodeableConcept] = Field(
        description="Meaning of the stratifier component",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="The human readable description of this stratifier component",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="Component of how the measure should be stratified",
        default=None,
    )
    groupDefinition: Optional[Reference] = Field(
        description="A group resource that defines this population",
        default=None,
    )

    @field_validator(
        *(
            "groupDefinition",
            "criteria",
            "description",
            "code",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("linkId",), mode="after", check_fields=None)
    @classmethod
    def FHIR_mea_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="$this.length() <= 255",
            human="Link ids should be 255 characters or less",
            key="mea-5",
            severity="warning",
        )


class MeasureGroupStratifier(BackboneElement):
    """
    The stratifier criteria for the measure report, specified as either the name of a valid CQL expression defined within a referenced library or a valid FHIR Resource Path.
    """

    linkId: Optional[String] = Field(
        description="Unique id for stratifier in measure",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    code: Optional[CodeableConcept] = Field(
        description="Meaning of the stratifier",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="The human readable description of this stratifier",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="How the measure should be stratified",
        default=None,
    )
    groupDefinition: Optional[Reference] = Field(
        description="A group resource that defines this population",
        default=None,
    )
    component: Optional[List[MeasureGroupStratifierComponent]] = Field(
        description="Stratifier criteria component for the measure",
        default=None,
    )

    @field_validator(
        *(
            "component",
            "groupDefinition",
            "criteria",
            "description",
            "code",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("linkId",), mode="after", check_fields=None)
    @classmethod
    def FHIR_mea_4_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="$this.length() <= 255",
            human="Link ids should be 255 characters or less",
            key="mea-4",
            severity="warning",
        )


class MeasureGroup(BackboneElement):
    """
    A group of population criteria for the measure.
    """

    linkId: Optional[String] = Field(
        description="Unique id for group in measure",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    code: Optional[CodeableConcept] = Field(
        description="Meaning of the group",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Summary description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="process | outcome | structure | patient-reported-outcome | composite",
        default=None,
    )
    subjectCodeableConcept: Optional[CodeableConcept] = Field(
        description="E.g. Patient, Practitioner, RelatedPerson, Organization, Location, Device",
        default=None,
    )
    subjectReference: Optional[Reference] = Field(
        description="E.g. Patient, Practitioner, RelatedPerson, Organization, Location, Device",
        default=None,
    )
    basis: Optional[Code] = Field(
        description="Population basis",
        default=None,
    )
    basis_ext: Optional[Element] = Field(
        description="Placeholder element for basis extensions",
        default=None,
        alias="_basis",
    )
    scoring: Optional[CodeableConcept] = Field(
        description="proportion | ratio | continuous-variable | cohort",
        default=None,
    )
    scoringUnit: Optional[CodeableConcept] = Field(
        description="What units?",
        default=None,
    )
    rateAggregation: Optional[Markdown] = Field(
        description="How is rate aggregation performed for this measure",
        default=None,
    )
    rateAggregation_ext: Optional[Element] = Field(
        description="Placeholder element for rateAggregation extensions",
        default=None,
        alias="_rateAggregation",
    )
    improvementNotation: Optional[CodeableConcept] = Field(
        description="increase | decrease",
        default=None,
    )
    library: Optional[List[Canonical]] = Field(
        description="Logic used by the measure group",
        default=None,
    )
    library_ext: Optional[Element] = Field(
        description="Placeholder element for library extensions",
        default=None,
        alias="_library",
    )
    population: Optional[List[MeasureGroupPopulation]] = Field(
        description="Population criteria",
        default=None,
    )
    stratifier: Optional[List[MeasureGroupStratifier]] = Field(
        description="Stratifier criteria for the measure",
        default=None,
    )

    @property
    def subject(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="subject",
        )

    @field_validator(
        *(
            "stratifier",
            "population",
            "library",
            "improvementNotation",
            "rateAggregation",
            "scoringUnit",
            "scoring",
            "basis",
            "type",
            "description",
            "code",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("linkId",), mode="after", check_fields=None)
    @classmethod
    def FHIR_mea_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="$this.length() <= 255",
            human="Link ids should be 255 characters or less",
            key="mea-2",
            severity="warning",
        )

    @model_validator(mode="after")
    def subject_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="subject",
            required=False,
        )


class MeasureSupplementalData(BackboneElement):
    """
    The supplemental data criteria for the measure report, specified as either the name of a valid CQL expression within a referenced library, or a valid FHIR Resource Path.
    """

    linkId: Optional[String] = Field(
        description="Unique id for supplementalData in measure",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    code: Optional[CodeableConcept] = Field(
        description="Meaning of the supplemental data",
        default=None,
    )
    usage: Optional[List[CodeableConcept]] = Field(
        description="supplemental-data | risk-adjustment-factor",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="The human readable description of this supplemental data",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="Expression describing additional data to be reported",
        default=None,
    )

    @field_validator(
        *(
            "criteria",
            "description",
            "usage",
            "code",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("linkId",), mode="after", check_fields=None)
    @classmethod
    def FHIR_mea_6_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="$this.length() <= 255",
            human="Link ids should be 255 characters or less",
            key="mea-6",
            severity="warning",
        )


class Measure(DomainResource):
    """
    The Measure resource provides the definition of a quality measure.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Measure"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this measure, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the measure",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the measure",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this measure (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this measure (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    subtitle: Optional[String] = Field(
        description="Subordinate title of the measure",
        default=None,
    )
    subtitle_ext: Optional[Element] = Field(
        description="Placeholder element for subtitle extensions",
        default=None,
        alias="_subtitle",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    subjectCodeableConcept: Optional[CodeableConcept] = Field(
        description="E.g. Patient, Practitioner, RelatedPerson, Organization, Location, Device",
        default=None,
    )
    subjectReference: Optional[Reference] = Field(
        description="E.g. Patient, Practitioner, RelatedPerson, Organization, Location, Device",
        default=None,
    )
    basis: Optional[Code] = Field(
        description="Population basis",
        default=None,
    )
    basis_ext: Optional[Element] = Field(
        description="Placeholder element for basis extensions",
        default=None,
        alias="_basis",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the measure",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for measure (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this measure is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    usage: Optional[Markdown] = Field(
        description="Describes the clinical usage of the measure",
        default=None,
    )
    usage_ext: Optional[Element] = Field(
        description="Placeholder element for usage extensions",
        default=None,
        alias="_usage",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    approvalDate: Optional[Date] = Field(
        description="When the measure was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the measure was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the measure is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="The category of the measure, such as Education, Treatment, Assessment, etc",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the content",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the content",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the content",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the content",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations, etc",
        default=None,
    )
    library: Optional[List[Canonical]] = Field(
        description="Logic used by the measure",
        default=None,
    )
    library_ext: Optional[Element] = Field(
        description="Placeholder element for library extensions",
        default=None,
        alias="_library",
    )
    disclaimer: Optional[Markdown] = Field(
        description="Disclaimer for use of the measure or its referenced content",
        default=None,
    )
    disclaimer_ext: Optional[Element] = Field(
        description="Placeholder element for disclaimer extensions",
        default=None,
        alias="_disclaimer",
    )
    scoring: Optional[CodeableConcept] = Field(
        description="proportion | ratio | continuous-variable | cohort",
        default=None,
    )
    scoringUnit: Optional[CodeableConcept] = Field(
        description="What units?",
        default=None,
    )
    compositeScoring: Optional[CodeableConcept] = Field(
        description="opportunity | all-or-nothing | linear | weighted",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="process | outcome | structure | patient-reported-outcome | composite",
        default=None,
    )
    riskAdjustment: Optional[Markdown] = Field(
        description="How risk adjustment is applied for this measure",
        default=None,
    )
    riskAdjustment_ext: Optional[Element] = Field(
        description="Placeholder element for riskAdjustment extensions",
        default=None,
        alias="_riskAdjustment",
    )
    rateAggregation: Optional[Markdown] = Field(
        description="How is rate aggregation performed for this measure",
        default=None,
    )
    rateAggregation_ext: Optional[Element] = Field(
        description="Placeholder element for rateAggregation extensions",
        default=None,
        alias="_rateAggregation",
    )
    rationale: Optional[Markdown] = Field(
        description="Detailed description of why the measure exists",
        default=None,
    )
    rationale_ext: Optional[Element] = Field(
        description="Placeholder element for rationale extensions",
        default=None,
        alias="_rationale",
    )
    clinicalRecommendationStatement: Optional[Markdown] = Field(
        description="Summary of clinical guidelines",
        default=None,
    )
    clinicalRecommendationStatement_ext: Optional[Element] = Field(
        description="Placeholder element for clinicalRecommendationStatement extensions",
        default=None,
        alias="_clinicalRecommendationStatement",
    )
    improvementNotation: Optional[CodeableConcept] = Field(
        description="increase | decrease",
        default=None,
    )
    term: Optional[List[MeasureTerm]] = Field(
        description="Defined terms used in the measure documentation",
        default=None,
    )
    guidance: Optional[Markdown] = Field(
        description="Additional guidance for implementers (deprecated)",
        default=None,
    )
    guidance_ext: Optional[Element] = Field(
        description="Placeholder element for guidance extensions",
        default=None,
        alias="_guidance",
    )
    group: Optional[List[MeasureGroup]] = Field(
        description="Population criteria group",
        default=None,
    )
    supplementalData: Optional[List[MeasureSupplementalData]] = Field(
        description="What other data should be reported with the measure",
        default=None,
    )
    resourceType: Literal["Measure"] = Field(
        description=None,
        default="Measure",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @property
    def subject(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="subject",
        )

    @field_validator(
        *(
            "supplementalData",
            "group",
            "guidance",
            "term",
            "improvementNotation",
            "clinicalRecommendationStatement",
            "rationale",
            "rateAggregation",
            "riskAdjustment",
            "type",
            "compositeScoring",
            "scoringUnit",
            "scoring",
            "disclaimer",
            "library",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyrightLabel",
            "copyright",
            "usage",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "basis",
            "experimental",
            "status",
            "subtitle",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def subject_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="subject",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_mea_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="group.stratifier.all((code | description | criteria).exists() xor component.exists())",
            human="Stratifier SHALL be either a single criteria or a set of criteria components",
            key="mea-1",
            severity="error",
        )
