# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Date,
    Boolean,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    MarketingStatus,
    BackboneElement,
    Quantity,
    Attachment,
    CodeableReference,
    DomainResource,
)


class ManufacturedItemDefinitionProperty(BackboneElement):
    """
    General characteristics of this item.
    """

    type: Optional[CodeableConcept] = Field(
        description="A code expressing the type of characteristic",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate: Optional[Date] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueMarkdown: Optional[Markdown] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for valueMarkdown extensions",
        default=None,
        alias="_valueMarkdown",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="A value for the characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                CodeableConcept,
                Quantity,
                Date,
                Boolean,
                Markdown,
                Attachment,
                Reference,
            ],
            field_name_base="value",
            required=False,
        )


class ManufacturedItemDefinitionComponentConstituent(BackboneElement):
    """
    A reference to a constituent of the manufactured item as a whole, linked here so that its component location within the item can be indicated. This not where the item's ingredient are primarily stated (for which see Ingredient.for or ManufacturedItemDefinition.ingredient).
    """

    amount: Optional[List[Quantity]] = Field(
        description="The measurable amount of the substance, expressable in different ways (e.g. by mass or volume)",
        default=None,
    )
    location: Optional[List[CodeableConcept]] = Field(
        description="The physical location of the constituent/ingredient within the component",
        default=None,
    )
    function: Optional[List[CodeableConcept]] = Field(
        description="The function of this constituent within the component e.g. binder",
        default=None,
    )
    hasIngredient: Optional[List[CodeableReference]] = Field(
        description="The ingredient that is the constituent of the given component",
        default=None,
    )

    @field_validator(
        *(
            "hasIngredient",
            "function",
            "location",
            "amount",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ManufacturedItemDefinitionComponentProperty(BackboneElement):
    """
    General characteristics of this component.
    """

    type: Optional[CodeableConcept] = Field(
        description="A code expressing the type of characteristic",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate: Optional[Date] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueMarkdown: Optional[Markdown] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for valueMarkdown extensions",
        default=None,
        alias="_valueMarkdown",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="A value for the characteristic",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="A value for the characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                CodeableConcept,
                Quantity,
                Date,
                Boolean,
                Markdown,
                Attachment,
                Reference,
            ],
            field_name_base="value",
            required=False,
        )


class ManufacturedItemDefinitionComponent(BackboneElement):
    """
    Physical parts of the manufactured item, that it is intrisically made from. This is distinct from the ingredients that are part of its chemical makeup.
    """

    type: Optional[CodeableConcept] = Field(
        description="Defining type of the component e.g. shell, layer, ink",
        default=None,
    )
    function: Optional[List[CodeableConcept]] = Field(
        description="The function of this component within the item e.g. delivers active ingredient, masks taste",
        default=None,
    )
    amount: Optional[List[Quantity]] = Field(
        description="The measurable amount of total quantity of all substances in the component, expressable in different ways (e.g. by mass or volume)",
        default=None,
    )
    constituent: Optional[List[ManufacturedItemDefinitionComponentConstituent]] = Field(
        description="A reference to a constituent of the manufactured item as a whole, linked here so that its component location within the item can be indicated. This not where the item\u0027s ingredient are primarily stated (for which see Ingredient.for or ManufacturedItemDefinition.ingredient)",
        default=None,
    )
    property_: Optional[List[ManufacturedItemDefinitionComponentProperty]] = Field(
        description="General characteristics of this component",
        default=None,
    )
    component: Optional[List["ManufacturedItemDefinitionComponent"]] = Field(
        description="A component that this component contains or is made from",
        default=None,
    )

    @field_validator(
        *(
            "component",
            "property_",
            "constituent",
            "amount",
            "function",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ManufacturedItemDefinition(DomainResource):
    """
    The definition and characteristics of a medicinal manufactured item, such as a tablet or capsule, as contained in a packaged medicinal product.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/ManufacturedItemDefinition"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Unique identifier",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    name: Optional[String] = Field(
        description="A descriptive name applied to this item",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    manufacturedDoseForm: Optional[CodeableConcept] = Field(
        description="Dose form as manufactured (before any necessary transformation)",
        default=None,
    )
    unitOfPresentation: Optional[CodeableConcept] = Field(
        description="The \u201creal-world\u201d units in which the quantity of the item is described",
        default=None,
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="Manufacturer of the item, one of several possible",
        default=None,
    )
    marketingStatus: Optional[List[MarketingStatus]] = Field(
        description="Allows specifying that an item is on the market for sale, or that it is not available, and the dates and locations associated",
        default=None,
    )
    ingredient: Optional[List[CodeableConcept]] = Field(
        description="The ingredients of this manufactured item. Only needed if these are not specified by incoming references from the Ingredient resource",
        default=None,
    )
    property_: Optional[List[ManufacturedItemDefinitionProperty]] = Field(
        description="General characteristics of this item",
        default=None,
    )
    component: Optional[List[ManufacturedItemDefinitionComponent]] = Field(
        description="Physical parts of the manufactured item, that it is intrisically made from. This is distinct from the ingredients that are part of its chemical makeup",
        default=None,
    )
    resourceType: Literal["ManufacturedItemDefinition"] = Field(
        description=None,
        default="ManufacturedItemDefinition",
    )

    @field_validator(
        *(
            "component",
            "property_",
            "ingredient",
            "marketingStatus",
            "manufacturer",
            "unitOfPresentation",
            "manufacturedDoseForm",
            "name",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
