# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Markdown,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    CodeableConcept,
    ExtendedContactDetail,
    Address,
    BackboneElement,
    Reference,
    Availability,
    VirtualServiceDetail,
    DomainResource,
)


class LocationPosition(BackboneElement):
    """
    The absolute geographic location of the Location, expressed using the WGS84 datum (This is the same co-ordinate system used in KML).
    """

    longitude: Optional[Decimal] = Field(
        description="Longitude with WGS84 datum",
        default=None,
    )
    longitude_ext: Optional[Element] = Field(
        description="Placeholder element for longitude extensions",
        default=None,
        alias="_longitude",
    )
    latitude: Optional[Decimal] = Field(
        description="Latitude with WGS84 datum",
        default=None,
    )
    latitude_ext: Optional[Element] = Field(
        description="Placeholder element for latitude extensions",
        default=None,
        alias="_latitude",
    )
    altitude: Optional[Decimal] = Field(
        description="Altitude with WGS84 datum",
        default=None,
    )
    altitude_ext: Optional[Element] = Field(
        description="Placeholder element for altitude extensions",
        default=None,
        alias="_altitude",
    )

    @field_validator(
        *(
            "altitude",
            "latitude",
            "longitude",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Location(DomainResource):
    """
    Details and position information for a place where services are provided and resources and participants may be stored, found, contained, or accommodated.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Location"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Unique code or number identifying the location to its users",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | suspended | inactive",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    operationalStatus: Optional[Coding] = Field(
        description="The operational status of the location (typically only for a bed/room)",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name of the location as used by humans",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    alias: Optional[List[String]] = Field(
        description="A list of alternate names that the location is known as, or was known as, in the past",
        default=None,
    )
    alias_ext: Optional[Element] = Field(
        description="Placeholder element for alias extensions",
        default=None,
        alias="_alias",
    )
    description: Optional[Markdown] = Field(
        description="Additional details about the location that could be displayed as further information to identify the location beyond its name",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    mode: Optional[Code] = Field(
        description="instance | kind",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Type of function performed",
        default=None,
    )
    contact: Optional[List[ExtendedContactDetail]] = Field(
        description="Official contact details for the location",
        default=None,
    )
    address: Optional[Address] = Field(
        description="Physical location",
        default=None,
    )
    form: Optional[CodeableConcept] = Field(
        description="Physical form of the location",
        default=None,
    )
    position: Optional[LocationPosition] = Field(
        description="The absolute geographic location",
        default=None,
    )
    managingOrganization: Optional[Reference] = Field(
        description="Organization responsible for provisioning and upkeep",
        default=None,
    )
    partOf: Optional[Reference] = Field(
        description="Another Location this one is physically a part of",
        default=None,
    )
    characteristic: Optional[List[CodeableConcept]] = Field(
        description="Collection of characteristics (attributes)",
        default=None,
    )
    hoursOfOperation: Optional[List[Availability]] = Field(
        description="What days/times during a week is this location usually open (including exceptions)",
        default=None,
    )
    virtualService: Optional[List[VirtualServiceDetail]] = Field(
        description="Connection details of a virtual service (e.g. conference call)",
        default=None,
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Technical endpoints providing access to services operated for the location",
        default=None,
    )
    resourceType: Literal["Location"] = Field(
        description=None,
        default="Location",
    )

    @field_validator(
        *(
            "endpoint",
            "virtualService",
            "hoursOfOperation",
            "characteristic",
            "partOf",
            "managingOrganization",
            "position",
            "form",
            "address",
            "contact",
            "type",
            "mode",
            "description",
            "alias",
            "name",
            "operationalStatus",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
