# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Quantity,
    CodeableReference,
    Annotation,
    DomainResource,
)


class InventoryReportInventoryListingItem(BackboneElement):
    """
    The item or items in this listing.
    """

    category: Optional[CodeableConcept] = Field(
        description="The inventory category or classification of the items being reported",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The quantity of the item or items being reported",
        default=None,
    )
    item: Optional[CodeableReference] = Field(
        description="The code or reference to the item type",
        default=None,
    )

    @field_validator(
        *(
            "item",
            "quantity",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InventoryReportInventoryListing(BackboneElement):
    """
    An inventory listing section (grouped by any of the attributes).
    """

    location: Optional[Reference] = Field(
        description="Location of the inventory items",
        default=None,
    )
    itemStatus: Optional[CodeableConcept] = Field(
        description="The status of the items that are being reported",
        default=None,
    )
    countingDateTime: Optional[DateTime] = Field(
        description="The date and time when the items were counted",
        default=None,
    )
    countingDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for countingDateTime extensions",
        default=None,
        alias="_countingDateTime",
    )
    item: Optional[List[InventoryReportInventoryListingItem]] = Field(
        description="The item or items in this listing",
        default=None,
    )

    @field_validator(
        *(
            "item",
            "countingDateTime",
            "itemStatus",
            "location",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InventoryReport(DomainResource):
    """
    A report of inventory or stock items.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/InventoryReport"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for the report",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | requested | active | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    countType: Optional[Code] = Field(
        description="snapshot | difference",
        default=None,
    )
    countType_ext: Optional[Element] = Field(
        description="Placeholder element for countType extensions",
        default=None,
        alias="_countType",
    )
    operationType: Optional[CodeableConcept] = Field(
        description="addition | subtraction",
        default=None,
    )
    operationTypeReason: Optional[CodeableConcept] = Field(
        description="The reason for this count - regular count, ad-hoc count, new arrivals, etc",
        default=None,
    )
    reportedDateTime: Optional[DateTime] = Field(
        description="When the report has been submitted",
        default=None,
    )
    reportedDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for reportedDateTime extensions",
        default=None,
        alias="_reportedDateTime",
    )
    reporter: Optional[Reference] = Field(
        description="Who submits the report",
        default=None,
    )
    reportingPeriod: Optional[Period] = Field(
        description="The period the report refers to",
        default=None,
    )
    inventoryListing: Optional[List[InventoryReportInventoryListing]] = Field(
        description="An inventory listing section (grouped by any of the attributes)",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="A note associated with the InventoryReport",
        default=None,
    )
    resourceType: Literal["InventoryReport"] = Field(
        description=None,
        default="InventoryReport",
    )

    @field_validator(
        *(
            "note",
            "inventoryListing",
            "reportingPeriod",
            "reporter",
            "reportedDateTime",
            "operationTypeReason",
            "operationType",
            "countType",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
