# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Integer,
    Decimal,
    Boolean,
    Url,
    DateTime,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    BackboneElement,
    Coding,
    Reference,
    Quantity,
    Ratio,
    Range,
    Annotation,
    Address,
    Duration,
    DomainResource,
)


class InventoryItemName(BackboneElement):
    """
    The item name(s) - the brand name, or common name, functional name, generic name.
    """

    nameType: Optional[Coding] = Field(
        description="The type of name e.g. \u0027brand-name\u0027, \u0027functional-name\u0027, \u0027common-name\u0027",
        default=None,
    )
    language: Optional[Code] = Field(
        description="The language used to express the item name",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    name: Optional[String] = Field(
        description="The name or designation of the item",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )

    @field_validator(
        *(
            "name",
            "language",
            "nameType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InventoryItemResponsibleOrganization(BackboneElement):
    """
    Organization(s) responsible for the product.
    """

    role: Optional[CodeableConcept] = Field(
        description="The role of the organization e.g. manufacturer, distributor, or other",
        default=None,
    )
    organization: Optional[Reference] = Field(
        description="An organization that is associated with the item",
        default=None,
    )

    @field_validator(
        *(
            "organization",
            "role",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InventoryItemDescription(BackboneElement):
    """
    The descriptive characteristics of the inventory item.
    """

    language: Optional[Code] = Field(
        description="The language that is used in the item description",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    description: Optional[String] = Field(
        description="Textual description of the item",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "language",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InventoryItemAssociation(BackboneElement):
    """
    Association with other items or products.
    """

    associationType: Optional[CodeableConcept] = Field(
        description="The type of association between the device and the other item",
        default=None,
    )
    relatedItem: Optional[Reference] = Field(
        description="The related item or product",
        default=None,
    )
    quantity: Optional[Ratio] = Field(
        description="The quantity of the product in this product",
        default=None,
    )

    @field_validator(
        *(
            "quantity",
            "relatedItem",
            "associationType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InventoryItemCharacteristic(BackboneElement):
    """
    The descriptive or identifying characteristics of the item.
    """

    characteristicType: Optional[CodeableConcept] = Field(
        description="The characteristic that is being defined",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueInteger: Optional[Integer] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueUrl: Optional[Url] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueUrl_ext: Optional[Element] = Field(
        description="Placeholder element for valueUrl extensions",
        default=None,
        alias="_valueUrl",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueRange: Optional[Range] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueRatio: Optional[Ratio] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueAnnotation: Optional[Annotation] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueAddress: Optional[Address] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueDuration: Optional[Duration] = Field(
        description="The value of the attribute",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="The value of the attribute",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("characteristicType", "modifierExtension", "extension"),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                String,
                Integer,
                Decimal,
                Boolean,
                Url,
                DateTime,
                Quantity,
                Range,
                Ratio,
                Annotation,
                Address,
                Duration,
                CodeableConcept,
            ],
            field_name_base="value",
            required=True,
        )


class InventoryItemInstance(BackboneElement):
    """
    Instances or occurrences of the product.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="The identifier for the physical instance, typically a serial number",
        default=None,
    )
    lotNumber: Optional[String] = Field(
        description="The lot or batch number of the item",
        default=None,
    )
    lotNumber_ext: Optional[Element] = Field(
        description="Placeholder element for lotNumber extensions",
        default=None,
        alias="_lotNumber",
    )
    expiry: Optional[DateTime] = Field(
        description="The expiry date or date and time for the product",
        default=None,
    )
    expiry_ext: Optional[Element] = Field(
        description="Placeholder element for expiry extensions",
        default=None,
        alias="_expiry",
    )
    subject: Optional[Reference] = Field(
        description="The subject that the item is associated with",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="The location that the item is associated with",
        default=None,
    )

    @field_validator(
        *(
            "location",
            "subject",
            "expiry",
            "lotNumber",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InventoryItem(DomainResource):
    """
    functional description of an inventory item used in inventory and supply-related workflows.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/InventoryItem"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for the inventory item",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | inactive | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Category or class of the item",
        default=None,
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="Code designating the specific type of item",
        default=None,
    )
    name: Optional[List[InventoryItemName]] = Field(
        description="The item name(s) - the brand name, or common name, functional name, generic name or others",
        default=None,
    )
    responsibleOrganization: Optional[List[InventoryItemResponsibleOrganization]] = (
        Field(
            description="Organization(s) responsible for the product",
            default=None,
        )
    )
    description: Optional[InventoryItemDescription] = Field(
        description="Descriptive characteristics of the item",
        default=None,
    )
    inventoryStatus: Optional[List[CodeableConcept]] = Field(
        description="The usage status like recalled, in use, discarded",
        default=None,
    )
    baseUnit: Optional[CodeableConcept] = Field(
        description="The base unit of measure - the unit in which the product is used or counted",
        default=None,
    )
    netContent: Optional[Quantity] = Field(
        description="Net content or amount present in the item",
        default=None,
    )
    association: Optional[List[InventoryItemAssociation]] = Field(
        description="Association with other items or products",
        default=None,
    )
    characteristic: Optional[List[InventoryItemCharacteristic]] = Field(
        description="Characteristic of the item",
        default=None,
    )
    instance: Optional[InventoryItemInstance] = Field(
        description="Instances or occurrences of the product",
        default=None,
    )
    productReference: Optional[Reference] = Field(
        description="Link to a product resource used in clinical workflows",
        default=None,
    )
    resourceType: Literal["InventoryItem"] = Field(
        description=None,
        default="InventoryItem",
    )

    @field_validator(
        *(
            "productReference",
            "instance",
            "characteristic",
            "association",
            "netContent",
            "baseUnit",
            "inventoryStatus",
            "description",
            "responsibleOrganization",
            "name",
            "code",
            "category",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
