# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, PositiveInt

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Period,
    Reference,
    ExtendedContactDetail,
    BackboneElement,
    Quantity,
    Money,
    DomainResource,
)


class InsurancePlanCoverageBenefitLimit(BackboneElement):
    """
    The specific limits on the benefit.
    """

    value: Optional[Quantity] = Field(
        description="Maximum value allowed",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Benefit limit details",
        default=None,
    )

    @field_validator(
        *(
            "code",
            "value",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlanCoverageBenefit(BackboneElement):
    """
    Specific benefits under this type of coverage.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of benefit",
        default=None,
    )
    requirement: Optional[String] = Field(
        description="Referral requirements",
        default=None,
    )
    requirement_ext: Optional[Element] = Field(
        description="Placeholder element for requirement extensions",
        default=None,
        alias="_requirement",
    )
    limit: Optional[List[InsurancePlanCoverageBenefitLimit]] = Field(
        description="Benefit limits",
        default=None,
    )

    @field_validator(
        *(
            "limit",
            "requirement",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlanCoverage(BackboneElement):
    """
    Details about the coverage offered by the insurance product.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of coverage",
        default=None,
    )
    network: Optional[List[Reference]] = Field(
        description="What networks provide coverage",
        default=None,
    )
    benefit: Optional[List[InsurancePlanCoverageBenefit]] = Field(
        description="List of benefits",
        default=None,
    )

    @field_validator(
        *(
            "benefit",
            "network",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlanPlanGeneralCost(BackboneElement):
    """
    Overall costs associated with the plan.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of cost",
        default=None,
    )
    groupSize: Optional[PositiveInt] = Field(
        description="Number of enrollees",
        default=None,
    )
    groupSize_ext: Optional[Element] = Field(
        description="Placeholder element for groupSize extensions",
        default=None,
        alias="_groupSize",
    )
    cost: Optional[Money] = Field(
        description="Cost value",
        default=None,
    )
    comment: Optional[String] = Field(
        description="Additional cost information",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )

    @field_validator(
        *(
            "comment",
            "cost",
            "groupSize",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlanPlanSpecificCostBenefitCost(BackboneElement):
    """
    List of the costs associated with a specific benefit.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of cost",
        default=None,
    )
    applicability: Optional[CodeableConcept] = Field(
        description="in-network | out-of-network | other",
        default=None,
    )
    qualifiers: Optional[List[CodeableConcept]] = Field(
        description="Additional information about the cost",
        default=None,
    )
    value: Optional[Quantity] = Field(
        description="The actual cost value",
        default=None,
    )

    @field_validator(
        *(
            "value",
            "qualifiers",
            "applicability",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlanPlanSpecificCostBenefit(BackboneElement):
    """
    List of the specific benefits under this category of benefit.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of specific benefit",
        default=None,
    )
    cost: Optional[List[InsurancePlanPlanSpecificCostBenefitCost]] = Field(
        description="List of the costs",
        default=None,
    )

    @field_validator(
        *(
            "cost",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlanPlanSpecificCost(BackboneElement):
    """
    Costs associated with the coverage provided by the product.
    """

    category: Optional[CodeableConcept] = Field(
        description="General category of benefit",
        default=None,
    )
    benefit: Optional[List[InsurancePlanPlanSpecificCostBenefit]] = Field(
        description="Benefits list",
        default=None,
    )

    @field_validator(
        *(
            "benefit",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlanPlan(BackboneElement):
    """
    Details about an insurance plan.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for Product",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of plan",
        default=None,
    )
    coverageArea: Optional[List[Reference]] = Field(
        description="Where product applies",
        default=None,
    )
    network: Optional[List[Reference]] = Field(
        description="What networks provide coverage",
        default=None,
    )
    generalCost: Optional[List[InsurancePlanPlanGeneralCost]] = Field(
        description="Overall costs",
        default=None,
    )
    specificCost: Optional[List[InsurancePlanPlanSpecificCost]] = Field(
        description="Specific costs",
        default=None,
    )

    @field_validator(
        *(
            "specificCost",
            "generalCost",
            "network",
            "coverageArea",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class InsurancePlan(DomainResource):
    """
    Details of a Health Insurance product/plan provided by an organization.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/InsurancePlan"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for Product",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Kind of product",
        default=None,
    )
    name: Optional[String] = Field(
        description="Official name",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    alias: Optional[List[String]] = Field(
        description="Alternate names",
        default=None,
    )
    alias_ext: Optional[Element] = Field(
        description="Placeholder element for alias extensions",
        default=None,
        alias="_alias",
    )
    period: Optional[Period] = Field(
        description="When the product is available",
        default=None,
    )
    ownedBy: Optional[Reference] = Field(
        description="Product issuer",
        default=None,
    )
    administeredBy: Optional[Reference] = Field(
        description="Product administrator",
        default=None,
    )
    coverageArea: Optional[List[Reference]] = Field(
        description="Where product applies",
        default=None,
    )
    contact: Optional[List[ExtendedContactDetail]] = Field(
        description="Official contact details relevant to the health insurance plan/product",
        default=None,
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Technical endpoint",
        default=None,
    )
    network: Optional[List[Reference]] = Field(
        description="What networks are Included",
        default=None,
    )
    coverage: Optional[List[InsurancePlanCoverage]] = Field(
        description="Coverage details",
        default=None,
    )
    plan: Optional[List[InsurancePlanPlan]] = Field(
        description="Plan details",
        default=None,
    )
    resourceType: Literal["InsurancePlan"] = Field(
        description=None,
        default="InsurancePlan",
    )

    @field_validator(
        *(
            "plan",
            "coverage",
            "network",
            "endpoint",
            "contact",
            "coverageArea",
            "administeredBy",
            "ownedBy",
            "period",
            "alias",
            "name",
            "type",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_ipn_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(identifier.count() + name.count()) > 0",
            human="The organization SHALL at least have a name or an identifier, and possibly more than one",
            key="ipn-1",
            severity="error",
        )
