# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Id,
    Canonical,
    Url,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Reference,
    DomainResource,
)


class ImplementationGuideDependsOn(BackboneElement):
    """
    Another implementation guide that this implementation depends on. Typically, an implementation guide uses value sets, profiles etc.defined in other implementation guides.
    """

    uri: Optional[Canonical] = Field(
        description="Identity of the IG that this depends on",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    packageId: Optional[Id] = Field(
        description="NPM Package name for IG this depends on",
        default=None,
    )
    packageId_ext: Optional[Element] = Field(
        description="Placeholder element for packageId extensions",
        default=None,
        alias="_packageId",
    )
    version: Optional[String] = Field(
        description="Version of the IG",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    reason: Optional[Markdown] = Field(
        description="Why dependency exists",
        default=None,
    )
    reason_ext: Optional[Element] = Field(
        description="Placeholder element for reason extensions",
        default=None,
        alias="_reason",
    )

    @field_validator(
        *(
            "reason",
            "version",
            "packageId",
            "uri",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideGlobal(BackboneElement):
    """
    A set of profiles that all resources covered by this implementation guide must conform to.
    """

    type: Optional[Code] = Field(
        description="Type this profile applies to",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    profile: Optional[Canonical] = Field(
        description="Profile that all resources must conform to",
        default=None,
    )
    profile_ext: Optional[Element] = Field(
        description="Placeholder element for profile extensions",
        default=None,
        alias="_profile",
    )

    @field_validator(
        *(
            "profile",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinitionGrouping(BackboneElement):
    """
    A logical group of resources. Logical groups can be used when building pages.
    """

    name: Optional[String] = Field(
        description="Descriptive name for the package",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[Markdown] = Field(
        description="Human readable text describing the package",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinitionResource(BackboneElement):
    """
    A resource that is part of the implementation guide. Conformance resources (value set, structure definition, capability statements etc.) are obvious candidates for inclusion, but any kind of resource can be included as an example resource.
    """

    reference: Optional[Reference] = Field(
        description="Location of the resource",
        default=None,
    )
    fhirVersion: Optional[List[Code]] = Field(
        description="Versions this applies to (if different to IG)",
        default=None,
    )
    fhirVersion_ext: Optional[Element] = Field(
        description="Placeholder element for fhirVersion extensions",
        default=None,
        alias="_fhirVersion",
    )
    name: Optional[String] = Field(
        description="Human readable name for the resource",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[Markdown] = Field(
        description="Reason why included in guide",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    isExample: Optional[Boolean] = Field(
        description="Is this an example",
        default=None,
    )
    isExample_ext: Optional[Element] = Field(
        description="Placeholder element for isExample extensions",
        default=None,
        alias="_isExample",
    )
    profile: Optional[List[Canonical]] = Field(
        description="Profile(s) this is an example of",
        default=None,
    )
    profile_ext: Optional[Element] = Field(
        description="Placeholder element for profile extensions",
        default=None,
        alias="_profile",
    )
    groupingId: Optional[Id] = Field(
        description="Grouping this is part of",
        default=None,
    )
    groupingId_ext: Optional[Element] = Field(
        description="Placeholder element for groupingId extensions",
        default=None,
        alias="_groupingId",
    )

    @field_validator(
        *(
            "groupingId",
            "profile",
            "isExample",
            "description",
            "name",
            "fhirVersion",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinitionPage(BackboneElement):
    """
    A page / section in the implementation guide. The root page is the implementation guide home page.
    """

    sourceUrl: Optional[Url] = Field(
        description="Source for page",
        default=None,
    )
    sourceUrl_ext: Optional[Element] = Field(
        description="Placeholder element for sourceUrl extensions",
        default=None,
        alias="_sourceUrl",
    )
    sourceString: Optional[String] = Field(
        description="Source for page",
        default=None,
    )
    sourceString_ext: Optional[Element] = Field(
        description="Placeholder element for sourceString extensions",
        default=None,
        alias="_sourceString",
    )
    sourceMarkdown: Optional[Markdown] = Field(
        description="Source for page",
        default=None,
    )
    sourceMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for sourceMarkdown extensions",
        default=None,
        alias="_sourceMarkdown",
    )
    name: Optional[Url] = Field(
        description="Name of the page when published",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Short title shown for navigational assistance",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    generation: Optional[Code] = Field(
        description="html | markdown | xml | generated",
        default=None,
    )
    generation_ext: Optional[Element] = Field(
        description="Placeholder element for generation extensions",
        default=None,
        alias="_generation",
    )
    page: Optional[List["ImplementationGuideDefinitionPage"]] = Field(
        description="Nested Pages / Sections",
        default=None,
    )

    @property
    def source(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="source",
        )

    @field_validator(
        *(
            "page",
            "generation",
            "title",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def source_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Url, String, Markdown],
            field_name_base="source",
            required=False,
        )


class ImplementationGuideDefinitionParameter(BackboneElement):
    """
    A set of parameters that defines how the implementation guide is built. The parameters are defined by the relevant tools that build the implementation guides.
    """

    code: Optional[Coding] = Field(
        description="Code that identifies parameter",
        default=None,
    )
    value: Optional[String] = Field(
        description="Value for named type",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinitionTemplate(BackboneElement):
    """
    A template for building resources.
    """

    code: Optional[Code] = Field(
        description="Type of template specified",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    source: Optional[String] = Field(
        description="The source location for the template",
        default=None,
    )
    source_ext: Optional[Element] = Field(
        description="Placeholder element for source extensions",
        default=None,
        alias="_source",
    )
    scope: Optional[String] = Field(
        description="The scope in which the template applies",
        default=None,
    )
    scope_ext: Optional[Element] = Field(
        description="Placeholder element for scope extensions",
        default=None,
        alias="_scope",
    )

    @field_validator(
        *(
            "scope",
            "source",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinition(BackboneElement):
    """
    The information needed by an IG publisher tool to publish the whole implementation guide.
    """

    grouping: Optional[List[ImplementationGuideDefinitionGrouping]] = Field(
        description="Grouping used to present related resources in the IG",
        default=None,
    )
    resource: Optional[List[ImplementationGuideDefinitionResource]] = Field(
        description="Resource in the implementation guide",
        default=None,
    )
    page: Optional[ImplementationGuideDefinitionPage] = Field(
        description="Page/Section in the Guide",
        default=None,
    )
    parameter: Optional[List[ImplementationGuideDefinitionParameter]] = Field(
        description="Defines how IG is built by tools",
        default=None,
    )
    template: Optional[List[ImplementationGuideDefinitionTemplate]] = Field(
        description="A template for building resources",
        default=None,
    )

    @field_validator(
        *(
            "template",
            "parameter",
            "page",
            "resource",
            "grouping",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("page",), mode="after", check_fields=None)
    @classmethod
    def FHIR_ig_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="generation='generated' implies source.empty()",
            human="Source must be absent if 'generated' is generated",
            key="ig-3",
            severity="error",
        )


class ImplementationGuideManifestResource(BackboneElement):
    """
    A resource that is part of the implementation guide. Conformance resources (value set, structure definition, capability statements etc.) are obvious candidates for inclusion, but any kind of resource can be included as an example resource.
    """

    reference: Optional[Reference] = Field(
        description="Location of the resource",
        default=None,
    )
    isExample: Optional[Boolean] = Field(
        description="Is this an example",
        default=None,
    )
    isExample_ext: Optional[Element] = Field(
        description="Placeholder element for isExample extensions",
        default=None,
        alias="_isExample",
    )
    profile: Optional[List[Canonical]] = Field(
        description="Profile(s) this is an example of",
        default=None,
    )
    profile_ext: Optional[Element] = Field(
        description="Placeholder element for profile extensions",
        default=None,
        alias="_profile",
    )
    relativePath: Optional[Url] = Field(
        description="Relative path for page in IG",
        default=None,
    )
    relativePath_ext: Optional[Element] = Field(
        description="Placeholder element for relativePath extensions",
        default=None,
        alias="_relativePath",
    )

    @field_validator(
        *(
            "relativePath",
            "profile",
            "isExample",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideManifestPage(BackboneElement):
    """
    Information about a page within the IG.
    """

    name: Optional[String] = Field(
        description="HTML page name",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Title of the page, for references",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    anchor: Optional[List[String]] = Field(
        description="Anchor available on the page",
        default=None,
    )
    anchor_ext: Optional[Element] = Field(
        description="Placeholder element for anchor extensions",
        default=None,
        alias="_anchor",
    )

    @field_validator(
        *(
            "anchor",
            "title",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideManifest(BackboneElement):
    """
    Information about an assembled implementation guide, created by the publication tooling.
    """

    rendering: Optional[Url] = Field(
        description="Location of rendered implementation guide",
        default=None,
    )
    rendering_ext: Optional[Element] = Field(
        description="Placeholder element for rendering extensions",
        default=None,
        alias="_rendering",
    )
    resource: Optional[List[ImplementationGuideManifestResource]] = Field(
        description="Resource in the implementation guide",
        default=None,
    )
    page: Optional[List[ImplementationGuideManifestPage]] = Field(
        description="HTML page within the parent IG",
        default=None,
    )
    image: Optional[List[String]] = Field(
        description="Image within the IG",
        default=None,
    )
    image_ext: Optional[Element] = Field(
        description="Placeholder element for image extensions",
        default=None,
        alias="_image",
    )
    other: Optional[List[String]] = Field(
        description="Additional linkable file in IG",
        default=None,
    )
    other_ext: Optional[Element] = Field(
        description="Placeholder element for other extensions",
        default=None,
        alias="_other",
    )

    @field_validator(
        *(
            "other",
            "image",
            "page",
            "resource",
            "rendering",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuide(DomainResource):
    """
    A set of rules of how a particular interoperability or standards problem is solved - typically through the use of FHIR resources. This resource is used to gather all the parts of an implementation guide into a logical whole and to publish a computable definition of all the parts.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ImplementationGuide"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this implementation guide, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the implementation guide (business identifier)",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the implementation guide",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this implementation guide (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this implementation guide (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the implementation guide",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for implementation guide (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this implementation guide is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    packageId: Optional[Id] = Field(
        description="NPM Package name for IG",
        default=None,
    )
    packageId_ext: Optional[Element] = Field(
        description="Placeholder element for packageId extensions",
        default=None,
        alias="_packageId",
    )
    license: Optional[Code] = Field(
        description="SPDX license code for this IG (or not-open-source)",
        default=None,
    )
    license_ext: Optional[Element] = Field(
        description="Placeholder element for license extensions",
        default=None,
        alias="_license",
    )
    fhirVersion: Optional[List[Code]] = Field(
        description="FHIR Version(s) this Implementation Guide targets",
        default=None,
    )
    fhirVersion_ext: Optional[Element] = Field(
        description="Placeholder element for fhirVersion extensions",
        default=None,
        alias="_fhirVersion",
    )
    dependsOn: Optional[List[ImplementationGuideDependsOn]] = Field(
        description="Another Implementation guide this depends on",
        default=None,
    )
    global_: Optional[List[ImplementationGuideGlobal]] = Field(
        description="Profiles that apply globally",
        default=None,
    )
    definition: Optional[ImplementationGuideDefinition] = Field(
        description="Information needed to build the IG",
        default=None,
    )
    manifest: Optional[ImplementationGuideManifest] = Field(
        description="Information about an assembled IG",
        default=None,
    )
    resourceType: Literal["ImplementationGuide"] = Field(
        description=None,
        default="ImplementationGuide",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "manifest",
            "definition",
            "global_",
            "dependsOn",
            "fhirVersion",
            "license",
            "packageId",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @field_validator(*("definition",), mode="after", check_fields=None)
    @classmethod
    def FHIR_ig_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="resource.groupingId.all(%context.grouping.id contains $this)",
            human="If a resource has a groupingId, it must refer to a grouping defined in the Implementation Guide",
            key="ig-1",
            severity="error",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_ig_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="definition.resource.fhirVersion.all(%context.fhirVersion contains $this)",
            human="If a resource has a fhirVersion, it must be one of the versions defined for the Implementation Guide",
            key="ig-2",
            severity="error",
        )
