# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    BackboneElement,
    CodeableConcept,
    DomainResource,
)


class ImmunizationRecommendationRecommendationDateCriterion(BackboneElement):
    """
    Vaccine date recommendations.  For example, earliest date to administer, latest date to administer, etc.
    """

    code: Optional[CodeableConcept] = Field(
        description="Type of date",
        default=None,
    )
    value: Optional[DateTime] = Field(
        description="Recommended date",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationRecommendationRecommendation(BackboneElement):
    """
    Vaccine administration recommendations.
    """

    vaccineCode: Optional[List[CodeableConcept]] = Field(
        description="Vaccine  or vaccine group recommendation applies to",
        default=None,
    )
    targetDisease: Optional[List[CodeableConcept]] = Field(
        description="Disease to be immunized against",
        default=None,
    )
    contraindicatedVaccineCode: Optional[List[CodeableConcept]] = Field(
        description="Vaccine which is contraindicated to fulfill the recommendation",
        default=None,
    )
    forecastStatus: Optional[CodeableConcept] = Field(
        description="Vaccine recommendation status",
        default=None,
    )
    forecastReason: Optional[List[CodeableConcept]] = Field(
        description="Vaccine administration status reason",
        default=None,
    )
    dateCriterion: Optional[
        List[ImmunizationRecommendationRecommendationDateCriterion]
    ] = Field(
        description="Dates governing proposed immunization",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Protocol details",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    series: Optional[String] = Field(
        description="Name of vaccination series",
        default=None,
    )
    series_ext: Optional[Element] = Field(
        description="Placeholder element for series extensions",
        default=None,
        alias="_series",
    )
    doseNumber: Optional[String] = Field(
        description="Recommended dose number within series",
        default=None,
    )
    doseNumber_ext: Optional[Element] = Field(
        description="Placeholder element for doseNumber extensions",
        default=None,
        alias="_doseNumber",
    )
    seriesDoses: Optional[String] = Field(
        description="Recommended number of doses for immunity",
        default=None,
    )
    seriesDoses_ext: Optional[Element] = Field(
        description="Placeholder element for seriesDoses extensions",
        default=None,
        alias="_seriesDoses",
    )
    supportingImmunization: Optional[List[Reference]] = Field(
        description="Past immunizations supporting recommendation",
        default=None,
    )
    supportingPatientInformation: Optional[List[Reference]] = Field(
        description="Patient observations supporting recommendation",
        default=None,
    )

    @field_validator(
        *(
            "supportingPatientInformation",
            "supportingImmunization",
            "seriesDoses",
            "doseNumber",
            "series",
            "description",
            "dateCriterion",
            "forecastReason",
            "forecastStatus",
            "contraindicatedVaccineCode",
            "targetDisease",
            "vaccineCode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationRecommendation(DomainResource):
    """
    A patient's point-in-time set of recommendations (i.e. forecasting) according to a published schedule with optional supporting justification.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/ImmunizationRecommendation"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="Who this profile is for",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="Date recommendation(s) created",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    authority: Optional[Reference] = Field(
        description="Who is responsible for protocol",
        default=None,
    )
    recommendation: Optional[List[ImmunizationRecommendationRecommendation]] = Field(
        description="Vaccine administration recommendations",
        default=None,
    )
    resourceType: Literal["ImmunizationRecommendation"] = Field(
        description=None,
        default="ImmunizationRecommendation",
    )

    @field_validator(
        *(
            "recommendation",
            "authority",
            "date",
            "patient",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("recommendation",), mode="after", check_fields=None)
    @classmethod
    def FHIR_imr_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="vaccineCode.exists() or targetDisease.exists()",
            human="One of vaccineCode or targetDisease SHALL be present",
            key="imr-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
