# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Date,
    DateTime,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    CodeableReference,
    Quantity,
    BackboneElement,
    Annotation,
    DomainResource,
)


class ImmunizationPerformer(BackboneElement):
    """
    Indicates who performed the immunization event.
    """

    function: Optional[CodeableConcept] = Field(
        description="What type of performance was done",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Individual or organization who was performing",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationProgramEligibility(BackboneElement):
    """
    Indicates a patient's eligibility for a funding program.
    """

    program: Optional[CodeableConcept] = Field(
        description="The program that eligibility is declared for",
        default=None,
    )
    programStatus: Optional[CodeableConcept] = Field(
        description="The patient\u0027s eligibility status for the program",
        default=None,
    )

    @field_validator(
        *(
            "programStatus",
            "program",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationReaction(BackboneElement):
    """
    Categorical data indicating that an adverse event is associated in time to an immunization.
    """

    date: Optional[DateTime] = Field(
        description="When reaction started",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    manifestation: Optional[CodeableReference] = Field(
        description="Additional information on reaction",
        default=None,
    )
    reported: Optional[Boolean] = Field(
        description="Indicates self-reported reaction",
        default=None,
    )
    reported_ext: Optional[Element] = Field(
        description="Placeholder element for reported extensions",
        default=None,
        alias="_reported",
    )

    @field_validator(
        *(
            "reported",
            "manifestation",
            "date",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationProtocolApplied(BackboneElement):
    """
    The protocol (set of recommendations) being followed by the provider who administered the dose.
    """

    series: Optional[String] = Field(
        description="Name of vaccine series",
        default=None,
    )
    series_ext: Optional[Element] = Field(
        description="Placeholder element for series extensions",
        default=None,
        alias="_series",
    )
    authority: Optional[Reference] = Field(
        description="Who is responsible for publishing the recommendations",
        default=None,
    )
    targetDisease: Optional[List[CodeableConcept]] = Field(
        description="Vaccine preventatable disease being targeted",
        default=None,
    )
    doseNumber: Optional[String] = Field(
        description="Dose number within series",
        default=None,
    )
    doseNumber_ext: Optional[Element] = Field(
        description="Placeholder element for doseNumber extensions",
        default=None,
        alias="_doseNumber",
    )
    seriesDoses: Optional[String] = Field(
        description="Recommended number of doses for immunity",
        default=None,
    )
    seriesDoses_ext: Optional[Element] = Field(
        description="Placeholder element for seriesDoses extensions",
        default=None,
        alias="_seriesDoses",
    )

    @field_validator(
        *(
            "seriesDoses",
            "doseNumber",
            "targetDisease",
            "authority",
            "series",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Immunization(DomainResource):
    """
    Describes the event of a patient being administered a vaccine or a record of an immunization as reported by a patient, a clinician or another party.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Immunization"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Authority that the immunization event is based on",
        default=None,
    )
    status: Optional[Code] = Field(
        description="completed | entered-in-error | not-done",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableConcept] = Field(
        description="Reason for current status",
        default=None,
    )
    vaccineCode: Optional[CodeableConcept] = Field(
        description="Vaccine administered",
        default=None,
    )
    administeredProduct: Optional[CodeableReference] = Field(
        description="Product that was administered",
        default=None,
    )
    manufacturer: Optional[CodeableReference] = Field(
        description="Vaccine manufacturer",
        default=None,
    )
    lotNumber: Optional[String] = Field(
        description="Vaccine lot number",
        default=None,
    )
    lotNumber_ext: Optional[Element] = Field(
        description="Placeholder element for lotNumber extensions",
        default=None,
        alias="_lotNumber",
    )
    expirationDate: Optional[Date] = Field(
        description="Vaccine expiration date",
        default=None,
    )
    expirationDate_ext: Optional[Element] = Field(
        description="Placeholder element for expirationDate extensions",
        default=None,
        alias="_expirationDate",
    )
    patient: Optional[Reference] = Field(
        description="Who was immunized",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter immunization was part of",
        default=None,
    )
    supportingInformation: Optional[List[Reference]] = Field(
        description="Additional information in support of the immunization",
        default=None,
    )
    occurrenceDateTime: Optional[DateTime] = Field(
        description="Vaccine administration date",
        default=None,
    )
    occurrenceDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDateTime extensions",
        default=None,
        alias="_occurrenceDateTime",
    )
    occurrenceString: Optional[String] = Field(
        description="Vaccine administration date",
        default=None,
    )
    occurrenceString_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceString extensions",
        default=None,
        alias="_occurrenceString",
    )
    primarySource: Optional[Boolean] = Field(
        description="Indicates context the data was captured in",
        default=None,
    )
    primarySource_ext: Optional[Element] = Field(
        description="Placeholder element for primarySource extensions",
        default=None,
        alias="_primarySource",
    )
    informationSource: Optional[CodeableReference] = Field(
        description="Indicates the source of a  reported record",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where immunization occurred",
        default=None,
    )
    site: Optional[CodeableConcept] = Field(
        description="Body site vaccine  was administered",
        default=None,
    )
    route: Optional[CodeableConcept] = Field(
        description="How vaccine entered body",
        default=None,
    )
    doseQuantity: Optional[Quantity] = Field(
        description="Amount of vaccine administered",
        default=None,
    )
    performer: Optional[List[ImmunizationPerformer]] = Field(
        description="Who performed event",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional immunization notes",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Why immunization occurred",
        default=None,
    )
    isSubpotent: Optional[Boolean] = Field(
        description="Dose potency",
        default=None,
    )
    isSubpotent_ext: Optional[Element] = Field(
        description="Placeholder element for isSubpotent extensions",
        default=None,
        alias="_isSubpotent",
    )
    subpotentReason: Optional[List[CodeableConcept]] = Field(
        description="Reason for being subpotent",
        default=None,
    )
    programEligibility: Optional[List[ImmunizationProgramEligibility]] = Field(
        description="Patient eligibility for a specific vaccination program",
        default=None,
    )
    fundingSource: Optional[CodeableConcept] = Field(
        description="Funding source for the vaccine",
        default=None,
    )
    reaction: Optional[List[ImmunizationReaction]] = Field(
        description="Details of a reaction that follows immunization",
        default=None,
    )
    protocolApplied: Optional[List[ImmunizationProtocolApplied]] = Field(
        description="Protocol followed by the provider",
        default=None,
    )
    resourceType: Literal["Immunization"] = Field(
        description=None,
        default="Immunization",
    )

    @property
    def occurrence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurrence",
        )

    @field_validator(
        *(
            "protocolApplied",
            "reaction",
            "fundingSource",
            "programEligibility",
            "subpotentReason",
            "isSubpotent",
            "reason",
            "note",
            "performer",
            "doseQuantity",
            "route",
            "site",
            "location",
            "informationSource",
            "primarySource",
            "supportingInformation",
            "encounter",
            "patient",
            "expirationDate",
            "lotNumber",
            "manufacturer",
            "administeredProduct",
            "vaccineCode",
            "statusReason",
            "status",
            "basedOn",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurrence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, String],
            field_name_base="occurrence",
            required=True,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
