# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Instant,
    Id,
    UnsignedInt,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    BackboneElement,
    CodeableConcept,
    CodeableReference,
    Coding,
    DomainResource,
)


class ImagingSelectionPerformer(BackboneElement):
    """
    Selector of the instances – human or machine.
    """

    function: Optional[CodeableConcept] = Field(
        description="Type of performer",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Author (human or machine)",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImagingSelectionInstanceImageRegion2D(BackboneElement):
    """
    Each imaging selection instance or frame list might includes an image region, specified by a region type and a set of 2D coordinates.
       If the parent imagingSelection.instance contains a subset element of type frame, the image region applies to all frames in the subset list.
    """

    regionType: Optional[Code] = Field(
        description="point | polyline | interpolated | circle | ellipse",
        default=None,
    )
    regionType_ext: Optional[Element] = Field(
        description="Placeholder element for regionType extensions",
        default=None,
        alias="_regionType",
    )
    coordinate: Optional[List[Decimal]] = Field(
        description="Specifies the coordinates that define the image region",
        default=None,
    )
    coordinate_ext: Optional[Element] = Field(
        description="Placeholder element for coordinate extensions",
        default=None,
        alias="_coordinate",
    )

    @field_validator(
        *(
            "coordinate",
            "regionType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImagingSelectionInstanceImageRegion3D(BackboneElement):
    """
    Each imaging selection might includes a 3D image region, specified by a region type and a set of 3D coordinates.
    """

    regionType: Optional[Code] = Field(
        description="point | multipoint | polyline | polygon | ellipse | ellipsoid",
        default=None,
    )
    regionType_ext: Optional[Element] = Field(
        description="Placeholder element for regionType extensions",
        default=None,
        alias="_regionType",
    )
    coordinate: Optional[List[Decimal]] = Field(
        description="Specifies the coordinates that define the image region",
        default=None,
    )
    coordinate_ext: Optional[Element] = Field(
        description="Placeholder element for coordinate extensions",
        default=None,
        alias="_coordinate",
    )

    @field_validator(
        *(
            "coordinate",
            "regionType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImagingSelectionInstance(BackboneElement):
    """
    Each imaging selection includes one or more selected DICOM SOP instances.
    """

    uid: Optional[Id] = Field(
        description="DICOM SOP Instance UID",
        default=None,
    )
    uid_ext: Optional[Element] = Field(
        description="Placeholder element for uid extensions",
        default=None,
        alias="_uid",
    )
    number: Optional[UnsignedInt] = Field(
        description="DICOM Instance Number",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    sopClass: Optional[Coding] = Field(
        description="DICOM SOP Class UID",
        default=None,
    )
    subset: Optional[List[String]] = Field(
        description="The selected subset of the SOP Instance",
        default=None,
    )
    subset_ext: Optional[Element] = Field(
        description="Placeholder element for subset extensions",
        default=None,
        alias="_subset",
    )
    imageRegion2D: Optional[List[ImagingSelectionInstanceImageRegion2D]] = Field(
        description="A specific 2D region in a DICOM image / frame",
        default=None,
    )
    imageRegion3D: Optional[List[ImagingSelectionInstanceImageRegion3D]] = Field(
        description="A specific 3D region in a DICOM frame of reference",
        default=None,
    )

    @field_validator(
        *(
            "imageRegion3D",
            "imageRegion2D",
            "subset",
            "sopClass",
            "number",
            "uid",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImagingSelection(DomainResource):
    """
    A selection of DICOM SOP instances and/or frames within a single Study and Series. This might include additional specifics such as an image region, an Observation UID or a Segmentation Number, allowing linkage to an Observation Resource or transferring this information along with the ImagingStudy Resource.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ImagingSelection"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for Imaging Selection",
        default=None,
    )
    status: Optional[Code] = Field(
        description="available | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    subject: Optional[Reference] = Field(
        description="Subject of the selected instances",
        default=None,
    )
    issued: Optional[Instant] = Field(
        description="Date / Time when this imaging selection was created",
        default=None,
    )
    issued_ext: Optional[Element] = Field(
        description="Placeholder element for issued extensions",
        default=None,
        alias="_issued",
    )
    performer: Optional[List[ImagingSelectionPerformer]] = Field(
        description="Selector of the instances (human or machine)",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Associated request",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Classifies the imaging selection",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Imaging Selection purpose text or code",
        default=None,
    )
    studyUid: Optional[Id] = Field(
        description="DICOM Study Instance UID",
        default=None,
    )
    studyUid_ext: Optional[Element] = Field(
        description="Placeholder element for studyUid extensions",
        default=None,
        alias="_studyUid",
    )
    derivedFrom: Optional[List[Reference]] = Field(
        description="The imaging study from which the imaging selection is derived",
        default=None,
    )
    endpoint: Optional[List[Reference]] = Field(
        description="The network service providing retrieval for the images referenced in the imaging selection",
        default=None,
    )
    seriesUid: Optional[Id] = Field(
        description="DICOM Series Instance UID",
        default=None,
    )
    seriesUid_ext: Optional[Element] = Field(
        description="Placeholder element for seriesUid extensions",
        default=None,
        alias="_seriesUid",
    )
    seriesNumber: Optional[UnsignedInt] = Field(
        description="DICOM Series Number",
        default=None,
    )
    seriesNumber_ext: Optional[Element] = Field(
        description="Placeholder element for seriesNumber extensions",
        default=None,
        alias="_seriesNumber",
    )
    frameOfReferenceUid: Optional[Id] = Field(
        description="The Frame of Reference UID for the selected images",
        default=None,
    )
    frameOfReferenceUid_ext: Optional[Element] = Field(
        description="Placeholder element for frameOfReferenceUid extensions",
        default=None,
        alias="_frameOfReferenceUid",
    )
    bodySite: Optional[CodeableReference] = Field(
        description="Body part examined",
        default=None,
    )
    focus: Optional[List[Reference]] = Field(
        description="Related resource that is the focus for the imaging selection",
        default=None,
    )
    instance: Optional[List[ImagingSelectionInstance]] = Field(
        description="The selected instances",
        default=None,
    )
    resourceType: Literal["ImagingSelection"] = Field(
        description=None,
        default="ImagingSelection",
    )

    @field_validator(
        *(
            "instance",
            "focus",
            "bodySite",
            "frameOfReferenceUid",
            "seriesNumber",
            "seriesUid",
            "endpoint",
            "derivedFrom",
            "studyUid",
            "code",
            "category",
            "basedOn",
            "performer",
            "issued",
            "subject",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
