# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Attachment,
    ExtendedContactDetail,
    BackboneElement,
    Availability,
    DomainResource,
)


class HealthcareServiceEligibility(BackboneElement):
    """
    Does this service have specific eligibility requirements that need to be met in order to use the service?
    """

    code: Optional[CodeableConcept] = Field(
        description="Coded value for the eligibility",
        default=None,
    )
    comment: Optional[Markdown] = Field(
        description="Describes the eligibility conditions for the service",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )

    @field_validator(
        *(
            "comment",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class HealthcareService(DomainResource):
    """
    The details of a healthcare service available at a location or in a catalog.  In the case where there is a hierarchy of services (for example, Lab -> Pathology -> Wound Cultures), this can be represented using a set of linked HealthcareServices.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/HealthcareService"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External identifiers for this item",
        default=None,
    )
    active: Optional[Boolean] = Field(
        description="Whether this HealthcareService record is in active use",
        default=None,
    )
    active_ext: Optional[Element] = Field(
        description="Placeholder element for active extensions",
        default=None,
        alias="_active",
    )
    providedBy: Optional[Reference] = Field(
        description="Organization that provides this service",
        default=None,
    )
    offeredIn: Optional[List[Reference]] = Field(
        description="The service within which this service is offered",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Broad category of service being performed or delivered",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Type of service that may be delivered or performed",
        default=None,
    )
    specialty: Optional[List[CodeableConcept]] = Field(
        description="Specialties handled by the HealthcareService",
        default=None,
    )
    location: Optional[List[Reference]] = Field(
        description="Location(s) where service may be provided",
        default=None,
    )
    name: Optional[String] = Field(
        description="Description of service as presented to a consumer while searching",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    comment: Optional[Markdown] = Field(
        description="Additional description and/or any specific issues not covered elsewhere",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )
    extraDetails: Optional[Markdown] = Field(
        description="Extra details about the service that can\u0027t be placed in the other fields",
        default=None,
    )
    extraDetails_ext: Optional[Element] = Field(
        description="Placeholder element for extraDetails extensions",
        default=None,
        alias="_extraDetails",
    )
    photo: Optional[Attachment] = Field(
        description="Facilitates quick identification of the service",
        default=None,
    )
    contact: Optional[List[ExtendedContactDetail]] = Field(
        description="Official contact details for the HealthcareService",
        default=None,
    )
    coverageArea: Optional[List[Reference]] = Field(
        description="Location(s) service is intended for/available to",
        default=None,
    )
    serviceProvisionCode: Optional[List[CodeableConcept]] = Field(
        description="Conditions under which service is available/offered",
        default=None,
    )
    eligibility: Optional[List[HealthcareServiceEligibility]] = Field(
        description="Specific eligibility requirements required to use the service",
        default=None,
    )
    program: Optional[List[CodeableConcept]] = Field(
        description="Programs that this service is applicable to",
        default=None,
    )
    characteristic: Optional[List[CodeableConcept]] = Field(
        description="Collection of characteristics (attributes)",
        default=None,
    )
    communication: Optional[List[CodeableConcept]] = Field(
        description="The language that this service is offered in",
        default=None,
    )
    referralMethod: Optional[List[CodeableConcept]] = Field(
        description="Ways that the service accepts referrals",
        default=None,
    )
    appointmentRequired: Optional[Boolean] = Field(
        description="If an appointment is required for access to this service",
        default=None,
    )
    appointmentRequired_ext: Optional[Element] = Field(
        description="Placeholder element for appointmentRequired extensions",
        default=None,
        alias="_appointmentRequired",
    )
    availability: Optional[List[Availability]] = Field(
        description="Times the healthcare service is available (including exceptions)",
        default=None,
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Technical endpoints providing access to electronic services operated for the healthcare service",
        default=None,
    )
    resourceType: Literal["HealthcareService"] = Field(
        description=None,
        default="HealthcareService",
    )

    @field_validator(
        *(
            "endpoint",
            "availability",
            "appointmentRequired",
            "referralMethod",
            "communication",
            "characteristic",
            "program",
            "eligibility",
            "serviceProvisionCode",
            "coverageArea",
            "contact",
            "photo",
            "extraDetails",
            "comment",
            "name",
            "location",
            "specialty",
            "type",
            "category",
            "offeredIn",
            "providedBy",
            "active",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
