# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Canonical,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    CodeableReference,
    Annotation,
    BackboneElement,
    DomainResource,
)


class GenomicStudyAnalysisInput(BackboneElement):
    """
    Inputs for the analysis event.
    """

    file: Optional[Reference] = Field(
        description="File containing input data",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of input data (e.g., BAM, CRAM, or FASTA)",
        default=None,
    )
    generatedByIdentifier: Optional[Identifier] = Field(
        description="The analysis event or other GenomicStudy that generated this input file",
        default=None,
    )
    generatedByReference: Optional[Reference] = Field(
        description="The analysis event or other GenomicStudy that generated this input file",
        default=None,
    )

    @property
    def generatedBy(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="generatedBy",
        )

    @field_validator(
        *(
            "type",
            "file",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def generatedBy_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Identifier, Reference],
            field_name_base="generatedBy",
            required=False,
        )


class GenomicStudyAnalysisOutput(BackboneElement):
    """
    Outputs for the analysis event.
    """

    file: Optional[Reference] = Field(
        description="File containing output data",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of output data (e.g., VCF, MAF, or BAM)",
        default=None,
    )

    @field_validator(
        *(
            "type",
            "file",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class GenomicStudyAnalysisPerformer(BackboneElement):
    """
    Performer for the analysis event.
    """

    actor: Optional[Reference] = Field(
        description="The organization, healthcare professional, or others who participated in performing this analysis",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="Role of the actor for this analysis",
        default=None,
    )

    @field_validator(
        *(
            "role",
            "actor",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class GenomicStudyAnalysisDevice(BackboneElement):
    """
    Devices used for the analysis (e.g., instruments, software), with settings and parameters.
    """

    device: Optional[Reference] = Field(
        description="Device used for the analysis",
        default=None,
    )
    function: Optional[CodeableConcept] = Field(
        description="Specific function for the device used for the analysis",
        default=None,
    )

    @field_validator(
        *(
            "function",
            "device",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class GenomicStudyAnalysis(BackboneElement):
    """
    The details about a specific analysis that was performed in this GenomicStudy.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="Identifiers for the analysis event",
        default=None,
    )
    methodType: Optional[List[CodeableConcept]] = Field(
        description="Type of the methods used in the analysis (e.g., FISH, Karyotyping, MSI)",
        default=None,
    )
    changeType: Optional[List[CodeableConcept]] = Field(
        description="Type of the genomic changes studied in the analysis (e.g., DNA, RNA, or AA change)",
        default=None,
    )
    genomeBuild: Optional[CodeableConcept] = Field(
        description="Genome build that is used in this analysis",
        default=None,
    )
    instantiatesCanonical: Optional[Canonical] = Field(
        description="The defined protocol that describes the analysis",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[Uri] = Field(
        description="The URL pointing to an externally maintained protocol that describes the analysis",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    title: Optional[String] = Field(
        description="Name of the analysis event (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    focus: Optional[List[Reference]] = Field(
        description="What the genomic analysis is about, when it is not about the subject of record",
        default=None,
    )
    specimen: Optional[List[Reference]] = Field(
        description="The specimen used in the analysis event",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="The date of the analysis event",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    note: Optional[List[Annotation]] = Field(
        description="Any notes capture with the analysis event",
        default=None,
    )
    protocolPerformed: Optional[Reference] = Field(
        description="The protocol that was performed for the analysis event",
        default=None,
    )
    regionsStudied: Optional[List[Reference]] = Field(
        description="The genomic regions to be studied in the analysis (BED file)",
        default=None,
    )
    regionsCalled: Optional[List[Reference]] = Field(
        description="Genomic regions actually called in the analysis event (BED file)",
        default=None,
    )
    input: Optional[List[GenomicStudyAnalysisInput]] = Field(
        description="Inputs for the analysis event",
        default=None,
    )
    output: Optional[List[GenomicStudyAnalysisOutput]] = Field(
        description="Outputs for the analysis event",
        default=None,
    )
    performer: Optional[List[GenomicStudyAnalysisPerformer]] = Field(
        description="Performer for the analysis event",
        default=None,
    )
    device: Optional[List[GenomicStudyAnalysisDevice]] = Field(
        description="Devices used for the analysis (e.g., instruments, software), with settings and parameters",
        default=None,
    )

    @field_validator(
        *(
            "device",
            "performer",
            "output",
            "input",
            "regionsCalled",
            "regionsStudied",
            "protocolPerformed",
            "note",
            "date",
            "specimen",
            "focus",
            "title",
            "instantiatesUri",
            "instantiatesCanonical",
            "genomeBuild",
            "changeType",
            "methodType",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class GenomicStudy(DomainResource):
    """
    A set of analyses performed to analyze and generate genomic data.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/GenomicStudy"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifiers for this genomic study",
        default=None,
    )
    status: Optional[Code] = Field(
        description="registered | available | cancelled | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="The type of the study (e.g., Familial variant segregation, Functional variation detection, or Gene expression profiling)",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="The primary subject of the genomic study",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="The healthcare event with which this genomics study is associated",
        default=None,
    )
    startDate: Optional[DateTime] = Field(
        description="When the genomic study was started",
        default=None,
    )
    startDate_ext: Optional[Element] = Field(
        description="Placeholder element for startDate extensions",
        default=None,
        alias="_startDate",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Event resources that the genomic study is based on",
        default=None,
    )
    referrer: Optional[Reference] = Field(
        description="Healthcare professional who requested or referred the genomic study",
        default=None,
    )
    interpreter: Optional[List[Reference]] = Field(
        description="Healthcare professionals who interpreted the genomic study",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Why the genomic study was performed",
        default=None,
    )
    instantiatesCanonical: Optional[Canonical] = Field(
        description="The defined protocol that describes the study",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[Uri] = Field(
        description="The URL pointing to an externally maintained protocol that describes the study",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments related to the genomic study",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Description of the genomic study",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    analysis: Optional[List[GenomicStudyAnalysis]] = Field(
        description="Genomic Analysis Event",
        default=None,
    )
    resourceType: Literal["GenomicStudy"] = Field(
        description=None,
        default="GenomicStudy",
    )

    @field_validator(
        *(
            "analysis",
            "description",
            "note",
            "instantiatesUri",
            "instantiatesCanonical",
            "reason",
            "interpreter",
            "referrer",
            "basedOn",
            "startDate",
            "encounter",
            "subject",
            "type",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
