# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    PositiveInt,
    Boolean,
    Date,
    Decimal,
    UnsignedInt,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Quantity,
    Attachment,
    Coding,
    Address,
    Money,
    CodeableReference,
    DomainResource,
)


class ExplanationOfBenefitRelated(BackboneElement):
    """
    Other claims which are related to this claim such as prior submissions or claims for related services or for the same event.
    """

    claim: Optional[Reference] = Field(
        description="Reference to the related claim",
        default=None,
    )
    relationship: Optional[CodeableConcept] = Field(
        description="How the reference claim is related",
        default=None,
    )
    reference: Optional[Identifier] = Field(
        description="File or case reference",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "relationship",
            "claim",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitEvent(BackboneElement):
    """
    Information code for an event with a corresponding date or period.
    """

    type: Optional[CodeableConcept] = Field(
        description="Specific event",
        default=None,
    )
    whenDateTime: Optional[DateTime] = Field(
        description="Occurance date or period",
        default=None,
    )
    whenDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for whenDateTime extensions",
        default=None,
        alias="_whenDateTime",
    )
    whenPeriod: Optional[Period] = Field(
        description="Occurance date or period",
        default=None,
    )

    @property
    def when(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="when",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def when_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="when",
            required=True,
        )


class ExplanationOfBenefitPayee(BackboneElement):
    """
    The party to be reimbursed for cost of the products and services according to the terms of the policy.
    """

    type: Optional[CodeableConcept] = Field(
        description="Category of recipient",
        default=None,
    )
    party: Optional[Reference] = Field(
        description="Recipient reference",
        default=None,
    )

    @field_validator(
        *(
            "party",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitCareTeam(BackboneElement):
    """
    The members of the team who provided the products and services.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Order of care team",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    provider: Optional[Reference] = Field(
        description="Practitioner or organization",
        default=None,
    )
    responsible: Optional[Boolean] = Field(
        description="Indicator of the lead practitioner",
        default=None,
    )
    responsible_ext: Optional[Element] = Field(
        description="Placeholder element for responsible extensions",
        default=None,
        alias="_responsible",
    )
    role: Optional[CodeableConcept] = Field(
        description="Function within the team",
        default=None,
    )
    specialty: Optional[CodeableConcept] = Field(
        description="Practitioner or provider specialization",
        default=None,
    )

    @field_validator(
        *(
            "specialty",
            "role",
            "responsible",
            "provider",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitSupportingInfo(BackboneElement):
    """
    Additional information codes regarding exceptions, special considerations, the condition, situation, prior or concurrent issues.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Information instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    category: Optional[CodeableConcept] = Field(
        description="Classification of the supplied information",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Type of information",
        default=None,
    )
    timingDate: Optional[Date] = Field(
        description="When it occurred",
        default=None,
    )
    timingDate_ext: Optional[Element] = Field(
        description="Placeholder element for timingDate extensions",
        default=None,
        alias="_timingDate",
    )
    timingPeriod: Optional[Period] = Field(
        description="When it occurred",
        default=None,
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Data to be provided",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueString: Optional[String] = Field(
        description="Data to be provided",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Data to be provided",
        default=None,
    )
    valueAttachment: Optional[Attachment] = Field(
        description="Data to be provided",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="Data to be provided",
        default=None,
    )
    valueIdentifier: Optional[Identifier] = Field(
        description="Data to be provided",
        default=None,
    )
    reason: Optional[Coding] = Field(
        description="Explanation for the information",
        default=None,
    )

    @property
    def timing(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="timing",
        )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "reason",
            "code",
            "category",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def timing_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Period],
            field_name_base="timing",
            required=False,
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, String, Quantity, Attachment, Reference, Identifier],
            field_name_base="value",
            required=False,
        )


class ExplanationOfBenefitDiagnosis(BackboneElement):
    """
    Information about diagnoses relevant to the claim items.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Diagnosis instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    diagnosisCodeableConcept: Optional[CodeableConcept] = Field(
        description="Nature of illness or problem",
        default=None,
    )
    diagnosisReference: Optional[Reference] = Field(
        description="Nature of illness or problem",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Timing or nature of the diagnosis",
        default=None,
    )
    onAdmission: Optional[CodeableConcept] = Field(
        description="Present on admission",
        default=None,
    )

    @property
    def diagnosis(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="diagnosis",
        )

    @field_validator(
        *(
            "onAdmission",
            "type",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def diagnosis_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="diagnosis",
            required=True,
        )


class ExplanationOfBenefitProcedure(BackboneElement):
    """
    Procedures performed on the patient relevant to the billing items with the claim.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Procedure instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Category of Procedure",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="When the procedure was performed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    procedureCodeableConcept: Optional[CodeableConcept] = Field(
        description="Specific clinical procedure",
        default=None,
    )
    procedureReference: Optional[Reference] = Field(
        description="Specific clinical procedure",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )

    @property
    def procedure(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="procedure",
        )

    @field_validator(
        *(
            "udi",
            "date",
            "type",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def procedure_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="procedure",
            required=True,
        )


class ExplanationOfBenefitInsurance(BackboneElement):
    """
    Financial instruments for reimbursement for the health care products and services specified on the claim.
    """

    focal: Optional[Boolean] = Field(
        description="Coverage to be used for adjudication",
        default=None,
    )
    focal_ext: Optional[Element] = Field(
        description="Placeholder element for focal extensions",
        default=None,
        alias="_focal",
    )
    coverage: Optional[Reference] = Field(
        description="Insurance information",
        default=None,
    )
    preAuthRef: Optional[List[String]] = Field(
        description="Prior authorization reference number",
        default=None,
    )
    preAuthRef_ext: Optional[Element] = Field(
        description="Placeholder element for preAuthRef extensions",
        default=None,
        alias="_preAuthRef",
    )

    @field_validator(
        *(
            "preAuthRef",
            "coverage",
            "focal",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitAccident(BackboneElement):
    """
    Details of a accident which resulted in injuries which required the products and services listed in the claim.
    """

    date: Optional[Date] = Field(
        description="When the incident occurred",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    type: Optional[CodeableConcept] = Field(
        description="The nature of the accident",
        default=None,
    )
    locationAddress: Optional[Address] = Field(
        description="Where the event occurred",
        default=None,
    )
    locationReference: Optional[Reference] = Field(
        description="Where the event occurred",
        default=None,
    )

    @property
    def location(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="location",
        )

    @field_validator(
        *(
            "type",
            "date",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def location_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Address, Reference],
            field_name_base="location",
            required=False,
        )


class ExplanationOfBenefitItemBodySite(BackboneElement):
    """
    Physical location where the service is performed or applies.
    """

    site: Optional[List[CodeableReference]] = Field(
        description="Location",
        default=None,
    )
    subSite: Optional[List[CodeableConcept]] = Field(
        description="Sub-location",
        default=None,
    )

    @field_validator(
        *(
            "subSite",
            "site",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitItemReviewOutcome(BackboneElement):
    """
    The high-level results of the adjudication if adjudication has been performed.
    """

    decision: Optional[CodeableConcept] = Field(
        description="Result of the adjudication",
        default=None,
    )
    reason: Optional[List[CodeableConcept]] = Field(
        description="Reason for result of the adjudication",
        default=None,
    )
    preAuthRef: Optional[String] = Field(
        description="Preauthorization reference",
        default=None,
    )
    preAuthRef_ext: Optional[Element] = Field(
        description="Placeholder element for preAuthRef extensions",
        default=None,
        alias="_preAuthRef",
    )
    preAuthPeriod: Optional[Period] = Field(
        description="Preauthorization reference effective period",
        default=None,
    )

    @field_validator(
        *(
            "preAuthPeriod",
            "preAuthRef",
            "reason",
            "decision",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitItemAdjudication(BackboneElement):
    """
    If this item is a group then the values here are a summary of the adjudication of the detail items. If this item is a simple product or service then this is the result of the adjudication of this item.
    """

    category: Optional[CodeableConcept] = Field(
        description="Type of adjudication information",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Explanation of adjudication outcome",
        default=None,
    )
    amount: Optional[Money] = Field(
        description="Monetary amount",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Non-monitary value",
        default=None,
    )

    @field_validator(
        *(
            "quantity",
            "amount",
            "reason",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitItemDetailReviewOutcome(BackboneElement):
    """
    The high-level results of the adjudication if adjudication has been performed.
    """

    decision: Optional[CodeableConcept] = Field(
        description="Result of the adjudication",
        default=None,
    )
    reason: Optional[List[CodeableConcept]] = Field(
        description="Reason for result of the adjudication",
        default=None,
    )
    preAuthRef: Optional[String] = Field(
        description="Preauthorization reference",
        default=None,
    )
    preAuthRef_ext: Optional[Element] = Field(
        description="Placeholder element for preAuthRef extensions",
        default=None,
        alias="_preAuthRef",
    )
    preAuthPeriod: Optional[Period] = Field(
        description="Preauthorization reference effective period",
        default=None,
    )

    @field_validator(
        *(
            "preAuthPeriod",
            "preAuthRef",
            "reason",
            "decision",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitItemDetailAdjudication(BackboneElement):
    """
    The adjudication results.
    """

    category: Optional[CodeableConcept] = Field(
        description="Type of adjudication information",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Explanation of adjudication outcome",
        default=None,
    )
    amount: Optional[Money] = Field(
        description="Monetary amount",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Non-monitary value",
        default=None,
    )

    @field_validator(
        *(
            "quantity",
            "amount",
            "reason",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitItemDetailSubDetail(BackboneElement):
    """
    Third-tier of goods and services.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Product or service provided",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    traceNumber: Optional[List[Identifier]] = Field(
        description="Number for tracking",
        default=None,
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    productOrServiceEnd: Optional[CodeableConcept] = Field(
        description="End of a range of codes",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    patientPaid: Optional[Money] = Field(
        description="Paid by the patient",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    tax: Optional[Money] = Field(
        description="Total tax",
        default=None,
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    reviewOutcome: Optional[ExplanationOfBenefitItemReviewOutcome] = Field(
        description="Subdetail level adjudication results",
        default=None,
    )
    adjudication: Optional[List[ExplanationOfBenefitItemAdjudication]] = Field(
        description="Subdetail level adjudication details",
        default=None,
    )

    @field_validator(
        *(
            "adjudication",
            "reviewOutcome",
            "noteNumber",
            "udi",
            "net",
            "tax",
            "factor",
            "unitPrice",
            "quantity",
            "patientPaid",
            "programCode",
            "modifier",
            "productOrServiceEnd",
            "productOrService",
            "category",
            "revenue",
            "traceNumber",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitItemDetail(BackboneElement):
    """
    Second-tier of goods and services.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Product or service provided",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    traceNumber: Optional[List[Identifier]] = Field(
        description="Number for tracking",
        default=None,
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    productOrServiceEnd: Optional[CodeableConcept] = Field(
        description="End of a range of codes",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    patientPaid: Optional[Money] = Field(
        description="Paid by the patient",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    tax: Optional[Money] = Field(
        description="Total tax",
        default=None,
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    reviewOutcome: Optional[ExplanationOfBenefitItemDetailReviewOutcome] = Field(
        description="Detail level adjudication results",
        default=None,
    )
    adjudication: Optional[List[ExplanationOfBenefitItemDetailAdjudication]] = Field(
        description="Detail level adjudication details",
        default=None,
    )
    subDetail: Optional[List[ExplanationOfBenefitItemDetailSubDetail]] = Field(
        description="Additional items",
        default=None,
    )

    @field_validator(
        *(
            "subDetail",
            "adjudication",
            "reviewOutcome",
            "noteNumber",
            "udi",
            "net",
            "tax",
            "factor",
            "unitPrice",
            "quantity",
            "patientPaid",
            "programCode",
            "modifier",
            "productOrServiceEnd",
            "productOrService",
            "category",
            "revenue",
            "traceNumber",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitItem(BackboneElement):
    """
    A claim line. Either a simple (a product or service) or a 'group' of details which can also be a simple items or groups of sub-details.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Item instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    careTeamSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable care team members",
        default=None,
    )
    careTeamSequence_ext: Optional[Element] = Field(
        description="Placeholder element for careTeamSequence extensions",
        default=None,
        alias="_careTeamSequence",
    )
    diagnosisSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable diagnoses",
        default=None,
    )
    diagnosisSequence_ext: Optional[Element] = Field(
        description="Placeholder element for diagnosisSequence extensions",
        default=None,
        alias="_diagnosisSequence",
    )
    procedureSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable procedures",
        default=None,
    )
    procedureSequence_ext: Optional[Element] = Field(
        description="Placeholder element for procedureSequence extensions",
        default=None,
        alias="_procedureSequence",
    )
    informationSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable exception and supporting information",
        default=None,
    )
    informationSequence_ext: Optional[Element] = Field(
        description="Placeholder element for informationSequence extensions",
        default=None,
        alias="_informationSequence",
    )
    traceNumber: Optional[List[Identifier]] = Field(
        description="Number for tracking",
        default=None,
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    productOrServiceEnd: Optional[CodeableConcept] = Field(
        description="End of a range of codes",
        default=None,
    )
    request: Optional[List[Reference]] = Field(
        description="Request or Referral for Service",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Product or service billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    servicedDate: Optional[Date] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    servicedDate_ext: Optional[Element] = Field(
        description="Placeholder element for servicedDate extensions",
        default=None,
        alias="_servicedDate",
    )
    servicedPeriod: Optional[Period] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    locationCodeableConcept: Optional[CodeableConcept] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationAddress: Optional[Address] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationReference: Optional[Reference] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    patientPaid: Optional[Money] = Field(
        description="Paid by the patient",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    tax: Optional[Money] = Field(
        description="Total tax",
        default=None,
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )
    bodySite: Optional[List[ExplanationOfBenefitItemBodySite]] = Field(
        description="Anatomical location",
        default=None,
    )
    encounter: Optional[List[Reference]] = Field(
        description="Encounters associated with the listed treatments",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    reviewOutcome: Optional[ExplanationOfBenefitItemReviewOutcome] = Field(
        description="Adjudication results",
        default=None,
    )
    adjudication: Optional[List[ExplanationOfBenefitItemAdjudication]] = Field(
        description="Adjudication details",
        default=None,
    )
    detail: Optional[List[ExplanationOfBenefitItemDetail]] = Field(
        description="Additional items",
        default=None,
    )

    @property
    def serviced(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="serviced",
        )

    @property
    def location(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="location",
        )

    @field_validator(
        *(
            "detail",
            "adjudication",
            "reviewOutcome",
            "noteNumber",
            "encounter",
            "bodySite",
            "udi",
            "net",
            "tax",
            "factor",
            "unitPrice",
            "quantity",
            "patientPaid",
            "programCode",
            "modifier",
            "request",
            "productOrServiceEnd",
            "productOrService",
            "category",
            "revenue",
            "traceNumber",
            "informationSequence",
            "procedureSequence",
            "diagnosisSequence",
            "careTeamSequence",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def serviced_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Period],
            field_name_base="serviced",
            required=False,
        )

    @model_validator(mode="after")
    def location_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Address, Reference],
            field_name_base="location",
            required=False,
        )


class ExplanationOfBenefitAddItemBodySite(BackboneElement):
    """
    Physical location where the service is performed or applies.
    """

    site: Optional[List[CodeableReference]] = Field(
        description="Location",
        default=None,
    )
    subSite: Optional[List[CodeableConcept]] = Field(
        description="Sub-location",
        default=None,
    )

    @field_validator(
        *(
            "subSite",
            "site",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitAddItemDetailSubDetail(BackboneElement):
    """
    The third-tier service adjudications for payor added services.
    """

    traceNumber: Optional[List[Identifier]] = Field(
        description="Number for tracking",
        default=None,
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    productOrServiceEnd: Optional[CodeableConcept] = Field(
        description="End of a range of codes",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    patientPaid: Optional[Money] = Field(
        description="Paid by the patient",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    tax: Optional[Money] = Field(
        description="Total tax",
        default=None,
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    reviewOutcome: Optional[ExplanationOfBenefitItemReviewOutcome] = Field(
        description="Additem subdetail level adjudication results",
        default=None,
    )
    adjudication: Optional[List[ExplanationOfBenefitItemAdjudication]] = Field(
        description="Added items adjudication",
        default=None,
    )

    @field_validator(
        *(
            "adjudication",
            "reviewOutcome",
            "noteNumber",
            "net",
            "tax",
            "factor",
            "unitPrice",
            "quantity",
            "patientPaid",
            "modifier",
            "productOrServiceEnd",
            "productOrService",
            "revenue",
            "traceNumber",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitAddItemDetail(BackboneElement):
    """
    The second-tier service adjudications for payor added services.
    """

    traceNumber: Optional[List[Identifier]] = Field(
        description="Number for tracking",
        default=None,
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    productOrServiceEnd: Optional[CodeableConcept] = Field(
        description="End of a range of codes",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    patientPaid: Optional[Money] = Field(
        description="Paid by the patient",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    tax: Optional[Money] = Field(
        description="Total tax",
        default=None,
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    reviewOutcome: Optional[ExplanationOfBenefitItemReviewOutcome] = Field(
        description="Additem detail level adjudication results",
        default=None,
    )
    adjudication: Optional[List[ExplanationOfBenefitItemAdjudication]] = Field(
        description="Added items adjudication",
        default=None,
    )
    subDetail: Optional[List[ExplanationOfBenefitAddItemDetailSubDetail]] = Field(
        description="Insurer added line items",
        default=None,
    )

    @field_validator(
        *(
            "subDetail",
            "adjudication",
            "reviewOutcome",
            "noteNumber",
            "net",
            "tax",
            "factor",
            "unitPrice",
            "quantity",
            "patientPaid",
            "modifier",
            "productOrServiceEnd",
            "productOrService",
            "revenue",
            "traceNumber",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitAddItem(BackboneElement):
    """
    The first-tier service adjudications for payor added product or service lines.
    """

    itemSequence: Optional[List[PositiveInt]] = Field(
        description="Item sequence number",
        default=None,
    )
    itemSequence_ext: Optional[Element] = Field(
        description="Placeholder element for itemSequence extensions",
        default=None,
        alias="_itemSequence",
    )
    detailSequence: Optional[List[PositiveInt]] = Field(
        description="Detail sequence number",
        default=None,
    )
    detailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for detailSequence extensions",
        default=None,
        alias="_detailSequence",
    )
    subDetailSequence: Optional[List[PositiveInt]] = Field(
        description="Subdetail sequence number",
        default=None,
    )
    subDetailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for subDetailSequence extensions",
        default=None,
        alias="_subDetailSequence",
    )
    traceNumber: Optional[List[Identifier]] = Field(
        description="Number for tracking",
        default=None,
    )
    provider: Optional[List[Reference]] = Field(
        description="Authorized providers",
        default=None,
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    productOrServiceEnd: Optional[CodeableConcept] = Field(
        description="End of a range of codes",
        default=None,
    )
    request: Optional[List[Reference]] = Field(
        description="Request or Referral for Service",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    servicedDate: Optional[Date] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    servicedDate_ext: Optional[Element] = Field(
        description="Placeholder element for servicedDate extensions",
        default=None,
        alias="_servicedDate",
    )
    servicedPeriod: Optional[Period] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    locationCodeableConcept: Optional[CodeableConcept] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationAddress: Optional[Address] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationReference: Optional[Reference] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    patientPaid: Optional[Money] = Field(
        description="Paid by the patient",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    tax: Optional[Money] = Field(
        description="Total tax",
        default=None,
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    bodySite: Optional[List[ExplanationOfBenefitAddItemBodySite]] = Field(
        description="Anatomical location",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    reviewOutcome: Optional[ExplanationOfBenefitItemReviewOutcome] = Field(
        description="Additem level adjudication results",
        default=None,
    )
    adjudication: Optional[List[ExplanationOfBenefitItemAdjudication]] = Field(
        description="Added items adjudication",
        default=None,
    )
    detail: Optional[List[ExplanationOfBenefitAddItemDetail]] = Field(
        description="Insurer added line items",
        default=None,
    )

    @property
    def serviced(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="serviced",
        )

    @property
    def location(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="location",
        )

    @field_validator(
        *(
            "detail",
            "adjudication",
            "reviewOutcome",
            "noteNumber",
            "bodySite",
            "net",
            "tax",
            "factor",
            "unitPrice",
            "quantity",
            "patientPaid",
            "programCode",
            "modifier",
            "request",
            "productOrServiceEnd",
            "productOrService",
            "revenue",
            "provider",
            "traceNumber",
            "subDetailSequence",
            "detailSequence",
            "itemSequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def serviced_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Period],
            field_name_base="serviced",
            required=False,
        )

    @model_validator(mode="after")
    def location_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Address, Reference],
            field_name_base="location",
            required=False,
        )


class ExplanationOfBenefitTotal(BackboneElement):
    """
    Categorized monetary totals for the adjudication.
    """

    category: Optional[CodeableConcept] = Field(
        description="Type of adjudication information",
        default=None,
    )
    amount: Optional[Money] = Field(
        description="Financial total for the category",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitPayment(BackboneElement):
    """
    Payment details for the adjudication of the claim.
    """

    type: Optional[CodeableConcept] = Field(
        description="Partial or complete payment",
        default=None,
    )
    adjustment: Optional[Money] = Field(
        description="Payment adjustment for non-claim issues",
        default=None,
    )
    adjustmentReason: Optional[CodeableConcept] = Field(
        description="Explanation for the variance",
        default=None,
    )
    date: Optional[Date] = Field(
        description="Expected date of payment",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    amount: Optional[Money] = Field(
        description="Payable amount after adjustment",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="Business identifier for the payment",
        default=None,
    )

    @field_validator(
        *(
            "identifier",
            "amount",
            "date",
            "adjustmentReason",
            "adjustment",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitProcessNote(BackboneElement):
    """
    A note that describes or explains adjudication results in a human readable form.
    """

    number: Optional[PositiveInt] = Field(
        description="Note instance identifier",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    type: Optional[CodeableConcept] = Field(
        description="Note purpose",
        default=None,
    )
    text: Optional[String] = Field(
        description="Note explanatory text",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    language: Optional[CodeableConcept] = Field(
        description="Language of the text",
        default=None,
    )

    @field_validator(
        *(
            "language",
            "text",
            "type",
            "number",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefitBenefitBalanceFinancial(BackboneElement):
    """
    Benefits Used to date.
    """

    type: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    allowedUnsignedInt: Optional[UnsignedInt] = Field(
        description="Benefits allowed",
        default=None,
    )
    allowedUnsignedInt_ext: Optional[Element] = Field(
        description="Placeholder element for allowedUnsignedInt extensions",
        default=None,
        alias="_allowedUnsignedInt",
    )
    allowedString: Optional[String] = Field(
        description="Benefits allowed",
        default=None,
    )
    allowedString_ext: Optional[Element] = Field(
        description="Placeholder element for allowedString extensions",
        default=None,
        alias="_allowedString",
    )
    allowedMoney: Optional[Money] = Field(
        description="Benefits allowed",
        default=None,
    )
    usedUnsignedInt: Optional[UnsignedInt] = Field(
        description="Benefits used",
        default=None,
    )
    usedUnsignedInt_ext: Optional[Element] = Field(
        description="Placeholder element for usedUnsignedInt extensions",
        default=None,
        alias="_usedUnsignedInt",
    )
    usedMoney: Optional[Money] = Field(
        description="Benefits used",
        default=None,
    )

    @property
    def allowed(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="allowed",
        )

    @property
    def used(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="used",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def allowed_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[UnsignedInt, String, Money],
            field_name_base="allowed",
            required=False,
        )

    @model_validator(mode="after")
    def used_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[UnsignedInt, Money],
            field_name_base="used",
            required=False,
        )


class ExplanationOfBenefitBenefitBalance(BackboneElement):
    """
    Balance by Benefit Category.
    """

    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    excluded: Optional[Boolean] = Field(
        description="Excluded from the plan",
        default=None,
    )
    excluded_ext: Optional[Element] = Field(
        description="Placeholder element for excluded extensions",
        default=None,
        alias="_excluded",
    )
    name: Optional[String] = Field(
        description="Short name for the benefit",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[String] = Field(
        description="Description of the benefit or services covered",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    network: Optional[CodeableConcept] = Field(
        description="In or out of network",
        default=None,
    )
    unit: Optional[CodeableConcept] = Field(
        description="Individual or family",
        default=None,
    )
    term: Optional[CodeableConcept] = Field(
        description="Annual or lifetime",
        default=None,
    )
    financial: Optional[List[ExplanationOfBenefitBenefitBalanceFinancial]] = Field(
        description="Benefit Summary",
        default=None,
    )

    @field_validator(
        *(
            "financial",
            "term",
            "unit",
            "network",
            "description",
            "name",
            "excluded",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExplanationOfBenefit(DomainResource):
    """
    This resource provides: the claim details; adjudication details from the processing of a Claim; and optionally account balance information, for informing the subscriber of the benefits provided.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ExplanationOfBenefit"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for the resource",
        default=None,
    )
    traceNumber: Optional[List[Identifier]] = Field(
        description="Number for tracking",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | cancelled | draft | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[CodeableConcept] = Field(
        description="Category or discipline",
        default=None,
    )
    subType: Optional[CodeableConcept] = Field(
        description="More granular claim type",
        default=None,
    )
    use: Optional[Code] = Field(
        description="claim | preauthorization | predetermination",
        default=None,
    )
    use_ext: Optional[Element] = Field(
        description="Placeholder element for use extensions",
        default=None,
        alias="_use",
    )
    patient: Optional[Reference] = Field(
        description="The recipient of the products and services",
        default=None,
    )
    billablePeriod: Optional[Period] = Field(
        description="Relevant time frame for the claim",
        default=None,
    )
    created: Optional[DateTime] = Field(
        description="Response creation date",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    enterer: Optional[Reference] = Field(
        description="Author of the claim",
        default=None,
    )
    insurer: Optional[Reference] = Field(
        description="Party responsible for reimbursement",
        default=None,
    )
    provider: Optional[Reference] = Field(
        description="Party responsible for the claim",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="Desired processing urgency",
        default=None,
    )
    fundsReserveRequested: Optional[CodeableConcept] = Field(
        description="For whom to reserve funds",
        default=None,
    )
    fundsReserve: Optional[CodeableConcept] = Field(
        description="Funds reserved status",
        default=None,
    )
    related: Optional[List[ExplanationOfBenefitRelated]] = Field(
        description="Prior or corollary claims",
        default=None,
    )
    prescription: Optional[Reference] = Field(
        description="Prescription authorizing services or products",
        default=None,
    )
    originalPrescription: Optional[Reference] = Field(
        description="Original prescription if superceded by fulfiller",
        default=None,
    )
    event: Optional[List[ExplanationOfBenefitEvent]] = Field(
        description="Event information",
        default=None,
    )
    payee: Optional[ExplanationOfBenefitPayee] = Field(
        description="Recipient of benefits payable",
        default=None,
    )
    referral: Optional[Reference] = Field(
        description="Treatment Referral",
        default=None,
    )
    encounter: Optional[List[Reference]] = Field(
        description="Encounters associated with the listed treatments",
        default=None,
    )
    facility: Optional[Reference] = Field(
        description="Servicing Facility",
        default=None,
    )
    claim: Optional[Reference] = Field(
        description="Claim reference",
        default=None,
    )
    claimResponse: Optional[Reference] = Field(
        description="Claim response reference",
        default=None,
    )
    outcome: Optional[Code] = Field(
        description="queued | complete | error | partial",
        default=None,
    )
    outcome_ext: Optional[Element] = Field(
        description="Placeholder element for outcome extensions",
        default=None,
        alias="_outcome",
    )
    decision: Optional[CodeableConcept] = Field(
        description="Result of the adjudication",
        default=None,
    )
    disposition: Optional[String] = Field(
        description="Disposition Message",
        default=None,
    )
    disposition_ext: Optional[Element] = Field(
        description="Placeholder element for disposition extensions",
        default=None,
        alias="_disposition",
    )
    preAuthRef: Optional[List[String]] = Field(
        description="Preauthorization reference",
        default=None,
    )
    preAuthRef_ext: Optional[Element] = Field(
        description="Placeholder element for preAuthRef extensions",
        default=None,
        alias="_preAuthRef",
    )
    preAuthRefPeriod: Optional[List[Period]] = Field(
        description="Preauthorization in-effect period",
        default=None,
    )
    diagnosisRelatedGroup: Optional[CodeableConcept] = Field(
        description="Package billing code",
        default=None,
    )
    careTeam: Optional[List[ExplanationOfBenefitCareTeam]] = Field(
        description="Care Team members",
        default=None,
    )
    supportingInfo: Optional[List[ExplanationOfBenefitSupportingInfo]] = Field(
        description="Supporting information",
        default=None,
    )
    diagnosis: Optional[List[ExplanationOfBenefitDiagnosis]] = Field(
        description="Pertinent diagnosis information",
        default=None,
    )
    procedure: Optional[List[ExplanationOfBenefitProcedure]] = Field(
        description="Clinical procedures performed",
        default=None,
    )
    precedence: Optional[PositiveInt] = Field(
        description="Precedence (primary, secondary, etc.)",
        default=None,
    )
    precedence_ext: Optional[Element] = Field(
        description="Placeholder element for precedence extensions",
        default=None,
        alias="_precedence",
    )
    insurance: Optional[List[ExplanationOfBenefitInsurance]] = Field(
        description="Patient insurance information",
        default=None,
    )
    accident: Optional[ExplanationOfBenefitAccident] = Field(
        description="Details of the event",
        default=None,
    )
    patientPaid: Optional[Money] = Field(
        description="Paid by the patient",
        default=None,
    )
    item: Optional[List[ExplanationOfBenefitItem]] = Field(
        description="Product or service provided",
        default=None,
    )
    addItem: Optional[List[ExplanationOfBenefitAddItem]] = Field(
        description="Insurer added line items",
        default=None,
    )
    adjudication: Optional[List[ExplanationOfBenefitItemAdjudication]] = Field(
        description="Header-level adjudication",
        default=None,
    )
    total: Optional[List[ExplanationOfBenefitTotal]] = Field(
        description="Adjudication totals",
        default=None,
    )
    payment: Optional[ExplanationOfBenefitPayment] = Field(
        description="Payment Details",
        default=None,
    )
    formCode: Optional[CodeableConcept] = Field(
        description="Printed form identifier",
        default=None,
    )
    form: Optional[Attachment] = Field(
        description="Printed reference or actual form",
        default=None,
    )
    processNote: Optional[List[ExplanationOfBenefitProcessNote]] = Field(
        description="Note concerning adjudication",
        default=None,
    )
    benefitPeriod: Optional[Period] = Field(
        description="When the benefits are applicable",
        default=None,
    )
    benefitBalance: Optional[List[ExplanationOfBenefitBenefitBalance]] = Field(
        description="Balance by Benefit Category",
        default=None,
    )
    resourceType: Literal["ExplanationOfBenefit"] = Field(
        description=None,
        default="ExplanationOfBenefit",
    )

    @field_validator(
        *(
            "benefitBalance",
            "benefitPeriod",
            "processNote",
            "form",
            "formCode",
            "payment",
            "total",
            "adjudication",
            "addItem",
            "item",
            "patientPaid",
            "accident",
            "insurance",
            "precedence",
            "procedure",
            "diagnosis",
            "supportingInfo",
            "careTeam",
            "diagnosisRelatedGroup",
            "preAuthRefPeriod",
            "preAuthRef",
            "disposition",
            "decision",
            "outcome",
            "claimResponse",
            "claim",
            "facility",
            "encounter",
            "referral",
            "payee",
            "event",
            "originalPrescription",
            "prescription",
            "related",
            "fundsReserve",
            "fundsReserveRequested",
            "priority",
            "provider",
            "insurer",
            "enterer",
            "created",
            "billablePeriod",
            "patient",
            "use",
            "subType",
            "type",
            "status",
            "traceNumber",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
