# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Reference,
    DomainResource,
)


class ExampleScenarioActor(BackboneElement):
    """
    A system or person who shares or receives an instance within the scenario.
    """

    key: Optional[String] = Field(
        description="ID or acronym of the actor",
        default=None,
    )
    key_ext: Optional[Element] = Field(
        description="Placeholder element for key extensions",
        default=None,
        alias="_key",
    )
    type: Optional[Code] = Field(
        description="person | system",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    title: Optional[String] = Field(
        description="Label for actor when rendering",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Details about actor",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "title",
            "type",
            "key",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioInstanceVersion(BackboneElement):
    """
    Represents the instance as it was at a specific time-point.
    """

    key: Optional[String] = Field(
        description="ID or acronym of the version",
        default=None,
    )
    key_ext: Optional[Element] = Field(
        description="Placeholder element for key extensions",
        default=None,
        alias="_key",
    )
    title: Optional[String] = Field(
        description="Label for instance version",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Details about version",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    content: Optional[Reference] = Field(
        description="Example instance version data",
        default=None,
    )

    @field_validator(
        *(
            "content",
            "description",
            "title",
            "key",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioInstanceContainedInstance(BackboneElement):
    """
    References to other instances that can be found within this instance (e.g. the observations contained in a bundle).
    """

    instanceReference: Optional[String] = Field(
        description="Key of contained instance",
        default=None,
    )
    instanceReference_ext: Optional[Element] = Field(
        description="Placeholder element for instanceReference extensions",
        default=None,
        alias="_instanceReference",
    )
    versionReference: Optional[String] = Field(
        description="Key of contained instance version",
        default=None,
    )
    versionReference_ext: Optional[Element] = Field(
        description="Placeholder element for versionReference extensions",
        default=None,
        alias="_versionReference",
    )

    @field_validator(
        *(
            "versionReference",
            "instanceReference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioInstance(BackboneElement):
    """
    A single data collection that is shared as part of the scenario.
    """

    key: Optional[String] = Field(
        description="ID or acronym of the instance",
        default=None,
    )
    key_ext: Optional[Element] = Field(
        description="Placeholder element for key extensions",
        default=None,
        alias="_key",
    )
    structureType: Optional[Coding] = Field(
        description="Data structure for example",
        default=None,
    )
    structureVersion: Optional[String] = Field(
        description="E.g. 4.0.1",
        default=None,
    )
    structureVersion_ext: Optional[Element] = Field(
        description="Placeholder element for structureVersion extensions",
        default=None,
        alias="_structureVersion",
    )
    structureProfileCanonical: Optional[Canonical] = Field(
        description="Rules instance adheres to",
        default=None,
    )
    structureProfileCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for structureProfileCanonical extensions",
        default=None,
        alias="_structureProfileCanonical",
    )
    structureProfileUri: Optional[Uri] = Field(
        description="Rules instance adheres to",
        default=None,
    )
    structureProfileUri_ext: Optional[Element] = Field(
        description="Placeholder element for structureProfileUri extensions",
        default=None,
        alias="_structureProfileUri",
    )
    title: Optional[String] = Field(
        description="Label for instance",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Human-friendly description of the instance",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    content: Optional[Reference] = Field(
        description="Example instance data",
        default=None,
    )
    version: Optional[List[ExampleScenarioInstanceVersion]] = Field(
        description="Snapshot of instance that changes",
        default=None,
    )
    containedInstance: Optional[List[ExampleScenarioInstanceContainedInstance]] = Field(
        description="Resources contained in the instance",
        default=None,
    )

    @property
    def structureProfile(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="structureProfile",
        )

    @field_validator(
        *(
            "containedInstance",
            "version",
            "content",
            "description",
            "title",
            "structureVersion",
            "structureType",
            "key",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("containedInstance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_14_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="%resource.instance.where(key=%context.instanceReference).exists()",
            human="InstanceReference must be a key of an instance defined in the ExampleScenario",
            key="exs-14",
            severity="error",
        )

    @field_validator(*("containedInstance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_15_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="versionReference.empty() implies %resource.instance.where(key=%context.instanceReference).version.empty()",
            human="versionReference must be specified if the referenced instance defines versions",
            key="exs-15",
            severity="error",
        )

    @field_validator(*("containedInstance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_16_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="versionReference.exists() implies %resource.instance.where(key=%context.instanceReference).version.where(key=%context.versionReference).exists()",
            human="versionReference must be a key of a version within the instance pointed to by instanceReference",
            key="exs-16",
            severity="error",
        )

    @model_validator(mode="after")
    def structureProfile_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Canonical, Uri],
            field_name_base="structureProfile",
            required=False,
        )


class ExampleScenarioProcessStepOperationRequest(BackboneElement):
    """
    A reference to the instance that is transmitted from requester to receiver as part of the invocation of the operation.
    """

    instanceReference: Optional[String] = Field(
        description="Key of contained instance",
        default=None,
    )
    instanceReference_ext: Optional[Element] = Field(
        description="Placeholder element for instanceReference extensions",
        default=None,
        alias="_instanceReference",
    )
    versionReference: Optional[String] = Field(
        description="Key of contained instance version",
        default=None,
    )
    versionReference_ext: Optional[Element] = Field(
        description="Placeholder element for versionReference extensions",
        default=None,
        alias="_versionReference",
    )

    @field_validator(
        *(
            "versionReference",
            "instanceReference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcessStepOperation(BackboneElement):
    """
    The step represents a single operation invoked on receiver by sender.
    """

    type: Optional[Coding] = Field(
        description="Kind of action",
        default=None,
    )
    title: Optional[String] = Field(
        description="Label for step",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    initiator: Optional[String] = Field(
        description="Who starts the operation",
        default=None,
    )
    initiator_ext: Optional[Element] = Field(
        description="Placeholder element for initiator extensions",
        default=None,
        alias="_initiator",
    )
    receiver: Optional[String] = Field(
        description="Who receives the operation",
        default=None,
    )
    receiver_ext: Optional[Element] = Field(
        description="Placeholder element for receiver extensions",
        default=None,
        alias="_receiver",
    )
    description: Optional[Markdown] = Field(
        description="Human-friendly description of the operation",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    initiatorActive: Optional[Boolean] = Field(
        description="Initiator stays active?",
        default=None,
    )
    initiatorActive_ext: Optional[Element] = Field(
        description="Placeholder element for initiatorActive extensions",
        default=None,
        alias="_initiatorActive",
    )
    receiverActive: Optional[Boolean] = Field(
        description="Receiver stays active?",
        default=None,
    )
    receiverActive_ext: Optional[Element] = Field(
        description="Placeholder element for receiverActive extensions",
        default=None,
        alias="_receiverActive",
    )
    request: Optional[ExampleScenarioProcessStepOperationRequest] = Field(
        description="Instance transmitted on invocation",
        default=None,
    )
    response: Optional[ExampleScenarioInstanceContainedInstance] = Field(
        description="Instance transmitted on invocation response",
        default=None,
    )

    @field_validator(
        *(
            "response",
            "request",
            "receiverActive",
            "initiatorActive",
            "description",
            "receiver",
            "initiator",
            "title",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcessStepAlternative(BackboneElement):
    """
    Indicates an alternative step that can be taken instead of the sub-process, scenario or operation.  E.g. to represent non-happy-path/exceptional/atypical circumstances.
    """

    title: Optional[String] = Field(
        description="Label for alternative",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Human-readable description of option",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    step: Optional[List["ExampleScenarioProcessStep"]] = Field(
        description="Alternative action(s)",
        default=None,
    )

    @field_validator(
        *(
            "step",
            "description",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcessStep(BackboneElement):
    """
    A significant action that occurs as part of the process.
    """

    number: Optional[String] = Field(
        description="Sequential number of the step",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    process: Optional["ExampleScenarioProcess"] = Field(
        description="Step is nested process",
        default=None,
    )
    workflow: Optional[Canonical] = Field(
        description="Step is nested workflow",
        default=None,
    )
    workflow_ext: Optional[Element] = Field(
        description="Placeholder element for workflow extensions",
        default=None,
        alias="_workflow",
    )
    operation: Optional[ExampleScenarioProcessStepOperation] = Field(
        description="Step is simple action",
        default=None,
    )
    alternative: Optional[List[ExampleScenarioProcessStepAlternative]] = Field(
        description="Alternate non-typical step action",
        default=None,
    )
    pause: Optional[Boolean] = Field(
        description="Pause in the flow?",
        default=None,
    )
    pause_ext: Optional[Element] = Field(
        description="Placeholder element for pause extensions",
        default=None,
        alias="_pause",
    )

    @field_validator(
        *(
            "pause",
            "alternative",
            "operation",
            "workflow",
            "process",
            "number",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("operation",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_17_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="initiator.exists() implies initiator = 'OTHER' or %resource.actor.where(key=%context.initiator).exists()",
            human="If specified, initiator must be a key of an actor within the ExampleScenario",
            key="exs-17",
            severity="error",
        )

    @field_validator(*("operation",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_18_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="receiver.exists() implies receiver = 'OTHER' or %resource.actor.where(key=%context.receiver).exists()",
            human="If specified, receiver must be a key of an actor within the ExampleScenario",
            key="exs-18",
            severity="error",
        )


class ExampleScenarioProcess(BackboneElement):
    """
    A group of operations that represents a significant step within a scenario.
    """

    title: Optional[String] = Field(
        description="Label for procss",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Human-friendly description of the process",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    preConditions: Optional[Markdown] = Field(
        description="Status before process starts",
        default=None,
    )
    preConditions_ext: Optional[Element] = Field(
        description="Placeholder element for preConditions extensions",
        default=None,
        alias="_preConditions",
    )
    postConditions: Optional[Markdown] = Field(
        description="Status after successful completion",
        default=None,
    )
    postConditions_ext: Optional[Element] = Field(
        description="Placeholder element for postConditions extensions",
        default=None,
        alias="_postConditions",
    )
    step: Optional[List[ExampleScenarioProcessStep]] = Field(
        description="Event within of the process",
        default=None,
    )

    @field_validator(
        *(
            "step",
            "postConditions",
            "preConditions",
            "description",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("step",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_13_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="alternative.title.count() = alternative.title.distinct().count()",
            human="Alternative titles must be unique within a step",
            key="exs-13",
            severity="error",
        )

    @field_validator(*("step",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_22_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(process.exists() implies workflow.empty() and operation.empty()) and (workflow.exists() implies operation.empty())",
            human="Can have a process, a workflow, one or more operations or none of these, but cannot have a combination",
            key="exs-22",
            severity="error",
        )


class ExampleScenario(DomainResource):
    """
    A walkthrough of a workflow showing the interaction between systems and the instances shared, possibly including the evolution of instances over time.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ExampleScenario"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this example scenario, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the example scenario",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the example scenario",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="To be removed?",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this example scenario (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the ExampleScenario",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for example scenario (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="The purpose of the example, e.g. to illustrate a scenario",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    actor: Optional[List[ExampleScenarioActor]] = Field(
        description="Individual involved in exchange",
        default=None,
    )
    instance: Optional[List[ExampleScenarioInstance]] = Field(
        description="Data used in the scenario",
        default=None,
    )
    process: Optional[List[ExampleScenarioProcess]] = Field(
        description="Major process within scenario",
        default=None,
    )
    resourceType: Literal["ExampleScenario"] = Field(
        description=None,
        default="ExampleScenario",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "process",
            "instance",
            "actor",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @field_validator(*("actor",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_19_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="%resource.process.descendants().select(operation).where(initiator=%context.key or receiver=%context.key).exists()",
            human="Actor should be referenced in at least one operation",
            key="exs-19",
            severity="warning",
        )

    @field_validator(*("actor",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_23_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="key != 'OTHER'",
            human="actor.key canot be 'OTHER'",
            key="exs-23",
            severity="error",
        )

    @field_validator(*("instance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="structureType.exists() and structureType.memberOf('http://hl7.org/fhir/ValueSet/resource-types').not() implies structureVersion.exists()",
            human="StructureVersion is required if structureType is not FHIR (but may still be present even if FHIR)",
            key="exs-1",
            severity="error",
        )

    @field_validator(*("instance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="content.exists() implies version.empty()",
            human="instance.content is only allowed if there are no instance.versions",
            key="exs-2",
            severity="error",
        )

    @field_validator(*("instance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_10_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="version.key.count() = version.key.distinct().count()",
            human="Version keys must be unique within an instance",
            key="exs-10",
            severity="error",
        )

    @field_validator(*("instance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_11_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="version.title.count() = version.title.distinct().count()",
            human="Version titles must be unique within an instance",
            key="exs-11",
            severity="error",
        )

    @field_validator(*("instance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_20_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="%resource.process.descendants().select(instanceReference).where($this=%context.key).exists()",
            human="Instance should be referenced in at least one location",
            key="exs-20",
            severity="warning",
        )

    @field_validator(*("instance",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_21_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="version.exists() implies version.key.intersect(%resource.process.descendants().where(instanceReference = %context.key).versionReference).exists()",
            human="Instance version should be referenced in at least one operation",
            key="exs-21",
            severity="warning",
        )

    @field_validator(*("process",), mode="after", check_fields=None)
    @classmethod
    def FHIR_exs_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="%resource.status='active' or %resource.status='retired' implies step.exists()",
            human="Processes must have steps if ExampleScenario status is active or required",
            key="exs-5",
            severity="error",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_exs_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="status='active' or status='retired' implies actor.exists()",
            human="Must have actors if status is active or required",
            key="exs-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_exs_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="status='active' or status='retired' implies process.exists()",
            human="Must have processes if status is active or required",
            key="exs-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_exs_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="actor.key.count() = actor.key.distinct().count()",
            human="Actor keys must be unique",
            key="exs-6",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_exs_7_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="actor.title.count() = actor.title.distinct().count()",
            human="Actor titles must be unique",
            key="exs-7",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_exs_8_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="instance.key.count() = instance.key.distinct().count()",
            human="Instance keys must be unique",
            key="exs-8",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_exs_9_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="instance.title.count() = instance.title.distinct().count()",
            human="Instance titles must be unique",
            key="exs-9",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_exs_12_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="process.title.count() = process.title.distinct().count()",
            human="Process titles must be unique",
            key="exs-12",
            severity="error",
        )


ExampleScenarioProcessStepAlternative.model_rebuild()
ExampleScenarioProcessStep.model_rebuild()
ExampleScenarioProcess.model_rebuild()
ExampleScenario.model_rebuild()
