# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Markdown,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    UsageContext,
    Identifier,
    Reference,
    CodeableConcept,
    Annotation,
    RelatedArtifact,
    BackboneElement,
    Quantity,
    Range,
    Period,
    ContactDetail,
    DomainResource,
)


class EvidenceReportSubjectCharacteristic(BackboneElement):
    """
    Characteristic.
    """

    code: Optional[CodeableConcept] = Field(
        description="Characteristic code",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="Characteristic value",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Characteristic value",
        default=None,
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Characteristic value",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Characteristic value",
        default=None,
    )
    valueRange: Optional[Range] = Field(
        description="Characteristic value",
        default=None,
    )
    exclude: Optional[Boolean] = Field(
        description="Is used to express not the characteristic",
        default=None,
    )
    exclude_ext: Optional[Element] = Field(
        description="Placeholder element for exclude extensions",
        default=None,
        alias="_exclude",
    )
    period: Optional[Period] = Field(
        description="Timeframe for the characteristic",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "period",
            "exclude",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, CodeableConcept, Boolean, Quantity, Range],
            field_name_base="value",
            required=True,
        )


class EvidenceReportSubject(BackboneElement):
    """
    Specifies the subject or focus of the report. Answers "What is this report about?".
    """

    characteristic: Optional[List[EvidenceReportSubjectCharacteristic]] = Field(
        description="Characteristic",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnotes and/or explanatory notes",
        default=None,
    )

    @field_validator(
        *(
            "note",
            "characteristic",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceReportRelatesToTarget(BackboneElement):
    """
    The target composition/document of this relationship.
    """

    url: Optional[Uri] = Field(
        description="Target of the relationship URL",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[Identifier] = Field(
        description="Target of the relationship Identifier",
        default=None,
    )
    display: Optional[Markdown] = Field(
        description="Target of the relationship Display",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    resource: Optional[Reference] = Field(
        description="Target of the relationship Resource reference",
        default=None,
    )

    @field_validator(
        *(
            "resource",
            "display",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceReportRelatesTo(BackboneElement):
    """
    Relationships that this composition has with other compositions or documents that already exist.
    """

    code: Optional[Code] = Field(
        description="replaces | amends | appends | transforms | replacedWith | amendedWith | appendedWith | transformedWith",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    target: Optional[EvidenceReportRelatesToTarget] = Field(
        description="Target of the relationship",
        default=None,
    )

    @field_validator(
        *(
            "target",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceReportSection(BackboneElement):
    """
    The root of the sections that make up the composition.
    """

    title: Optional[String] = Field(
        description="Label for section (e.g. for ToC)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    focus: Optional[CodeableConcept] = Field(
        description="Classification of section (recommended)",
        default=None,
    )
    focusReference: Optional[Reference] = Field(
        description="Classification of section by Resource",
        default=None,
    )
    author: Optional[List[Reference]] = Field(
        description="Who and/or what authored the section",
        default=None,
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the section, for human interpretation",
        default=None,
    )
    mode: Optional[Code] = Field(
        description="working | snapshot | changes",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    orderedBy: Optional[CodeableConcept] = Field(
        description="Order of section entries",
        default=None,
    )
    entryClassifier: Optional[List[CodeableConcept]] = Field(
        description="Extensible classifiers as content",
        default=None,
    )
    entryReference: Optional[List[Reference]] = Field(
        description="Reference to resources as content",
        default=None,
    )
    entryQuantity: Optional[List[Quantity]] = Field(
        description="Quantity as content",
        default=None,
    )
    emptyReason: Optional[CodeableConcept] = Field(
        description="Why the section is empty",
        default=None,
    )
    section: Optional[List["EvidenceReportSection"]] = Field(
        description="Nested Section",
        default=None,
    )

    @field_validator(
        *(
            "section",
            "emptyReason",
            "entryQuantity",
            "entryReference",
            "entryClassifier",
            "orderedBy",
            "mode",
            "text",
            "author",
            "focusReference",
            "focus",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceReport(DomainResource):
    """
    The EvidenceReport Resource is a specialized container for a collection of resources and codeable concepts, adapted to support compositions of Evidence, EvidenceVariable, and Citation resources and related concepts.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/EvidenceReport"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this EvidenceReport, represented as a globally unique URI",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Unique identifier for the evidence report",
        default=None,
    )
    relatedIdentifier: Optional[List[Identifier]] = Field(
        description="Identifiers for articles that may relate to more than one evidence report",
        default=None,
    )
    citeAsReference: Optional[Reference] = Field(
        description="Citation for this report",
        default=None,
    )
    citeAsMarkdown: Optional[Markdown] = Field(
        description="Citation for this report",
        default=None,
    )
    citeAsMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for citeAsMarkdown extensions",
        default=None,
        alias="_citeAsMarkdown",
    )
    type: Optional[CodeableConcept] = Field(
        description="Kind of report",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Used for footnotes and annotations",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Link, description or reference to artifact associated with the report",
        default=None,
    )
    subject: Optional[EvidenceReportSubject] = Field(
        description="Focus of the report",
        default=None,
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the content",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the content",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the content",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the content",
        default=None,
    )
    relatesTo: Optional[List[EvidenceReportRelatesTo]] = Field(
        description="Relationships to other compositions/documents",
        default=None,
    )
    section: Optional[List[EvidenceReportSection]] = Field(
        description="Composition is broken into sections",
        default=None,
    )
    resourceType: Literal["EvidenceReport"] = Field(
        description=None,
        default="EvidenceReport",
    )

    @property
    def citeAs(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="citeAs",
        )

    @field_validator(
        *(
            "section",
            "relatesTo",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "contact",
            "publisher",
            "subject",
            "relatedArtifact",
            "note",
            "type",
            "relatedIdentifier",
            "identifier",
            "useContext",
            "status",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def citeAs_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, Markdown],
            field_name_base="citeAs",
            required=False,
        )
