# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Markdown,
    Boolean,
    DateTime,
    Date,
    UnsignedInt,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    Reference,
    ContactDetail,
    UsageContext,
    RelatedArtifact,
    Annotation,
    BackboneElement,
    CodeableConcept,
    Quantity,
    Range,
    DomainResource,
)


class EvidenceVariableDefinition(BackboneElement):
    """
    Evidence variable such as population, exposure, or outcome.
    """

    description: Optional[Markdown] = Field(
        description="A text description or summary of the variable",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnotes and/or explanatory notes",
        default=None,
    )
    variableRole: Optional[CodeableConcept] = Field(
        description="population | subpopulation | exposure | referenceExposure | measuredVariable | confounder",
        default=None,
    )
    observed: Optional[Reference] = Field(
        description="Definition of the actual variable related to the statistic(s)",
        default=None,
    )
    intended: Optional[Reference] = Field(
        description="Definition of the intended variable related to the Evidence",
        default=None,
    )
    directnessMatch: Optional[CodeableConcept] = Field(
        description="low | moderate | high | exact",
        default=None,
    )

    @field_validator(
        *(
            "directnessMatch",
            "intended",
            "observed",
            "variableRole",
            "note",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceStatisticSampleSize(BackboneElement):
    """
    Number of samples in the statistic.
    """

    description: Optional[Markdown] = Field(
        description="Textual description of sample size for statistic",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnote or explanatory note about the sample size",
        default=None,
    )
    numberOfStudies: Optional[UnsignedInt] = Field(
        description="Number of contributing studies",
        default=None,
    )
    numberOfStudies_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfStudies extensions",
        default=None,
        alias="_numberOfStudies",
    )
    numberOfParticipants: Optional[UnsignedInt] = Field(
        description="Cumulative number of participants",
        default=None,
    )
    numberOfParticipants_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfParticipants extensions",
        default=None,
        alias="_numberOfParticipants",
    )
    knownDataCount: Optional[UnsignedInt] = Field(
        description="Number of participants with known results for measured variables",
        default=None,
    )
    knownDataCount_ext: Optional[Element] = Field(
        description="Placeholder element for knownDataCount extensions",
        default=None,
        alias="_knownDataCount",
    )

    @field_validator(
        *(
            "knownDataCount",
            "numberOfParticipants",
            "numberOfStudies",
            "note",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceStatisticAttributeEstimate(BackboneElement):
    """
    A statistical attribute of the statistic such as a measure of heterogeneity.
    """

    description: Optional[Markdown] = Field(
        description="Textual description of the attribute estimate",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnote or explanatory note about the estimate",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The type of attribute estimate, e.g., confidence interval or p value",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The singular quantity of the attribute estimate, for attribute estimates represented as single values; also used to report unit of measure",
        default=None,
    )
    level: Optional[Decimal] = Field(
        description="Level of confidence interval, e.g., 0.95 for 95% confidence interval",
        default=None,
    )
    level_ext: Optional[Element] = Field(
        description="Placeholder element for level extensions",
        default=None,
        alias="_level",
    )
    range: Optional[Range] = Field(
        description="Lower and upper bound values of the attribute estimate",
        default=None,
    )
    attributeEstimate: Optional[List["EvidenceStatisticAttributeEstimate"]] = Field(
        description="A nested attribute estimate; which is the attribute estimate of an attribute estimate",
        default=None,
    )

    @field_validator(
        *(
            "attributeEstimate",
            "range",
            "level",
            "quantity",
            "type",
            "note",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceStatisticModelCharacteristicVariable(BackboneElement):
    """
    A variable adjusted for in the adjusted analysis.
    """

    variableDefinition: Optional[Reference] = Field(
        description="Description of the variable",
        default=None,
    )
    handling: Optional[Code] = Field(
        description="continuous | dichotomous | ordinal | polychotomous",
        default=None,
    )
    handling_ext: Optional[Element] = Field(
        description="Placeholder element for handling extensions",
        default=None,
        alias="_handling",
    )
    valueCategory: Optional[List[CodeableConcept]] = Field(
        description="Description for grouping of ordinal or polychotomous variables",
        default=None,
    )
    valueQuantity: Optional[List[Quantity]] = Field(
        description="Discrete value for grouping of ordinal or polychotomous variables",
        default=None,
    )
    valueRange: Optional[List[Range]] = Field(
        description="Range of values for grouping of ordinal or polychotomous variables",
        default=None,
    )

    @field_validator(
        *(
            "valueRange",
            "valueQuantity",
            "valueCategory",
            "handling",
            "variableDefinition",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceStatisticModelCharacteristicAttributeEstimate(BackboneElement):
    """
    An attribute of the statistic used as a model characteristic.
    """

    description: Optional[Markdown] = Field(
        description="Textual description of the attribute estimate",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnote or explanatory note about the estimate",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The type of attribute estimate, e.g., confidence interval or p value",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The singular quantity of the attribute estimate, for attribute estimates represented as single values; also used to report unit of measure",
        default=None,
    )
    level: Optional[Decimal] = Field(
        description="Level of confidence interval, e.g., 0.95 for 95% confidence interval",
        default=None,
    )
    level_ext: Optional[Element] = Field(
        description="Placeholder element for level extensions",
        default=None,
        alias="_level",
    )
    range: Optional[Range] = Field(
        description="Lower and upper bound values of the attribute estimate",
        default=None,
    )
    attributeEstimate: Optional[List[EvidenceStatisticAttributeEstimate]] = Field(
        description="A nested attribute estimate; which is the attribute estimate of an attribute estimate",
        default=None,
    )

    @field_validator(
        *(
            "attributeEstimate",
            "range",
            "level",
            "quantity",
            "type",
            "note",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceStatisticModelCharacteristic(BackboneElement):
    """
    A component of the method to generate the statistic.
    """

    code: Optional[CodeableConcept] = Field(
        description="Model specification",
        default=None,
    )
    value: Optional[Quantity] = Field(
        description="Numerical value to complete model specification",
        default=None,
    )
    variable: Optional[List[EvidenceStatisticModelCharacteristicVariable]] = Field(
        description="A variable adjusted for in the adjusted analysis",
        default=None,
    )
    attributeEstimate: Optional[
        List[EvidenceStatisticModelCharacteristicAttributeEstimate]
    ] = Field(
        description="An attribute of the statistic used as a model characteristic",
        default=None,
    )

    @field_validator(
        *(
            "attributeEstimate",
            "variable",
            "value",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceStatistic(BackboneElement):
    """
    Values and parameters for a single statistic.
    """

    description: Optional[Markdown] = Field(
        description="Description of content",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnotes and/or explanatory notes",
        default=None,
    )
    statisticType: Optional[CodeableConcept] = Field(
        description="Type of statistic, e.g., relative risk",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Associated category for categorical variable",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Statistic value",
        default=None,
    )
    numberOfEvents: Optional[UnsignedInt] = Field(
        description="The number of events associated with the statistic",
        default=None,
    )
    numberOfEvents_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfEvents extensions",
        default=None,
        alias="_numberOfEvents",
    )
    numberAffected: Optional[UnsignedInt] = Field(
        description="The number of participants affected",
        default=None,
    )
    numberAffected_ext: Optional[Element] = Field(
        description="Placeholder element for numberAffected extensions",
        default=None,
        alias="_numberAffected",
    )
    sampleSize: Optional[EvidenceStatisticSampleSize] = Field(
        description="Number of samples in the statistic",
        default=None,
    )
    attributeEstimate: Optional[List[EvidenceStatisticAttributeEstimate]] = Field(
        description="An attribute of the Statistic",
        default=None,
    )
    modelCharacteristic: Optional[List[EvidenceStatisticModelCharacteristic]] = Field(
        description="An aspect of the statistical model",
        default=None,
    )

    @field_validator(
        *(
            "modelCharacteristic",
            "attributeEstimate",
            "sampleSize",
            "numberAffected",
            "numberOfEvents",
            "quantity",
            "category",
            "statisticType",
            "note",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EvidenceCertainty(BackboneElement):
    """
    Assessment of certainty, confidence in the estimates, or quality of the evidence.
    """

    description: Optional[Markdown] = Field(
        description="Textual description of certainty",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnotes and/or explanatory notes",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Aspect of certainty being rated",
        default=None,
    )
    rating: Optional[CodeableConcept] = Field(
        description="Assessment or judgement of the aspect",
        default=None,
    )
    rater: Optional[String] = Field(
        description="Individual or group who did the rating",
        default=None,
    )
    rater_ext: Optional[Element] = Field(
        description="Placeholder element for rater extensions",
        default=None,
        alias="_rater",
    )
    subcomponent: Optional[List["EvidenceCertainty"]] = Field(
        description="A domain or subdomain of certainty",
        default=None,
    )

    @field_validator(
        *(
            "subcomponent",
            "rater",
            "rating",
            "type",
            "note",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Evidence(DomainResource):
    """
    The Evidence Resource provides a machine-interpretable expression of an evidence concept including the evidence variables (e.g., population, exposures/interventions, comparators, outcomes, measured variables, confounding variables), the statistics, and the certainty of this evidence.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Evidence"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this evidence, represented as a globally unique URI",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the summary",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of this summary",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this summary (machine friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this summary (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    citeAsReference: Optional[Reference] = Field(
        description="Citation for this evidence",
        default=None,
    )
    citeAsMarkdown: Optional[Markdown] = Field(
        description="Citation for this evidence",
        default=None,
    )
    citeAsMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for citeAsMarkdown extensions",
        default=None,
        alias="_citeAsMarkdown",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    approvalDate: Optional[Date] = Field(
        description="When the summary was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the summary was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the content",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the content",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the content",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the content",
        default=None,
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this Evidence is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Link or citation to artifact associated with the summary",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Description of the particular summary",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    assertion: Optional[Markdown] = Field(
        description="Declarative description of the Evidence",
        default=None,
    )
    assertion_ext: Optional[Element] = Field(
        description="Placeholder element for assertion extensions",
        default=None,
        alias="_assertion",
    )
    note: Optional[List[Annotation]] = Field(
        description="Footnotes and/or explanatory notes",
        default=None,
    )
    variableDefinition: Optional[List[EvidenceVariableDefinition]] = Field(
        description="Evidence variable such as population, exposure, or outcome",
        default=None,
    )
    synthesisType: Optional[CodeableConcept] = Field(
        description="The method to combine studies",
        default=None,
    )
    studyDesign: Optional[List[CodeableConcept]] = Field(
        description="The design of the study that produced this evidence",
        default=None,
    )
    statistic: Optional[List[EvidenceStatistic]] = Field(
        description="Values and parameters for a single statistic",
        default=None,
    )
    certainty: Optional[List[EvidenceCertainty]] = Field(
        description="Certainty or quality of the evidence",
        default=None,
    )
    resourceType: Literal["Evidence"] = Field(
        description=None,
        default="Evidence",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @property
    def citeAs(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="citeAs",
        )

    @field_validator(
        *(
            "certainty",
            "statistic",
            "studyDesign",
            "synthesisType",
            "variableDefinition",
            "note",
            "assertion",
            "description",
            "relatedArtifact",
            "copyrightLabel",
            "copyright",
            "purpose",
            "useContext",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "contact",
            "publisher",
            "lastReviewDate",
            "approvalDate",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def citeAs_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, Markdown],
            field_name_base="citeAs",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )
