# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Reference,
    Identifier,
    CodeableConcept,
    CodeableReference,
    Period,
    Duration,
    BackboneElement,
    DomainResource,
)


class EncounterHistoryLocation(BackboneElement):
    """
    The location of the patient at this point in the encounter, the multiple cardinality permits de-normalizing the levels of the location hierarchy, such as site/ward/room/bed.
    """

    location: Optional[Reference] = Field(
        description="Location the encounter takes place",
        default=None,
    )
    form: Optional[CodeableConcept] = Field(
        description="The physical type of the location (usually the level in the location hierarchy - bed, room, ward, virtual etc.)",
        default=None,
    )

    @field_validator(
        *(
            "form",
            "location",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterHistory(DomainResource):
    """
    A record of significant events/milestones key data throughout the history of an Encounter
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/EncounterHistory"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="The Encounter associated with this set of historic values",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier(s) by which this encounter is known",
        default=None,
    )
    status: Optional[Code] = Field(
        description="planned | in-progress | on-hold | discharged | completed | cancelled | discontinued | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    class_: Optional[CodeableConcept] = Field(
        description="Classification of patient encounter",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Specific type of encounter",
        default=None,
    )
    serviceType: Optional[List[CodeableReference]] = Field(
        description="Specific type of service",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="The patient or group related to this encounter",
        default=None,
    )
    subjectStatus: Optional[CodeableConcept] = Field(
        description="The current status of the subject in relation to the Encounter",
        default=None,
    )
    actualPeriod: Optional[Period] = Field(
        description="The actual start and end time associated with this set of values associated with the encounter",
        default=None,
    )
    plannedStartDate: Optional[DateTime] = Field(
        description="The planned start date/time (or admission date) of the encounter",
        default=None,
    )
    plannedStartDate_ext: Optional[Element] = Field(
        description="Placeholder element for plannedStartDate extensions",
        default=None,
        alias="_plannedStartDate",
    )
    plannedEndDate: Optional[DateTime] = Field(
        description="The planned end date/time (or discharge date) of the encounter",
        default=None,
    )
    plannedEndDate_ext: Optional[Element] = Field(
        description="Placeholder element for plannedEndDate extensions",
        default=None,
        alias="_plannedEndDate",
    )
    length: Optional[Duration] = Field(
        description="Actual quantity of time the encounter lasted (less time absent)",
        default=None,
    )
    location: Optional[List[EncounterHistoryLocation]] = Field(
        description="Location of the patient at this point in the encounter",
        default=None,
    )
    resourceType: Literal["EncounterHistory"] = Field(
        description=None,
        default="EncounterHistory",
    )

    @field_validator(
        *(
            "location",
            "length",
            "plannedEndDate",
            "plannedStartDate",
            "actualPeriod",
            "subjectStatus",
            "subject",
            "serviceType",
            "type",
            "class_",
            "status",
            "identifier",
            "encounter",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
