# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    CodeableReference,
    Reference,
    BackboneElement,
    Period,
    VirtualServiceDetail,
    Duration,
    DomainResource,
)


class EncounterParticipant(BackboneElement):
    """
    The list of people responsible for providing the service.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="Role of participant in encounter",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Period of time during the encounter that the participant participated",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="The individual, device, or service participating in the encounter",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "period",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterReason(BackboneElement):
    """
    The list of medical reasons that are expected to be addressed during the episode of care.
    """

    use: Optional[List[CodeableConcept]] = Field(
        description="What the reason value should be used for/as",
        default=None,
    )
    value: Optional[List[CodeableReference]] = Field(
        description="Reason the encounter takes place (core or reference)",
        default=None,
    )

    @field_validator(
        *(
            "value",
            "use",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterDiagnosis(BackboneElement):
    """
    The list of diagnosis relevant to this encounter.
    """

    condition: Optional[List[CodeableReference]] = Field(
        description="The diagnosis relevant to the encounter",
        default=None,
    )
    use: Optional[List[CodeableConcept]] = Field(
        description="Role that this diagnosis has within the encounter (e.g. admission, billing, discharge \u2026)",
        default=None,
    )

    @field_validator(
        *(
            "use",
            "condition",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterAdmission(BackboneElement):
    """
        Details about the stay during which a healthcare service is provided.

    This does not describe the event of admitting the patient, but rather any information that is relevant from the time of admittance until the time of discharge.
    """

    preAdmissionIdentifier: Optional[Identifier] = Field(
        description="Pre-admission identifier",
        default=None,
    )
    origin: Optional[Reference] = Field(
        description="The location/organization from which the patient came before admission",
        default=None,
    )
    admitSource: Optional[CodeableConcept] = Field(
        description="From where patient was admitted (physician referral, transfer)",
        default=None,
    )
    reAdmission: Optional[CodeableConcept] = Field(
        description="Indicates that the patient is being re-admitted",
        default=None,
    )
    destination: Optional[Reference] = Field(
        description="Location/organization to which the patient is discharged",
        default=None,
    )
    dischargeDisposition: Optional[CodeableConcept] = Field(
        description="Category or kind of location after discharge",
        default=None,
    )

    @field_validator(
        *(
            "dischargeDisposition",
            "destination",
            "reAdmission",
            "admitSource",
            "origin",
            "preAdmissionIdentifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterLocation(BackboneElement):
    """
    List of locations where  the patient has been during this encounter.
    """

    location: Optional[Reference] = Field(
        description="Location the encounter takes place",
        default=None,
    )
    status: Optional[Code] = Field(
        description="planned | active | reserved | completed",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    form: Optional[CodeableConcept] = Field(
        description="The physical type of the location (usually the level in the location hierarchy - bed, room, ward, virtual etc.)",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Time period during which the patient was present at the location",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "form",
            "status",
            "location",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Encounter(DomainResource):
    """
    An interaction between healthcare provider(s), and/or patient(s) for the purpose of providing healthcare service(s) or assessing the health status of patient(s).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Encounter"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier(s) by which this encounter is known",
        default=None,
    )
    status: Optional[Code] = Field(
        description="planned | in-progress | on-hold | discharged | completed | cancelled | discontinued | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    class_: Optional[List[CodeableConcept]] = Field(
        description="Classification of patient encounter context - e.g. Inpatient, outpatient",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="Indicates the urgency of the encounter",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Specific type of encounter (e.g. e-mail consultation, surgical day-care, ...)",
        default=None,
    )
    serviceType: Optional[List[CodeableReference]] = Field(
        description="Specific type of service",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="The patient or group related to this encounter",
        default=None,
    )
    subjectStatus: Optional[CodeableConcept] = Field(
        description="The current status of the subject in relation to the Encounter",
        default=None,
    )
    episodeOfCare: Optional[List[Reference]] = Field(
        description="Episode(s) of care that this encounter should be recorded against",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="The request that initiated this encounter",
        default=None,
    )
    careTeam: Optional[List[Reference]] = Field(
        description="The group(s) that are allocated to participate in this encounter",
        default=None,
    )
    partOf: Optional[Reference] = Field(
        description="Another Encounter this encounter is part of",
        default=None,
    )
    serviceProvider: Optional[Reference] = Field(
        description="The organization (facility) responsible for this encounter",
        default=None,
    )
    participant: Optional[List[EncounterParticipant]] = Field(
        description="List of participants involved in the encounter",
        default=None,
    )
    appointment: Optional[List[Reference]] = Field(
        description="The appointment that scheduled this encounter",
        default=None,
    )
    virtualService: Optional[List[VirtualServiceDetail]] = Field(
        description="Connection details of a virtual service (e.g. conference call)",
        default=None,
    )
    actualPeriod: Optional[Period] = Field(
        description="The actual start and end time of the encounter",
        default=None,
    )
    plannedStartDate: Optional[DateTime] = Field(
        description="The planned start date/time (or admission date) of the encounter",
        default=None,
    )
    plannedStartDate_ext: Optional[Element] = Field(
        description="Placeholder element for plannedStartDate extensions",
        default=None,
        alias="_plannedStartDate",
    )
    plannedEndDate: Optional[DateTime] = Field(
        description="The planned end date/time (or discharge date) of the encounter",
        default=None,
    )
    plannedEndDate_ext: Optional[Element] = Field(
        description="Placeholder element for plannedEndDate extensions",
        default=None,
        alias="_plannedEndDate",
    )
    length: Optional[Duration] = Field(
        description="Actual quantity of time the encounter lasted (less time absent)",
        default=None,
    )
    reason: Optional[List[EncounterReason]] = Field(
        description="The list of medical reasons that are expected to be addressed during the episode of care",
        default=None,
    )
    diagnosis: Optional[List[EncounterDiagnosis]] = Field(
        description="The list of diagnosis relevant to this encounter",
        default=None,
    )
    account: Optional[List[Reference]] = Field(
        description="The set of accounts that may be used for billing for this Encounter",
        default=None,
    )
    dietPreference: Optional[List[CodeableConcept]] = Field(
        description="Diet preferences reported by the patient",
        default=None,
    )
    specialArrangement: Optional[List[CodeableConcept]] = Field(
        description="Wheelchair, translator, stretcher, etc",
        default=None,
    )
    specialCourtesy: Optional[List[CodeableConcept]] = Field(
        description="Special courtesies (VIP, board member)",
        default=None,
    )
    admission: Optional[EncounterAdmission] = Field(
        description="Details about the admission to a healthcare service",
        default=None,
    )
    location: Optional[List[EncounterLocation]] = Field(
        description="List of locations where the patient has been",
        default=None,
    )
    resourceType: Literal["Encounter"] = Field(
        description=None,
        default="Encounter",
    )

    @field_validator(
        *(
            "location",
            "admission",
            "specialCourtesy",
            "specialArrangement",
            "dietPreference",
            "account",
            "diagnosis",
            "reason",
            "length",
            "plannedEndDate",
            "plannedStartDate",
            "actualPeriod",
            "virtualService",
            "appointment",
            "participant",
            "serviceProvider",
            "partOf",
            "careTeam",
            "basedOn",
            "episodeOfCare",
            "subjectStatus",
            "subject",
            "serviceType",
            "type",
            "priority",
            "class_",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("participant",), mode="after", check_fields=None)
    @classmethod
    def FHIR_enc_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="actor.exists() or type.exists()",
            human="A type must be provided when no explicit actor is specified",
            key="enc-1",
            severity="error",
        )

    @field_validator(*("participant",), mode="after", check_fields=None)
    @classmethod
    def FHIR_enc_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="actor.exists(resolve() is Patient or resolve() is Group) implies type.exists().not()",
            human="A type cannot be provided for a patient or group participant",
            key="enc-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
