# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Instant,
    DateTime,
    Markdown,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    CodeableReference,
    Period,
    BackboneElement,
    Attachment,
    Coding,
    DomainResource,
)


class DocumentReferenceAttester(BackboneElement):
    """
    A participant who has authenticated the accuracy of the document.
    """

    mode: Optional[CodeableConcept] = Field(
        description="personal | professional | legal | official",
        default=None,
    )
    time: Optional[DateTime] = Field(
        description="When the document was attested",
        default=None,
    )
    time_ext: Optional[Element] = Field(
        description="Placeholder element for time extensions",
        default=None,
        alias="_time",
    )
    party: Optional[Reference] = Field(
        description="Who attested the document",
        default=None,
    )

    @field_validator(
        *(
            "party",
            "time",
            "mode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DocumentReferenceRelatesTo(BackboneElement):
    """
    Relationships that this document has with other document references that already exist.
    """

    code: Optional[CodeableConcept] = Field(
        description="The relationship type with another document",
        default=None,
    )
    target: Optional[Reference] = Field(
        description="Target of the relationship",
        default=None,
    )

    @field_validator(
        *(
            "target",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DocumentReferenceContentProfile(BackboneElement):
    """
    An identifier of the document constraints, encoding, structure, and template that the document conforms to beyond the base format indicated in the mimeType.
    """

    valueCoding: Optional[Coding] = Field(
        description="Code|uri|canonical",
        default=None,
    )
    valueUri: Optional[Uri] = Field(
        description="Code|uri|canonical",
        default=None,
    )
    valueUri_ext: Optional[Element] = Field(
        description="Placeholder element for valueUri extensions",
        default=None,
        alias="_valueUri",
    )
    valueCanonical: Optional[Canonical] = Field(
        description="Code|uri|canonical",
        default=None,
    )
    valueCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for valueCanonical extensions",
        default=None,
        alias="_valueCanonical",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Coding, Uri, Canonical],
            field_name_base="value",
            required=True,
        )


class DocumentReferenceContent(BackboneElement):
    """
    The document and format referenced.  If there are multiple content element repetitions, these must all represent the same document in different format, or attachment metadata.
    """

    attachment: Optional[Attachment] = Field(
        description="Where to access the document",
        default=None,
    )
    profile: Optional[List[DocumentReferenceContentProfile]] = Field(
        description="Content profile rules for the document",
        default=None,
    )

    @field_validator(
        *(
            "profile",
            "attachment",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DocumentReference(DomainResource):
    """
    A reference to a document of any kind for any purpose. While the term “document” implies a more narrow focus, for this resource this “document” encompasses *any* serialized object with a mime-type, it includes formal patient-centric documents (CDA), clinical notes, scanned paper, non-patient specific documents like policy text, as well as a photo, video, or audio recording acquired or used in healthcare.  The DocumentReference resource provides metadata about the document so that the document can be discovered and managed.  The actual content may be inline base64 encoded data or provided by direct reference.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/DocumentReference"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifiers for the document",
        default=None,
    )
    version: Optional[String] = Field(
        description="An explicitly assigned identifer of a variation of the content in the DocumentReference",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Procedure that caused this media to be created",
        default=None,
    )
    status: Optional[Code] = Field(
        description="current | superseded | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    docStatus: Optional[Code] = Field(
        description="registered | partial | preliminary | final | amended | corrected | appended | cancelled | entered-in-error | deprecated | unknown",
        default=None,
    )
    docStatus_ext: Optional[Element] = Field(
        description="Placeholder element for docStatus extensions",
        default=None,
        alias="_docStatus",
    )
    modality: Optional[List[CodeableConcept]] = Field(
        description="Imaging modality used",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Kind of document (LOINC if possible)",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Categorization of document",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who/what is the subject of the document",
        default=None,
    )
    context: Optional[List[Reference]] = Field(
        description="Context of the document content",
        default=None,
    )
    event: Optional[List[CodeableReference]] = Field(
        description="Main clinical acts documented",
        default=None,
    )
    bodySite: Optional[List[CodeableReference]] = Field(
        description="Body part included",
        default=None,
    )
    facilityType: Optional[CodeableConcept] = Field(
        description="Kind of facility where patient was seen",
        default=None,
    )
    practiceSetting: Optional[CodeableConcept] = Field(
        description="Additional details about where the content was created (e.g. clinical specialty)",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Time of service that is being documented",
        default=None,
    )
    date: Optional[Instant] = Field(
        description="When this document reference was created",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    author: Optional[List[Reference]] = Field(
        description="Who and/or what authored the document",
        default=None,
    )
    attester: Optional[List[DocumentReferenceAttester]] = Field(
        description="Attests to accuracy of the document",
        default=None,
    )
    custodian: Optional[Reference] = Field(
        description="Organization which maintains the document",
        default=None,
    )
    relatesTo: Optional[List[DocumentReferenceRelatesTo]] = Field(
        description="Relationships to other documents",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Human-readable description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    securityLabel: Optional[List[CodeableConcept]] = Field(
        description="Document security-tags",
        default=None,
    )
    content: Optional[List[DocumentReferenceContent]] = Field(
        description="Document referenced",
        default=None,
    )
    resourceType: Literal["DocumentReference"] = Field(
        description=None,
        default="DocumentReference",
    )

    @field_validator(
        *(
            "content",
            "securityLabel",
            "description",
            "relatesTo",
            "custodian",
            "attester",
            "author",
            "date",
            "period",
            "practiceSetting",
            "facilityType",
            "bodySite",
            "event",
            "context",
            "subject",
            "category",
            "type",
            "modality",
            "docStatus",
            "status",
            "basedOn",
            "version",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_docRef_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="facilityType.empty() or context.where(resolve() is Encounter).empty()",
            human="facilityType SHALL only be present if context is not an encounter",
            key="docRef-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_docRef_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="practiceSetting.empty() or context.where(resolve() is Encounter).empty()",
            human="practiceSetting SHALL only be present if context is not present",
            key="docRef-2",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
