# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableReference,
    CodeableConcept,
    BackboneElement,
    Quantity,
    Annotation,
    DomainResource,
)


class DeviceDispensePerformer(BackboneElement):
    """
    Indicates who or what performed the event.
    """

    function: Optional[CodeableConcept] = Field(
        description="Who performed the dispense and what they did",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Individual who was performing",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDispense(DomainResource):
    """
    Indicates that a device is to be or has been dispensed for a named person/patient.  This includes a description of the product (supply) provided and the instructions for using the device.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/DeviceDispense"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for this dispensation",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="The order or request that this dispense is fulfilling",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="The bigger event that this dispense is a part of",
        default=None,
    )
    status: Optional[Code] = Field(
        description="preparation | in-progress | cancelled | on-hold | completed | entered-in-error | stopped | declined | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableReference] = Field(
        description="Why a dispense was or was not performed",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Type of device dispense",
        default=None,
    )
    device: Optional[CodeableReference] = Field(
        description="What device was supplied",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who the dispense is for",
        default=None,
    )
    receiver: Optional[Reference] = Field(
        description="Who collected the device or where the medication was delivered",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter associated with event",
        default=None,
    )
    supportingInformation: Optional[List[Reference]] = Field(
        description="Information that supports the dispensing of the device",
        default=None,
    )
    performer: Optional[List[DeviceDispensePerformer]] = Field(
        description="Who performed event",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where the dispense occurred",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Trial fill, partial fill, emergency fill, etc",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Amount dispensed",
        default=None,
    )
    preparedDate: Optional[DateTime] = Field(
        description="When product was packaged and reviewed",
        default=None,
    )
    preparedDate_ext: Optional[Element] = Field(
        description="Placeholder element for preparedDate extensions",
        default=None,
        alias="_preparedDate",
    )
    whenHandedOver: Optional[DateTime] = Field(
        description="When product was given out",
        default=None,
    )
    whenHandedOver_ext: Optional[Element] = Field(
        description="Placeholder element for whenHandedOver extensions",
        default=None,
        alias="_whenHandedOver",
    )
    destination: Optional[Reference] = Field(
        description="Where the device was sent or should be sent",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Information about the dispense",
        default=None,
    )
    usageInstruction: Optional[Markdown] = Field(
        description="Full representation of the usage instructions",
        default=None,
    )
    usageInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for usageInstruction extensions",
        default=None,
        alias="_usageInstruction",
    )
    eventHistory: Optional[List[Reference]] = Field(
        description="A list of relevant lifecycle events",
        default=None,
    )
    resourceType: Literal["DeviceDispense"] = Field(
        description=None,
        default="DeviceDispense",
    )

    @field_validator(
        *(
            "eventHistory",
            "usageInstruction",
            "note",
            "destination",
            "whenHandedOver",
            "preparedDate",
            "quantity",
            "type",
            "location",
            "performer",
            "supportingInformation",
            "encounter",
            "receiver",
            "subject",
            "device",
            "category",
            "statusReason",
            "status",
            "partOf",
            "basedOn",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
