# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Markdown,
    Integer,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    BackboneElement,
    Period,
    Reference,
    CodeableConcept,
    RelatedArtifact,
    ProductShelfLife,
    Quantity,
    Range,
    Attachment,
    ContactPoint,
    Coding,
    CodeableReference,
    Annotation,
    UsageContext,
    DomainResource,
)


class DeviceDefinitionUdiDeviceIdentifierMarketDistribution(BackboneElement):
    """
    Indicates where and when the device is available on the market.
    """

    marketPeriod: Optional[Period] = Field(
        description="Begin and end dates for the commercial distribution of the device",
        default=None,
    )
    subJurisdiction: Optional[Uri] = Field(
        description="National state or territory where the device is commercialized",
        default=None,
    )
    subJurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for subJurisdiction extensions",
        default=None,
        alias="_subJurisdiction",
    )

    @field_validator(
        *(
            "subJurisdiction",
            "marketPeriod",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionUdiDeviceIdentifier(BackboneElement):
    """
    Unique device identifier (UDI) assigned to device label or package.  Note that the Device may include multiple udiCarriers as it either may include just the udiCarrier for the jurisdiction it is sold, or for multiple jurisdictions it could have been sold.
    """

    deviceIdentifier: Optional[String] = Field(
        description="The identifier that is to be associated with every Device that references this DeviceDefintiion for the issuer and jurisdiction provided in the DeviceDefinition.udiDeviceIdentifier",
        default=None,
    )
    deviceIdentifier_ext: Optional[Element] = Field(
        description="Placeholder element for deviceIdentifier extensions",
        default=None,
        alias="_deviceIdentifier",
    )
    issuer: Optional[Uri] = Field(
        description="The organization that assigns the identifier algorithm",
        default=None,
    )
    issuer_ext: Optional[Element] = Field(
        description="Placeholder element for issuer extensions",
        default=None,
        alias="_issuer",
    )
    jurisdiction: Optional[Uri] = Field(
        description="The jurisdiction to which the deviceIdentifier applies",
        default=None,
    )
    jurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for jurisdiction extensions",
        default=None,
        alias="_jurisdiction",
    )
    marketDistribution: Optional[
        List[DeviceDefinitionUdiDeviceIdentifierMarketDistribution]
    ] = Field(
        description="Indicates whether and when the device is available on the market",
        default=None,
    )

    @field_validator(
        *(
            "marketDistribution",
            "jurisdiction",
            "issuer",
            "deviceIdentifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionRegulatoryIdentifier(BackboneElement):
    """
    Identifier associated with the regulatory documentation (certificates, technical documentation, post-market surveillance documentation and reports) of a set of device models sharing the same intended purpose, risk class and essential design and manufacturing characteristics. One example is the Basic UDI-DI in Europe.
    """

    type: Optional[Code] = Field(
        description="basic | master | license",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    deviceIdentifier: Optional[String] = Field(
        description="The identifier itself",
        default=None,
    )
    deviceIdentifier_ext: Optional[Element] = Field(
        description="Placeholder element for deviceIdentifier extensions",
        default=None,
        alias="_deviceIdentifier",
    )
    issuer: Optional[Uri] = Field(
        description="The organization that issued this identifier",
        default=None,
    )
    issuer_ext: Optional[Element] = Field(
        description="Placeholder element for issuer extensions",
        default=None,
        alias="_issuer",
    )
    jurisdiction: Optional[Uri] = Field(
        description="The jurisdiction to which the deviceIdentifier applies",
        default=None,
    )
    jurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for jurisdiction extensions",
        default=None,
        alias="_jurisdiction",
    )

    @field_validator(
        *(
            "jurisdiction",
            "issuer",
            "deviceIdentifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionDeviceName(BackboneElement):
    """
    The name or names of the device as given by the manufacturer.
    """

    name: Optional[String] = Field(
        description="A name that is used to refer to the device",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[Code] = Field(
        description="registered-name | user-friendly-name | patient-reported-name",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )

    @field_validator(
        *(
            "type",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionClassification(BackboneElement):
    """
    What kind of device or device system this is.
    """

    type: Optional[CodeableConcept] = Field(
        description="A classification or risk class of the device model",
        default=None,
    )
    justification: Optional[List[RelatedArtifact]] = Field(
        description="Further information qualifying this classification of the device model",
        default=None,
    )

    @field_validator(
        *(
            "justification",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionConformsTo(BackboneElement):
    """
    Identifies the standards, specifications, or formal guidances for the capabilities supported by the device. The device may be certified as conformant to these specifications e.g., communication, performance, process, measurement, or specialization standards.
    """

    category: Optional[CodeableConcept] = Field(
        description="Describes the common type of the standard, specification, or formal guidance",
        default=None,
    )
    specification: Optional[CodeableConcept] = Field(
        description="Identifies the standard, specification, or formal guidance that the device adheres to the Device Specification type",
        default=None,
    )
    version: Optional[List[String]] = Field(
        description="The specific form or variant of the standard, specification or formal guidance",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    source: Optional[List[RelatedArtifact]] = Field(
        description="Standard, regulation, certification, or guidance website, document, or other publication, or similar, supporting the conformance",
        default=None,
    )

    @field_validator(
        *(
            "source",
            "version",
            "specification",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionHasPart(BackboneElement):
    """
    A device that is part (for example a component) of the present device.
    """

    reference: Optional[Reference] = Field(
        description="Reference to the part",
        default=None,
    )
    count: Optional[Integer] = Field(
        description="Number of occurrences of the part",
        default=None,
    )
    count_ext: Optional[Element] = Field(
        description="Placeholder element for count extensions",
        default=None,
        alias="_count",
    )

    @field_validator(
        *(
            "count",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionPackagingDistributor(BackboneElement):
    """
    An organization that distributes the packaged device.
    """

    name: Optional[String] = Field(
        description="Distributor\u0027s human-readable name",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    organizationReference: Optional[List[Reference]] = Field(
        description="Distributor as an Organization resource",
        default=None,
    )

    @field_validator(
        *(
            "organizationReference",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionUdiDeviceIdentifierMarketDistribution(BackboneElement):
    """
    Indicates where and when the device is available on the market.
    """

    marketPeriod: Optional[Period] = Field(
        description="Begin and end dates for the commercial distribution of the device",
        default=None,
    )
    subJurisdiction: Optional[Uri] = Field(
        description="National state or territory where the device is commercialized",
        default=None,
    )
    subJurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for subJurisdiction extensions",
        default=None,
        alias="_subJurisdiction",
    )

    @field_validator(
        *(
            "subJurisdiction",
            "marketPeriod",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionPackagingUdiDeviceIdentifier(BackboneElement):
    """
    Unique Device Identifier (UDI) Barcode string on the packaging.
    """

    deviceIdentifier: Optional[String] = Field(
        description="The identifier that is to be associated with every Device that references this DeviceDefintiion for the issuer and jurisdiction provided in the DeviceDefinition.udiDeviceIdentifier",
        default=None,
    )
    deviceIdentifier_ext: Optional[Element] = Field(
        description="Placeholder element for deviceIdentifier extensions",
        default=None,
        alias="_deviceIdentifier",
    )
    issuer: Optional[Uri] = Field(
        description="The organization that assigns the identifier algorithm",
        default=None,
    )
    issuer_ext: Optional[Element] = Field(
        description="Placeholder element for issuer extensions",
        default=None,
        alias="_issuer",
    )
    jurisdiction: Optional[Uri] = Field(
        description="The jurisdiction to which the deviceIdentifier applies",
        default=None,
    )
    jurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for jurisdiction extensions",
        default=None,
        alias="_jurisdiction",
    )
    marketDistribution: Optional[
        List[DeviceDefinitionUdiDeviceIdentifierMarketDistribution]
    ] = Field(
        description="Indicates whether and when the device is available on the market",
        default=None,
    )

    @field_validator(
        *(
            "marketDistribution",
            "jurisdiction",
            "issuer",
            "deviceIdentifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionPackaging(BackboneElement):
    """
    Information about the packaging of the device, i.e. how the device is packaged.
    """

    identifier: Optional[Identifier] = Field(
        description="Business identifier of the packaged medication",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="A code that defines the specific type of packaging",
        default=None,
    )
    count: Optional[Integer] = Field(
        description="The number of items contained in the package (devices or sub-packages)",
        default=None,
    )
    count_ext: Optional[Element] = Field(
        description="Placeholder element for count extensions",
        default=None,
        alias="_count",
    )
    distributor: Optional[List[DeviceDefinitionPackagingDistributor]] = Field(
        description="An organization that distributes the packaged device",
        default=None,
    )
    udiDeviceIdentifier: Optional[
        List[DeviceDefinitionPackagingUdiDeviceIdentifier]
    ] = Field(
        description="Unique Device Identifier (UDI) Barcode string on the packaging",
        default=None,
    )
    packaging: Optional[List["DeviceDefinitionPackaging"]] = Field(
        description="Allows packages within packages",
        default=None,
    )

    @field_validator(
        *(
            "packaging",
            "udiDeviceIdentifier",
            "distributor",
            "count",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionVersion(BackboneElement):
    """
    The version of the device or software.
    """

    type: Optional[CodeableConcept] = Field(
        description="The type of the device version, e.g. manufacturer, approved, internal",
        default=None,
    )
    component: Optional[Identifier] = Field(
        description="The hardware or software module of the device to which the version applies",
        default=None,
    )
    value: Optional[String] = Field(
        description="The version text",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "component",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionProperty(BackboneElement):
    """
    Static or essentially fixed characteristics or features of this kind of device that are otherwise not captured in more specific attributes, e.g., time or timing attributes, resolution, accuracy, and physical attributes.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code that specifies the property being represented",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Value of the property",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Value of the property",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Value of the property",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the property",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Value of the property",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueRange: Optional[Range] = Field(
        description="Value of the property",
        default=None,
    )
    valueAttachment: Optional[Attachment] = Field(
        description="Value of the property",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Quantity,
                CodeableConcept,
                String,
                Boolean,
                Integer,
                Range,
                Attachment,
            ],
            field_name_base="value",
            required=True,
        )


class DeviceDefinitionLink(BackboneElement):
    """
    An associated device, attached to, used with, communicating with or linking a previous or new device model to the focal device.
    """

    relation: Optional[Coding] = Field(
        description="The type indicates the relationship of the related device to the device instance",
        default=None,
    )
    relatedDevice: Optional[CodeableReference] = Field(
        description="A reference to the linked device",
        default=None,
    )

    @field_validator(
        *(
            "relatedDevice",
            "relation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionMaterial(BackboneElement):
    """
    A substance used to create the material(s) of which the device is made.
    """

    substance: Optional[CodeableConcept] = Field(
        description="A relevant substance that the device contains, may contain, or is made of",
        default=None,
    )
    alternate: Optional[Boolean] = Field(
        description="Indicates an alternative material of the device",
        default=None,
    )
    alternate_ext: Optional[Element] = Field(
        description="Placeholder element for alternate extensions",
        default=None,
        alias="_alternate",
    )
    allergenicIndicator: Optional[Boolean] = Field(
        description="Whether the substance is a known or suspected allergen",
        default=None,
    )
    allergenicIndicator_ext: Optional[Element] = Field(
        description="Placeholder element for allergenicIndicator extensions",
        default=None,
        alias="_allergenicIndicator",
    )

    @field_validator(
        *(
            "allergenicIndicator",
            "alternate",
            "substance",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionGuideline(BackboneElement):
    """
    Information aimed at providing directions for the usage of this model of device.
    """

    useContext: Optional[List[UsageContext]] = Field(
        description="The circumstances that form the setting for using the device",
        default=None,
    )
    usageInstruction: Optional[Markdown] = Field(
        description="Detailed written and visual directions for the user on how to use the device",
        default=None,
    )
    usageInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for usageInstruction extensions",
        default=None,
        alias="_usageInstruction",
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="A source of information or reference for this guideline",
        default=None,
    )
    indication: Optional[List[CodeableConcept]] = Field(
        description="A clinical condition for which the device was designed to be used",
        default=None,
    )
    contraindication: Optional[List[CodeableConcept]] = Field(
        description="A specific situation when a device should not be used because it may cause harm",
        default=None,
    )
    warning: Optional[List[CodeableConcept]] = Field(
        description="Specific hazard alert information that a user needs to know before using the device",
        default=None,
    )
    intendedUse: Optional[String] = Field(
        description="A description of the general purpose or medical use of the device or its function",
        default=None,
    )
    intendedUse_ext: Optional[Element] = Field(
        description="Placeholder element for intendedUse extensions",
        default=None,
        alias="_intendedUse",
    )

    @field_validator(
        *(
            "intendedUse",
            "warning",
            "contraindication",
            "indication",
            "relatedArtifact",
            "usageInstruction",
            "useContext",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionCorrectiveAction(BackboneElement):
    """
    Tracking of latest field safety corrective action.
    """

    recall: Optional[Boolean] = Field(
        description="Whether the corrective action was a recall",
        default=None,
    )
    recall_ext: Optional[Element] = Field(
        description="Placeholder element for recall extensions",
        default=None,
        alias="_recall",
    )
    scope: Optional[Code] = Field(
        description="model | lot-numbers | serial-numbers",
        default=None,
    )
    scope_ext: Optional[Element] = Field(
        description="Placeholder element for scope extensions",
        default=None,
        alias="_scope",
    )
    period: Optional[Period] = Field(
        description="Start and end dates of the  corrective action",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "scope",
            "recall",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionChargeItem(BackboneElement):
    """
    Billing code or reference associated with the device.
    """

    chargeItemCode: Optional[CodeableReference] = Field(
        description="The code or reference for the charge item",
        default=None,
    )
    count: Optional[Quantity] = Field(
        description="Coefficient applicable to the billing code",
        default=None,
    )
    effectivePeriod: Optional[Period] = Field(
        description="A specific time period in which this charge item applies",
        default=None,
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context to which this charge item applies",
        default=None,
    )

    @field_validator(
        *(
            "useContext",
            "effectivePeriod",
            "count",
            "chargeItemCode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinition(DomainResource):
    """
    This is a specialized resource that defines the characteristics and capabilities of a device.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/DeviceDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Additional information to describe the device",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Instance identifier",
        default=None,
    )
    udiDeviceIdentifier: Optional[List[DeviceDefinitionUdiDeviceIdentifier]] = Field(
        description="Unique Device Identifier (UDI) Barcode string",
        default=None,
    )
    regulatoryIdentifier: Optional[List[DeviceDefinitionRegulatoryIdentifier]] = Field(
        description="Regulatory identifier(s) associated with this device",
        default=None,
    )
    partNumber: Optional[String] = Field(
        description="The part number or catalog number of the device",
        default=None,
    )
    partNumber_ext: Optional[Element] = Field(
        description="Placeholder element for partNumber extensions",
        default=None,
        alias="_partNumber",
    )
    manufacturer: Optional[Reference] = Field(
        description="Name of device manufacturer",
        default=None,
    )
    deviceName: Optional[List[DeviceDefinitionDeviceName]] = Field(
        description="The name or names of the device as given by the manufacturer",
        default=None,
    )
    modelNumber: Optional[String] = Field(
        description="The catalog or model number for the device for example as defined by the manufacturer",
        default=None,
    )
    modelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for modelNumber extensions",
        default=None,
        alias="_modelNumber",
    )
    classification: Optional[List[DeviceDefinitionClassification]] = Field(
        description="What kind of device or device system this is",
        default=None,
    )
    conformsTo: Optional[List[DeviceDefinitionConformsTo]] = Field(
        description="Identifies the standards, specifications, or formal guidances for the capabilities supported by the device",
        default=None,
    )
    hasPart: Optional[List[DeviceDefinitionHasPart]] = Field(
        description="A device, part of the current one",
        default=None,
    )
    packaging: Optional[List[DeviceDefinitionPackaging]] = Field(
        description="Information about the packaging of the device, i.e. how the device is packaged",
        default=None,
    )
    version: Optional[List[DeviceDefinitionVersion]] = Field(
        description="The version of the device or software",
        default=None,
    )
    safety: Optional[List[CodeableConcept]] = Field(
        description="Safety characteristics of the device",
        default=None,
    )
    shelfLifeStorage: Optional[List[ProductShelfLife]] = Field(
        description="Shelf Life and storage information",
        default=None,
    )
    languageCode: Optional[List[CodeableConcept]] = Field(
        description="Language code for the human-readable text strings produced by the device (all supported)",
        default=None,
    )
    property_: Optional[List[DeviceDefinitionProperty]] = Field(
        description="Inherent, essentially fixed, characteristics of this kind of device, e.g., time properties, size, etc",
        default=None,
    )
    owner: Optional[Reference] = Field(
        description="Organization responsible for device",
        default=None,
    )
    contact: Optional[List[ContactPoint]] = Field(
        description="Details for human/organization for support",
        default=None,
    )
    link: Optional[List[DeviceDefinitionLink]] = Field(
        description="An associated device, attached to, used with, communicating with or linking a previous or new device model to the focal device",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Device notes and comments",
        default=None,
    )
    material: Optional[List[DeviceDefinitionMaterial]] = Field(
        description="A substance used to create the material(s) of which the device is made",
        default=None,
    )
    productionIdentifierInUDI: Optional[List[Code]] = Field(
        description="lot-number | manufactured-date | serial-number | expiration-date | biological-source | software-version",
        default=None,
    )
    productionIdentifierInUDI_ext: Optional[Element] = Field(
        description="Placeholder element for productionIdentifierInUDI extensions",
        default=None,
        alias="_productionIdentifierInUDI",
    )
    guideline: Optional[DeviceDefinitionGuideline] = Field(
        description="Information aimed at providing directions for the usage of this model of device",
        default=None,
    )
    correctiveAction: Optional[DeviceDefinitionCorrectiveAction] = Field(
        description="Tracking of latest field safety corrective action",
        default=None,
    )
    chargeItem: Optional[List[DeviceDefinitionChargeItem]] = Field(
        description="Billing code or reference associated with the device",
        default=None,
    )
    resourceType: Literal["DeviceDefinition"] = Field(
        description=None,
        default="DeviceDefinition",
    )

    @field_validator(
        *(
            "chargeItem",
            "correctiveAction",
            "guideline",
            "productionIdentifierInUDI",
            "material",
            "note",
            "link",
            "contact",
            "owner",
            "property_",
            "languageCode",
            "shelfLifeStorage",
            "safety",
            "version",
            "packaging",
            "hasPart",
            "conformsTo",
            "classification",
            "modelNumber",
            "deviceName",
            "manufacturer",
            "partNumber",
            "regulatoryIdentifier",
            "udiDeviceIdentifier",
            "identifier",
            "description",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
