# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Base64Binary,
    DateTime,
    Boolean,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableReference,
    BackboneElement,
    CodeableConcept,
    Quantity,
    Range,
    Attachment,
    Count,
    Duration,
    Reference,
    ContactPoint,
    Annotation,
    DomainResource,
)


class DeviceUdiCarrier(BackboneElement):
    """
    Unique device identifier (UDI) assigned to device label or package.  Note that the Device may include multiple udiCarriers as it either may include just the udiCarrier for the jurisdiction it is sold, or for multiple jurisdictions it could have been sold.
    """

    deviceIdentifier: Optional[String] = Field(
        description="Mandatory fixed portion of UDI",
        default=None,
    )
    deviceIdentifier_ext: Optional[Element] = Field(
        description="Placeholder element for deviceIdentifier extensions",
        default=None,
        alias="_deviceIdentifier",
    )
    issuer: Optional[Uri] = Field(
        description="UDI Issuing Organization",
        default=None,
    )
    issuer_ext: Optional[Element] = Field(
        description="Placeholder element for issuer extensions",
        default=None,
        alias="_issuer",
    )
    jurisdiction: Optional[Uri] = Field(
        description="Regional UDI authority",
        default=None,
    )
    jurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for jurisdiction extensions",
        default=None,
        alias="_jurisdiction",
    )
    carrierAIDC: Optional[Base64Binary] = Field(
        description="UDI Machine Readable Barcode String",
        default=None,
    )
    carrierAIDC_ext: Optional[Element] = Field(
        description="Placeholder element for carrierAIDC extensions",
        default=None,
        alias="_carrierAIDC",
    )
    carrierHRF: Optional[String] = Field(
        description="UDI Human Readable Barcode String",
        default=None,
    )
    carrierHRF_ext: Optional[Element] = Field(
        description="Placeholder element for carrierHRF extensions",
        default=None,
        alias="_carrierHRF",
    )
    entryType: Optional[Code] = Field(
        description="barcode | rfid | manual | card | self-reported | electronic-transmission | unknown",
        default=None,
    )
    entryType_ext: Optional[Element] = Field(
        description="Placeholder element for entryType extensions",
        default=None,
        alias="_entryType",
    )

    @field_validator(
        *(
            "entryType",
            "carrierHRF",
            "carrierAIDC",
            "jurisdiction",
            "issuer",
            "deviceIdentifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceName(BackboneElement):
    """
    This represents the manufacturer's name of the device as provided by the device, from a UDI label, or by a person describing the Device.  This typically would be used when a person provides the name(s) or when the device represents one of the names available from DeviceDefinition.
    """

    value: Optional[String] = Field(
        description="The term that names the device",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    type: Optional[Code] = Field(
        description="registered-name | user-friendly-name | patient-reported-name",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    display: Optional[Boolean] = Field(
        description="The preferred device name",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )

    @field_validator(
        *(
            "display",
            "type",
            "value",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceVersion(BackboneElement):
    """
    The actual design of the device or software version running on the device.
    """

    type: Optional[CodeableConcept] = Field(
        description="The type of the device version, e.g. manufacturer, approved, internal",
        default=None,
    )
    component: Optional[Identifier] = Field(
        description="The hardware or software module of the device to which the version applies",
        default=None,
    )
    installDate: Optional[DateTime] = Field(
        description="The date the version was installed on the device",
        default=None,
    )
    installDate_ext: Optional[Element] = Field(
        description="Placeholder element for installDate extensions",
        default=None,
        alias="_installDate",
    )
    value: Optional[String] = Field(
        description="The version text",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "installDate",
            "component",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceConformsTo(BackboneElement):
    """
    Identifies the standards, specifications, or formal guidances for the capabilities supported by the device. The device may be certified as conformant to these specifications e.g., communication, performance, process, measurement, or specialization standards.
    """

    category: Optional[CodeableConcept] = Field(
        description="Describes the common type of the standard, specification, or formal guidance.  communication | performance | measurement",
        default=None,
    )
    specification: Optional[CodeableConcept] = Field(
        description="Identifies the standard, specification, or formal guidance that the device adheres to",
        default=None,
    )
    version: Optional[String] = Field(
        description="Specific form or variant of the standard",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )

    @field_validator(
        *(
            "version",
            "specification",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceProperty(BackboneElement):
    """
    Static or essentially fixed characteristics or features of the device (e.g., time or timing attributes, resolution, accuracy, intended use or instructions for use, and physical attributes) that are not otherwise captured in more specific attributes.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code that specifies the property being represented",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Value of the property",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Value of the property",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Value of the property",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the property",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Value of the property",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueRange: Optional[Range] = Field(
        description="Value of the property",
        default=None,
    )
    valueAttachment: Optional[Attachment] = Field(
        description="Value of the property",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Quantity,
                CodeableConcept,
                String,
                Boolean,
                Integer,
                Range,
                Attachment,
            ],
            field_name_base="value",
            required=True,
        )


class Device(DomainResource):
    """
    This resource describes the properties (regulated, has real time clock, etc.), adminstrative (manufacturer name, model number, serial number, firmware, etc.), and type (knee replacement, blood pressure cuff, MRI, etc.) of a physical unit (these values do not change much within a given module, for example the serail number, manufacturer name, and model number). An actual unit may consist of several modules in a distinct hierarchy and these are represented by multiple Device resources and bound through the 'parent' element.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Device"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Instance identifier",
        default=None,
    )
    displayName: Optional[String] = Field(
        description="The name used to display by default when the device is referenced",
        default=None,
    )
    displayName_ext: Optional[Element] = Field(
        description="Placeholder element for displayName extensions",
        default=None,
        alias="_displayName",
    )
    definition: Optional[CodeableReference] = Field(
        description="The reference to the definition for the device",
        default=None,
    )
    udiCarrier: Optional[List[DeviceUdiCarrier]] = Field(
        description="Unique Device Identifier (UDI) Barcode string",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | inactive | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    availabilityStatus: Optional[CodeableConcept] = Field(
        description="lost | damaged | destroyed | available",
        default=None,
    )
    biologicalSourceEvent: Optional[Identifier] = Field(
        description="An identifier that supports traceability to the event during which material in this product from one or more biological entities was obtained or pooled",
        default=None,
    )
    manufacturer: Optional[String] = Field(
        description="Name of device manufacturer",
        default=None,
    )
    manufacturer_ext: Optional[Element] = Field(
        description="Placeholder element for manufacturer extensions",
        default=None,
        alias="_manufacturer",
    )
    manufactureDate: Optional[DateTime] = Field(
        description="Date when the device was made",
        default=None,
    )
    manufactureDate_ext: Optional[Element] = Field(
        description="Placeholder element for manufactureDate extensions",
        default=None,
        alias="_manufactureDate",
    )
    expirationDate: Optional[DateTime] = Field(
        description="Date and time of expiry of this device (if applicable)",
        default=None,
    )
    expirationDate_ext: Optional[Element] = Field(
        description="Placeholder element for expirationDate extensions",
        default=None,
        alias="_expirationDate",
    )
    lotNumber: Optional[String] = Field(
        description="Lot number of manufacture",
        default=None,
    )
    lotNumber_ext: Optional[Element] = Field(
        description="Placeholder element for lotNumber extensions",
        default=None,
        alias="_lotNumber",
    )
    serialNumber: Optional[String] = Field(
        description="Serial number assigned by the manufacturer",
        default=None,
    )
    serialNumber_ext: Optional[Element] = Field(
        description="Placeholder element for serialNumber extensions",
        default=None,
        alias="_serialNumber",
    )
    name: Optional[List[DeviceName]] = Field(
        description="The name or names of the device as known to the manufacturer and/or patient",
        default=None,
    )
    modelNumber: Optional[String] = Field(
        description="The manufacturer\u0027s model number for the device",
        default=None,
    )
    modelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for modelNumber extensions",
        default=None,
        alias="_modelNumber",
    )
    partNumber: Optional[String] = Field(
        description="The part number or catalog number of the device",
        default=None,
    )
    partNumber_ext: Optional[Element] = Field(
        description="Placeholder element for partNumber extensions",
        default=None,
        alias="_partNumber",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Indicates a high-level grouping of the device",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="The kind or type of device",
        default=None,
    )
    version: Optional[List[DeviceVersion]] = Field(
        description="The actual design of the device or software version running on the device",
        default=None,
    )
    conformsTo: Optional[List[DeviceConformsTo]] = Field(
        description="Identifies the standards, specifications, or formal guidances for the capabilities supported by the device",
        default=None,
    )
    property_: Optional[List[DeviceProperty]] = Field(
        description="Inherent, essentially fixed, characteristics of the device.  e.g., time properties, size, material, etc.",
        default=None,
    )
    mode: Optional[CodeableConcept] = Field(
        description="The designated condition for performing a task",
        default=None,
    )
    cycle: Optional[Count] = Field(
        description="The series of occurrences that repeats during the operation of the device",
        default=None,
    )
    duration: Optional[Duration] = Field(
        description="A measurement of time during the device\u0027s operation (e.g., days, hours, mins, etc.)",
        default=None,
    )
    owner: Optional[Reference] = Field(
        description="Organization responsible for device",
        default=None,
    )
    contact: Optional[List[ContactPoint]] = Field(
        description="Details for human/organization for support",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where the device is found",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Network address to contact device",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Technical endpoints providing access to electronic services provided by the device",
        default=None,
    )
    gateway: Optional[List[CodeableReference]] = Field(
        description="Linked device acting as a communication/data collector, translator or controller",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Device notes and comments",
        default=None,
    )
    safety: Optional[List[CodeableConcept]] = Field(
        description="Safety Characteristics of Device",
        default=None,
    )
    parent: Optional[Reference] = Field(
        description="The higher level or encompassing device that this device is a logical part of",
        default=None,
    )
    resourceType: Literal["Device"] = Field(
        description=None,
        default="Device",
    )

    @field_validator(
        *(
            "parent",
            "safety",
            "note",
            "gateway",
            "endpoint",
            "url",
            "location",
            "contact",
            "owner",
            "duration",
            "cycle",
            "mode",
            "property_",
            "conformsTo",
            "version",
            "type",
            "category",
            "partNumber",
            "modelNumber",
            "name",
            "serialNumber",
            "lotNumber",
            "expirationDate",
            "manufactureDate",
            "manufacturer",
            "biologicalSourceEvent",
            "availabilityStatus",
            "status",
            "udiCarrier",
            "definition",
            "displayName",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dev_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.where(display=true).count() <= 1",
            human="only one Device.name.display SHALL be true when there is more than one Device.name",
            key="dev-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
