# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    PositiveInt,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    BackboneElement,
    Reference,
    CodeableConcept,
    Period,
    Quantity,
    Money,
    DomainResource,
)


class CoveragePaymentBy(BackboneElement):
    """
    Link to the paying party and optionally what specifically they will be responsible to pay.
    """

    party: Optional[Reference] = Field(
        description="Parties performing self-payment",
        default=None,
    )
    responsibility: Optional[String] = Field(
        description="Party\u0027s responsibility",
        default=None,
    )
    responsibility_ext: Optional[Element] = Field(
        description="Placeholder element for responsibility extensions",
        default=None,
        alias="_responsibility",
    )

    @field_validator(
        *(
            "responsibility",
            "party",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CoverageClass(BackboneElement):
    """
    A suite of underwriter specific classifiers.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of class such as \u0027group\u0027 or \u0027plan\u0027",
        default=None,
    )
    value: Optional[Identifier] = Field(
        description="Value associated with the type",
        default=None,
    )
    name: Optional[String] = Field(
        description="Human readable description of the type and value",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )

    @field_validator(
        *(
            "name",
            "value",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CoverageCostToBeneficiaryException(BackboneElement):
    """
    A suite of codes indicating exceptions or reductions to patient costs and their effective periods.
    """

    type: Optional[CodeableConcept] = Field(
        description="Exception category",
        default=None,
    )
    period: Optional[Period] = Field(
        description="The effective period of the exception",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CoverageCostToBeneficiary(BackboneElement):
    """
    A suite of codes indicating the cost category and associated amount which have been detailed in the policy and may have been  included on the health card.
    """

    type: Optional[CodeableConcept] = Field(
        description="Cost category",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    network: Optional[CodeableConcept] = Field(
        description="In or out of network",
        default=None,
    )
    unit: Optional[CodeableConcept] = Field(
        description="Individual or family",
        default=None,
    )
    term: Optional[CodeableConcept] = Field(
        description="Annual or lifetime",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="The amount or percentage due from the beneficiary",
        default=None,
    )
    valueMoney: Optional[Money] = Field(
        description="The amount or percentage due from the beneficiary",
        default=None,
    )
    exception: Optional[List[CoverageCostToBeneficiaryException]] = Field(
        description="Exceptions for patient payments",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "exception",
            "term",
            "unit",
            "network",
            "category",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Quantity, Money],
            field_name_base="value",
            required=False,
        )


class Coverage(DomainResource):
    """
    Financial instrument which may be used to reimburse or pay for health care products and services. Includes both insurance and self-payment.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Coverage"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier(s) for this coverage",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | cancelled | draft | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    kind: Optional[Code] = Field(
        description="insurance | self-pay | other",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    paymentBy: Optional[List[CoveragePaymentBy]] = Field(
        description="Self-pay parties and responsibility",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Coverage category such as medical or accident",
        default=None,
    )
    policyHolder: Optional[Reference] = Field(
        description="Owner of the policy",
        default=None,
    )
    subscriber: Optional[Reference] = Field(
        description="Subscriber to the policy",
        default=None,
    )
    subscriberId: Optional[List[Identifier]] = Field(
        description="ID assigned to the subscriber",
        default=None,
    )
    beneficiary: Optional[Reference] = Field(
        description="Plan beneficiary",
        default=None,
    )
    dependent: Optional[String] = Field(
        description="Dependent number",
        default=None,
    )
    dependent_ext: Optional[Element] = Field(
        description="Placeholder element for dependent extensions",
        default=None,
        alias="_dependent",
    )
    relationship: Optional[CodeableConcept] = Field(
        description="Beneficiary relationship to the subscriber",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Coverage start and end dates",
        default=None,
    )
    insurer: Optional[Reference] = Field(
        description="Issuer of the policy",
        default=None,
    )
    class_: Optional[List[CoverageClass]] = Field(
        description="Additional coverage classifications",
        default=None,
    )
    order: Optional[PositiveInt] = Field(
        description="Relative order of the coverage",
        default=None,
    )
    order_ext: Optional[Element] = Field(
        description="Placeholder element for order extensions",
        default=None,
        alias="_order",
    )
    network: Optional[String] = Field(
        description="Insurer network",
        default=None,
    )
    network_ext: Optional[Element] = Field(
        description="Placeholder element for network extensions",
        default=None,
        alias="_network",
    )
    costToBeneficiary: Optional[List[CoverageCostToBeneficiary]] = Field(
        description="Patient payments for services/products",
        default=None,
    )
    subrogation: Optional[Boolean] = Field(
        description="Reimbursement to insurer",
        default=None,
    )
    subrogation_ext: Optional[Element] = Field(
        description="Placeholder element for subrogation extensions",
        default=None,
        alias="_subrogation",
    )
    contract: Optional[List[Reference]] = Field(
        description="Contract details",
        default=None,
    )
    insurancePlan: Optional[Reference] = Field(
        description="Insurance plan details",
        default=None,
    )
    resourceType: Literal["Coverage"] = Field(
        description=None,
        default="Coverage",
    )

    @field_validator(
        *(
            "insurancePlan",
            "contract",
            "subrogation",
            "costToBeneficiary",
            "network",
            "order",
            "class_",
            "insurer",
            "period",
            "relationship",
            "dependent",
            "beneficiary",
            "subscriberId",
            "subscriber",
            "policyHolder",
            "type",
            "paymentBy",
            "kind",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
