# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Date,
    Url,
    Boolean,
    DateTime,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    Attachment,
    BackboneElement,
    Coding,
    Expression,
    DomainResource,
)


class ConsentPolicyBasis(BackboneElement):
    """
    A Reference or URL used to uniquely identify the policy the organization will enforce for this Consent. This Reference or URL should be specific to the version of the policy and should be dereferencable to a computable policy of some form.
    """

    reference: Optional[Reference] = Field(
        description="Reference backing policy resource",
        default=None,
    )
    url: Optional[Url] = Field(
        description="URL to a computable backing policy",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentVerification(BackboneElement):
    """
    Whether a treatment instruction (e.g. artificial respiration: yes or no) was verified with the patient, his/her family or another authorized person.
    """

    verified: Optional[Boolean] = Field(
        description="Has been verified",
        default=None,
    )
    verified_ext: Optional[Element] = Field(
        description="Placeholder element for verified extensions",
        default=None,
        alias="_verified",
    )
    verificationType: Optional[CodeableConcept] = Field(
        description="Business case of verification",
        default=None,
    )
    verifiedBy: Optional[Reference] = Field(
        description="Person conducting verification",
        default=None,
    )
    verifiedWith: Optional[Reference] = Field(
        description="Person who verified",
        default=None,
    )
    verificationDate: Optional[List[DateTime]] = Field(
        description="When consent verified",
        default=None,
    )
    verificationDate_ext: Optional[Element] = Field(
        description="Placeholder element for verificationDate extensions",
        default=None,
        alias="_verificationDate",
    )

    @field_validator(
        *(
            "verificationDate",
            "verifiedWith",
            "verifiedBy",
            "verificationType",
            "verified",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentProvisionActor(BackboneElement):
    """
    Who or what is controlled by this provision. Use group to identify a set of actors by some property they share (e.g. 'admitting officers').
    """

    role: Optional[CodeableConcept] = Field(
        description="How the actor is involved",
        default=None,
    )
    reference: Optional[Reference] = Field(
        description="Resource for the actor (or group, by role)",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "role",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentProvisionData(BackboneElement):
    """
    The resources controlled by this provision if specific resources are referenced.
    """

    meaning: Optional[Code] = Field(
        description="instance | related | dependents | authoredby",
        default=None,
    )
    meaning_ext: Optional[Element] = Field(
        description="Placeholder element for meaning extensions",
        default=None,
        alias="_meaning",
    )
    reference: Optional[Reference] = Field(
        description="The actual data reference",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "meaning",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentProvision(BackboneElement):
    """
    An exception to the base policy of this consent. An exception can be an addition or removal of access permissions.
    """

    period: Optional[Period] = Field(
        description="Timeframe for this provision",
        default=None,
    )
    actor: Optional[List[ConsentProvisionActor]] = Field(
        description="Who|what controlled by this provision (or group, by role)",
        default=None,
    )
    action: Optional[List[CodeableConcept]] = Field(
        description="Actions controlled by this provision",
        default=None,
    )
    securityLabel: Optional[List[Coding]] = Field(
        description="Security Labels that define affected resources",
        default=None,
    )
    purpose: Optional[List[Coding]] = Field(
        description="Context of activities covered by this provision",
        default=None,
    )
    documentType: Optional[List[Coding]] = Field(
        description="e.g. Resource Type, Profile, CDA, etc",
        default=None,
    )
    resourceType: Optional[List[Coding]] = Field(
        description="e.g. Resource Type, Profile, etc",
        default=None,
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="e.g. LOINC or SNOMED CT code, etc. in the content",
        default=None,
    )
    dataPeriod: Optional[Period] = Field(
        description="Timeframe for data controlled by this provision",
        default=None,
    )
    data: Optional[List[ConsentProvisionData]] = Field(
        description="Data controlled by this provision",
        default=None,
    )
    expression: Optional[Expression] = Field(
        description="A computable expression of the consent",
        default=None,
    )
    provision: Optional[List["ConsentProvision"]] = Field(
        description="Nested Exception Provisions",
        default=None,
    )

    @field_validator(
        *(
            "provision",
            "expression",
            "data",
            "dataPeriod",
            "code",
            "resourceType",
            "documentType",
            "purpose",
            "securityLabel",
            "action",
            "actor",
            "period",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Consent(DomainResource):
    """
    A record of a healthcare consumer’s  choices  or choices made on their behalf by a third party, which permits or denies identified recipient(s) or recipient role(s) to perform one or more actions within a given policy context, for specific purposes and periods of time.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Consent"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier for this record (external references)",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | inactive | not-done | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Classification of the consent statement - for indexing/retrieval",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who the consent applies to",
        default=None,
    )
    date: Optional[Date] = Field(
        description="Fully executed date of the consent",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    period: Optional[Period] = Field(
        description="Effective period for this Consent",
        default=None,
    )
    grantor: Optional[List[Reference]] = Field(
        description="Who is granting rights according to the policy and rules",
        default=None,
    )
    grantee: Optional[List[Reference]] = Field(
        description="Who is agreeing to the policy and rules",
        default=None,
    )
    manager: Optional[List[Reference]] = Field(
        description="Consent workflow management",
        default=None,
    )
    controller: Optional[List[Reference]] = Field(
        description="Consent Enforcer",
        default=None,
    )
    sourceAttachment: Optional[List[Attachment]] = Field(
        description="Source from which this consent is taken",
        default=None,
    )
    sourceReference: Optional[List[Reference]] = Field(
        description="Source from which this consent is taken",
        default=None,
    )
    regulatoryBasis: Optional[List[CodeableConcept]] = Field(
        description="Regulations establishing base Consent",
        default=None,
    )
    policyBasis: Optional[ConsentPolicyBasis] = Field(
        description="Computable version of the backing policy",
        default=None,
    )
    policyText: Optional[List[Reference]] = Field(
        description="Human Readable Policy",
        default=None,
    )
    verification: Optional[List[ConsentVerification]] = Field(
        description="Consent Verified by patient or family",
        default=None,
    )
    decision: Optional[Code] = Field(
        description="deny | permit",
        default=None,
    )
    decision_ext: Optional[Element] = Field(
        description="Placeholder element for decision extensions",
        default=None,
        alias="_decision",
    )
    provision: Optional[List[ConsentProvision]] = Field(
        description="Constraints to the base Consent.policyRule/Consent.policy",
        default=None,
    )
    resourceType: Literal["Consent"] = Field(
        description=None,
        default="Consent",
    )

    @field_validator(
        *(
            "provision",
            "decision",
            "verification",
            "policyText",
            "policyBasis",
            "regulatoryBasis",
            "sourceReference",
            "sourceAttachment",
            "controller",
            "manager",
            "grantee",
            "grantor",
            "period",
            "date",
            "subject",
            "category",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
