# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
    Canonical,
    Integer,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    Period,
    RelatedArtifact,
    BackboneElement,
    Quantity,
    DomainResource,
)


class ConceptMapProperty(BackboneElement):
    """
    A property defines a slot through which additional information can be provided about a map from source -> target.
    """

    code: Optional[Code] = Field(
        description="Identifies the property on the mappings, and when referred to in the $translate operation",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    uri: Optional[Uri] = Field(
        description="Formal identifier for the property",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    description: Optional[String] = Field(
        description="Why the property is defined, and/or what it conveys",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    type: Optional[Code] = Field(
        description="Coding | string | integer | boolean | dateTime | decimal | code",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    system: Optional[Canonical] = Field(
        description="The CodeSystem from which code values come",
        default=None,
    )
    system_ext: Optional[Element] = Field(
        description="Placeholder element for system extensions",
        default=None,
        alias="_system",
    )

    @field_validator(
        *(
            "system",
            "type",
            "description",
            "uri",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConceptMapAdditionalAttribute(BackboneElement):
    """
    An additionalAttribute defines an additional data element found in the source or target data model where the data will come from or be mapped to. Some mappings are based on data in addition to the source data element, where codes in multiple fields are combined to a single field (or vice versa).
    """

    code: Optional[Code] = Field(
        description="Identifies this additional attribute through this resource",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    uri: Optional[Uri] = Field(
        description="Formal identifier for the data element referred to in this attribte",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    description: Optional[String] = Field(
        description="Why the additional attribute is defined, and/or what the data element it refers to is",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    type: Optional[Code] = Field(
        description="code | Coding | string | boolean | Quantity",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )

    @field_validator(
        *(
            "type",
            "description",
            "uri",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConceptMapGroupElementTargetProperty(BackboneElement):
    """
    A property value for this source -> target mapping.
    """

    code: Optional[Code] = Field(
        description="Reference to ConceptMap.property.code",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    valueCoding: Optional[Coding] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueInteger: Optional[Integer] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )
    valueCode: Optional[Code] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueCode_ext: Optional[Element] = Field(
        description="Placeholder element for valueCode extensions",
        default=None,
        alias="_valueCode",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("code", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Coding, String, Integer, Boolean, DateTime, Decimal, Code],
            field_name_base="value",
            required=True,
        )


class ConceptMapGroupElementTargetDependsOn(BackboneElement):
    """
    A set of additional dependencies for this mapping to hold. This mapping is only applicable if the specified data attribute can be resolved, and it has the specified value.
    """

    attribute: Optional[Code] = Field(
        description="A reference to a mapping attribute defined in ConceptMap.additionalAttribute",
        default=None,
    )
    attribute_ext: Optional[Element] = Field(
        description="Placeholder element for attribute extensions",
        default=None,
        alias="_attribute",
    )
    valueCode: Optional[Code] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueCode_ext: Optional[Element] = Field(
        description="Placeholder element for valueCode extensions",
        default=None,
        alias="_valueCode",
    )
    valueCoding: Optional[Coding] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueSet: Optional[Canonical] = Field(
        description="The mapping depends on a data element with a value from this value set",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "valueSet",
            "attribute",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Code, Coding, String, Boolean, Quantity],
            field_name_base="value",
            required=False,
        )


class ConceptMapGroupElementTargetProduct(BackboneElement):
    """
    Product is the output of a ConceptMap that provides additional values that go in other attributes / data elemnts of the target data.
    """

    attribute: Optional[Code] = Field(
        description="A reference to a mapping attribute defined in ConceptMap.additionalAttribute",
        default=None,
    )
    attribute_ext: Optional[Element] = Field(
        description="Placeholder element for attribute extensions",
        default=None,
        alias="_attribute",
    )
    valueCode: Optional[Code] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueCode_ext: Optional[Element] = Field(
        description="Placeholder element for valueCode extensions",
        default=None,
        alias="_valueCode",
    )
    valueCoding: Optional[Coding] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Value of the referenced data element",
        default=None,
    )
    valueSet: Optional[Canonical] = Field(
        description="The mapping depends on a data element with a value from this value set",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "valueSet",
            "attribute",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Code, Coding, String, Boolean, Quantity],
            field_name_base="value",
            required=False,
        )


class ConceptMapGroupElementTarget(BackboneElement):
    """
    A concept from the target value set that this concept maps to.
    """

    code: Optional[Code] = Field(
        description="Code that identifies the target element",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="Display for the code",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    valueSet: Optional[Canonical] = Field(
        description="Identifies the set of target concepts",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )
    relationship: Optional[Code] = Field(
        description="related-to | equivalent | source-is-narrower-than-target | source-is-broader-than-target | not-related-to",
        default=None,
    )
    relationship_ext: Optional[Element] = Field(
        description="Placeholder element for relationship extensions",
        default=None,
        alias="_relationship",
    )
    comment: Optional[String] = Field(
        description="Description of status/issues in mapping",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )
    property_: Optional[List[ConceptMapGroupElementTargetProperty]] = Field(
        description="Property value for the source -\u003e target mapping",
        default=None,
    )
    dependsOn: Optional[List[ConceptMapGroupElementTargetDependsOn]] = Field(
        description="Other properties required for this mapping",
        default=None,
    )
    product: Optional[List[ConceptMapGroupElementTargetProduct]] = Field(
        description="Other data elements that this mapping also produces",
        default=None,
    )

    @field_validator(
        *(
            "product",
            "dependsOn",
            "property_",
            "comment",
            "relationship",
            "valueSet",
            "display",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("dependsOn",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_6_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(value.exists() and valueSet.empty()) or (value.empty() and valueSet.exists())",
            human="One of value[x] or valueSet must exist, but not both.",
            key="cmd-6",
            severity="error",
        )


class ConceptMapGroupElement(BackboneElement):
    """
    Mappings for an individual concept in the source to one or more concepts in the target.
    """

    code: Optional[Code] = Field(
        description="Identifies element being mapped",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="Display for the code",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    valueSet: Optional[Canonical] = Field(
        description="Identifies the set of concepts being mapped",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )
    noMap: Optional[Boolean] = Field(
        description="No mapping to a target concept for this source concept",
        default=None,
    )
    noMap_ext: Optional[Element] = Field(
        description="Placeholder element for noMap extensions",
        default=None,
        alias="_noMap",
    )
    target: Optional[List[ConceptMapGroupElementTarget]] = Field(
        description="Concept in target system for element",
        default=None,
    )

    @field_validator(
        *(
            "target",
            "noMap",
            "valueSet",
            "display",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("target",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="comment.exists() or (%resource.status = 'draft') or relationship.empty() or ((relationship != 'source-is-broader-than-target') and (relationship != 'not-related-to'))",
            human="If the map is source-is-broader-than-target or not-related-to, there SHALL be some comments, unless the status is 'draft'",
            key="cmd-1",
            severity="error",
        )

    @field_validator(*("target",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_7_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(code.exists() and valueSet.empty()) or (code.empty() and valueSet.exists())",
            human="Either code or valueSet SHALL be present but not both.",
            key="cmd-7",
            severity="error",
        )


class ConceptMapGroupUnmapped(BackboneElement):
    """
    What to do when there is no mapping to a target concept from the source concept and ConceptMap.group.element.noMap is not true. This provides the "default" to be applied when there is no target concept mapping specified or the expansion of ConceptMap.group.element.target.valueSet is empty.
    """

    mode: Optional[Code] = Field(
        description="use-source-code | fixed | other-map",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    code: Optional[Code] = Field(
        description="Fixed code when mode = fixed",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="Display for the code",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    valueSet: Optional[Canonical] = Field(
        description="Fixed code set when mode = fixed",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )
    relationship: Optional[Code] = Field(
        description="related-to | equivalent | source-is-narrower-than-target | source-is-broader-than-target | not-related-to",
        default=None,
    )
    relationship_ext: Optional[Element] = Field(
        description="Placeholder element for relationship extensions",
        default=None,
        alias="_relationship",
    )
    otherMap: Optional[Canonical] = Field(
        description="canonical reference to an additional ConceptMap to use for mapping if the source concept is unmapped",
        default=None,
    )
    otherMap_ext: Optional[Element] = Field(
        description="Placeholder element for otherMap extensions",
        default=None,
        alias="_otherMap",
    )

    @field_validator(
        *(
            "otherMap",
            "relationship",
            "valueSet",
            "display",
            "code",
            "mode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConceptMapGroup(BackboneElement):
    """
    A group of mappings that all have the same source and target system.
    """

    source: Optional[Canonical] = Field(
        description="Source system where concepts to be mapped are defined",
        default=None,
    )
    source_ext: Optional[Element] = Field(
        description="Placeholder element for source extensions",
        default=None,
        alias="_source",
    )
    target: Optional[Canonical] = Field(
        description="Target system that the concepts are to be mapped to",
        default=None,
    )
    target_ext: Optional[Element] = Field(
        description="Placeholder element for target extensions",
        default=None,
        alias="_target",
    )
    element: Optional[List[ConceptMapGroupElement]] = Field(
        description="Mappings for a concept from the source set",
        default=None,
    )
    unmapped: Optional[ConceptMapGroupUnmapped] = Field(
        description="What to do when there is no mapping target for the source concept and ConceptMap.group.element.noMap is not true",
        default=None,
    )

    @field_validator(
        *(
            "unmapped",
            "element",
            "target",
            "source",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("element",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_4_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(noMap.exists() and noMap=true) implies target.empty()",
            human="If noMap is present, target SHALL NOT be present",
            key="cmd-4",
            severity="error",
        )

    @field_validator(*("element",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(code.exists() and valueSet.empty()) or (code.empty() and valueSet.exists())",
            human="Either code or valueSet SHALL be present but not both.",
            key="cmd-5",
            severity="error",
        )

    @field_validator(*("unmapped",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(mode = 'fixed') implies ((code.exists() and valueSet.empty()) or (code.empty() and valueSet.exists()))",
            human="If the mode is 'fixed', either a code or valueSet must be provided, but not both.",
            key="cmd-2",
            severity="error",
        )

    @field_validator(*("unmapped",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(mode = 'other-map') implies otherMap.exists()",
            human="If the mode is 'other-map', a url for the other map must be provided",
            key="cmd-3",
            severity="error",
        )

    @field_validator(*("unmapped",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_8_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(mode != 'fixed') implies (code.empty() and display.empty() and valueSet.empty())",
            human="If the mode is not 'fixed', code, display and valueSet are not allowed",
            key="cmd-8",
            severity="error",
        )

    @field_validator(*("unmapped",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_9_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(mode != 'other-map') implies relationship.exists()",
            human="If the mode is not 'other-map', relationship must be provided",
            key="cmd-9",
            severity="error",
        )

    @field_validator(*("unmapped",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_10_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(mode != 'other-map') implies otherMap.empty()",
            human="If the mode is not 'other-map', otherMap is not allowed",
            key="cmd-10",
            severity="error",
        )


class ConceptMap(DomainResource):
    """
    A statement of relationships from one set of concepts to one or more other concepts - either concepts in code systems, or data element/data element concepts, or classes in class models.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ConceptMap"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this concept map, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the concept map",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the concept map",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this concept map (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this concept map (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the concept map",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for concept map (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this concept map is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    approvalDate: Optional[Date] = Field(
        description="When the ConceptMap was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the ConceptMap was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the ConceptMap is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="E.g. Education, Treatment, Assessment, etc",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the ConceptMap",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the ConceptMap",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the ConceptMap",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the ConceptMap",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations, etc",
        default=None,
    )
    property_: Optional[List[ConceptMapProperty]] = Field(
        description="Additional properties of the mapping",
        default=None,
    )
    additionalAttribute: Optional[List[ConceptMapAdditionalAttribute]] = Field(
        description="Definition of an additional attribute to act as a data source or target",
        default=None,
    )
    sourceScopeUri: Optional[Uri] = Field(
        description="The source value set that contains the concepts that are being mapped",
        default=None,
    )
    sourceScopeUri_ext: Optional[Element] = Field(
        description="Placeholder element for sourceScopeUri extensions",
        default=None,
        alias="_sourceScopeUri",
    )
    sourceScopeCanonical: Optional[Canonical] = Field(
        description="The source value set that contains the concepts that are being mapped",
        default=None,
    )
    sourceScopeCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for sourceScopeCanonical extensions",
        default=None,
        alias="_sourceScopeCanonical",
    )
    targetScopeUri: Optional[Uri] = Field(
        description="The target value set which provides context for the mappings",
        default=None,
    )
    targetScopeUri_ext: Optional[Element] = Field(
        description="Placeholder element for targetScopeUri extensions",
        default=None,
        alias="_targetScopeUri",
    )
    targetScopeCanonical: Optional[Canonical] = Field(
        description="The target value set which provides context for the mappings",
        default=None,
    )
    targetScopeCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for targetScopeCanonical extensions",
        default=None,
        alias="_targetScopeCanonical",
    )
    group: Optional[List[ConceptMapGroup]] = Field(
        description="Same source and target systems",
        default=None,
    )
    resourceType: Literal["ConceptMap"] = Field(
        description=None,
        default="ConceptMap",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @property
    def sourceScope(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="sourceScope",
        )

    @property
    def targetScope(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="targetScope",
        )

    @field_validator(
        *(
            "group",
            "additionalAttribute",
            "property_",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @field_validator(*("property_",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmd_11_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type = 'code' implies system.exists()",
            human="If the property type is code, a system SHALL be specified",
            key="cmd-11",
            severity="error",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def sourceScope_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Uri, Canonical],
            field_name_base="sourceScope",
            required=False,
        )

    @model_validator(mode="after")
    def targetScope_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Uri, Canonical],
            field_name_base="targetScope",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )
