# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    UsageContext,
    Annotation,
    BackboneElement,
    RelatedArtifact,
    Period,
    CodeableReference,
    DomainResource,
)


class CompositionAttester(BackboneElement):
    """
    A participant who has attested to the accuracy of the composition/document.
    """

    mode: Optional[CodeableConcept] = Field(
        description="personal | professional | legal | official",
        default=None,
    )
    time: Optional[DateTime] = Field(
        description="When the composition was attested",
        default=None,
    )
    time_ext: Optional[Element] = Field(
        description="Placeholder element for time extensions",
        default=None,
        alias="_time",
    )
    party: Optional[Reference] = Field(
        description="Who attested the composition",
        default=None,
    )

    @field_validator(
        *(
            "party",
            "time",
            "mode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CompositionEvent(BackboneElement):
    """
    The clinical service, such as a colonoscopy or an appendectomy, being documented.
    """

    period: Optional[Period] = Field(
        description="The period covered by the documentation",
        default=None,
    )
    detail: Optional[List[CodeableReference]] = Field(
        description="The event(s) being documented, as code(s), reference(s), or both",
        default=None,
    )

    @field_validator(
        *(
            "detail",
            "period",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CompositionSection(BackboneElement):
    """
    The root of the sections that make up the composition.
    """

    title: Optional[String] = Field(
        description="Label for section (e.g. for ToC)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    code: Optional[CodeableConcept] = Field(
        description="Classification of section (recommended)",
        default=None,
    )
    author: Optional[List[Reference]] = Field(
        description="Who and/or what authored the section",
        default=None,
    )
    focus: Optional[Reference] = Field(
        description="Who/what the section is about, when it is not about the subject of composition",
        default=None,
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the section, for human interpretation",
        default=None,
    )
    orderedBy: Optional[CodeableConcept] = Field(
        description="Order of section entries",
        default=None,
    )
    entry: Optional[List[Reference]] = Field(
        description="A reference to data that supports this section",
        default=None,
    )
    emptyReason: Optional[CodeableConcept] = Field(
        description="Why the section is empty",
        default=None,
    )
    section: Optional[List["CompositionSection"]] = Field(
        description="Nested Section",
        default=None,
    )

    @field_validator(
        *(
            "section",
            "emptyReason",
            "entry",
            "orderedBy",
            "text",
            "focus",
            "author",
            "code",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Composition(DomainResource):
    """
    A set of healthcare-related information that is assembled together into a single logical package that provides a single coherent statement of meaning, establishes its own context and that has clinical attestation with regard to who is making the statement. A Composition defines the structure and narrative content necessary for a document. However, a Composition alone does not constitute a document. Rather, the Composition must be the first entry in a Bundle where Bundle.type=document, and any other resources referenced from Composition must be included as subsequent entries in the Bundle (for example Patient, Practitioner, Encounter, etc.).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Composition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this Composition, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Version-independent identifier for the Composition",
        default=None,
    )
    version: Optional[String] = Field(
        description="An explicitly assigned identifer of a variation of the content in the Composition",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    status: Optional[Code] = Field(
        description="registered | partial | preliminary | final | amended | corrected | appended | cancelled | entered-in-error | deprecated | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[CodeableConcept] = Field(
        description="Kind of composition (LOINC if possible)",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Categorization of Composition",
        default=None,
    )
    subject: Optional[List[Reference]] = Field(
        description="Who and/or what the composition is about",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Context of the Composition",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="Composition editing time",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    author: Optional[List[Reference]] = Field(
        description="Who and/or what authored the composition",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this Composition (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Human Readable name/title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    note: Optional[List[Annotation]] = Field(
        description="For any additional notes",
        default=None,
    )
    attester: Optional[List[CompositionAttester]] = Field(
        description="Attests to accuracy of composition",
        default=None,
    )
    custodian: Optional[Reference] = Field(
        description="Organization which maintains the composition",
        default=None,
    )
    relatesTo: Optional[List[RelatedArtifact]] = Field(
        description="Relationships to other compositions/documents",
        default=None,
    )
    event: Optional[List[CompositionEvent]] = Field(
        description="The clinical service(s) being documented",
        default=None,
    )
    section: Optional[List[CompositionSection]] = Field(
        description="Composition is broken into sections",
        default=None,
    )
    resourceType: Literal["Composition"] = Field(
        description=None,
        default="Composition",
    )

    @field_validator(
        *(
            "section",
            "event",
            "relatesTo",
            "custodian",
            "attester",
            "note",
            "title",
            "name",
            "author",
            "useContext",
            "date",
            "encounter",
            "subject",
            "category",
            "type",
            "status",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("section",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmp_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="text.exists() or entry.exists() or section.exists()",
            human="A section must contain at least one of text, entries, or sub-sections",
            key="cmp-1",
            severity="error",
        )

    @field_validator(*("section",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cmp_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="emptyReason.empty() or entry.empty()",
            human="A section can only have an emptyReason if it is empty",
            key="cmp-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
