# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
    Canonical,
    UnsignedInt,
    Integer,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    Period,
    RelatedArtifact,
    BackboneElement,
    DomainResource,
)


class CodeSystemFilter(BackboneElement):
    """
    A filter that can be used in a value set compose statement when selecting concepts using a filter.
    """

    code: Optional[Code] = Field(
        description="Code that identifies the filter",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    description: Optional[String] = Field(
        description="How or why the filter is used",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    operator: Optional[List[Code]] = Field(
        description="= | is-a | descendent-of | is-not-a | regex | in | not-in | generalizes | child-of | descendent-leaf | exists",
        default=None,
    )
    operator_ext: Optional[Element] = Field(
        description="Placeholder element for operator extensions",
        default=None,
        alias="_operator",
    )
    value: Optional[String] = Field(
        description="What to use for the value",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "operator",
            "description",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CodeSystemProperty(BackboneElement):
    """
    A property defines an additional slot through which additional information can be provided about a concept.
    """

    code: Optional[Code] = Field(
        description="Identifies the property on the concepts, and when referred to in operations",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    uri: Optional[Uri] = Field(
        description="Formal identifier for the property",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    description: Optional[String] = Field(
        description="Why the property is defined, and/or what it conveys",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    type: Optional[Code] = Field(
        description="code | Coding | string | integer | boolean | dateTime | decimal",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )

    @field_validator(
        *(
            "type",
            "description",
            "uri",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CodeSystemConceptDesignation(BackboneElement):
    """
    Additional representations for the concept - other languages, aliases, specialized purposes, used for particular purposes, etc.
    """

    language: Optional[Code] = Field(
        description="Human language of the designation",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    use: Optional[Coding] = Field(
        description="Details how this designation would be used",
        default=None,
    )
    additionalUse: Optional[List[Coding]] = Field(
        description="Additional ways how this designation would be used",
        default=None,
    )
    value: Optional[String] = Field(
        description="The text value for this designation",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "additionalUse",
            "use",
            "language",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CodeSystemConceptProperty(BackboneElement):
    """
    A property value for this concept.
    """

    code: Optional[Code] = Field(
        description="Reference to CodeSystem.property.code",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    valueCode: Optional[Code] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueCode_ext: Optional[Element] = Field(
        description="Placeholder element for valueCode extensions",
        default=None,
        alias="_valueCode",
    )
    valueCoding: Optional[Coding] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueInteger: Optional[Integer] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("code", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Code, Coding, String, Integer, Boolean, DateTime, Decimal],
            field_name_base="value",
            required=True,
        )


class CodeSystemConcept(BackboneElement):
    """
    Concepts that are in the code system. The concept definitions are inherently hierarchical, but the definitions must be consulted to determine what the meanings of the hierarchical relationships are.
    """

    code: Optional[Code] = Field(
        description="Code that identifies concept",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="Text to display to the user",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    definition: Optional[String] = Field(
        description="Formal definition",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    designation: Optional[List[CodeSystemConceptDesignation]] = Field(
        description="Additional representations for the concept",
        default=None,
    )
    property_: Optional[List[CodeSystemConceptProperty]] = Field(
        description="Property value for the concept",
        default=None,
    )
    concept: Optional[List["CodeSystemConcept"]] = Field(
        description="Child Concepts (is-a/contains/categorizes)",
        default=None,
    )

    @field_validator(
        *(
            "concept",
            "property_",
            "designation",
            "definition",
            "display",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("designation",), mode="after", check_fields=None)
    @classmethod
    def FHIR_csd_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="additionalUse.exists() implies use.exists()",
            human="Must have a value for concept.designation.use if concept.designation.additionalUse is present",
            key="csd-5",
            severity="error",
        )


class CodeSystem(DomainResource):
    """
    The CodeSystem resource is used to declare the existence of and describe a code system or code system supplement and its key properties, and optionally define a part or all of its content.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/CodeSystem"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this code system, represented as a URI (globally unique) (Coding.system)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the code system (business identifier)",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the code system (Coding.version)",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this code system (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this code system (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the code system",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for code system (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this code system is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    approvalDate: Optional[Date] = Field(
        description="When the CodeSystem was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the CodeSystem was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the CodeSystem is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="E.g. Education, Treatment, Assessment, etc",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the CodeSystem",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the CodeSystem",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the CodeSystem",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the CodeSystem",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations, etc",
        default=None,
    )
    caseSensitive: Optional[Boolean] = Field(
        description="If code comparison is case sensitive",
        default=None,
    )
    caseSensitive_ext: Optional[Element] = Field(
        description="Placeholder element for caseSensitive extensions",
        default=None,
        alias="_caseSensitive",
    )
    valueSet: Optional[Canonical] = Field(
        description="Canonical reference to the value set with entire code system",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )
    hierarchyMeaning: Optional[Code] = Field(
        description="grouped-by | is-a | part-of | classified-with",
        default=None,
    )
    hierarchyMeaning_ext: Optional[Element] = Field(
        description="Placeholder element for hierarchyMeaning extensions",
        default=None,
        alias="_hierarchyMeaning",
    )
    compositional: Optional[Boolean] = Field(
        description="If code system defines a compositional grammar",
        default=None,
    )
    compositional_ext: Optional[Element] = Field(
        description="Placeholder element for compositional extensions",
        default=None,
        alias="_compositional",
    )
    versionNeeded: Optional[Boolean] = Field(
        description="If definitions are not stable",
        default=None,
    )
    versionNeeded_ext: Optional[Element] = Field(
        description="Placeholder element for versionNeeded extensions",
        default=None,
        alias="_versionNeeded",
    )
    content: Optional[Code] = Field(
        description="not-present | example | fragment | complete | supplement",
        default=None,
    )
    content_ext: Optional[Element] = Field(
        description="Placeholder element for content extensions",
        default=None,
        alias="_content",
    )
    supplements: Optional[Canonical] = Field(
        description="Canonical URL of Code System this adds designations and properties to",
        default=None,
    )
    supplements_ext: Optional[Element] = Field(
        description="Placeholder element for supplements extensions",
        default=None,
        alias="_supplements",
    )
    count: Optional[UnsignedInt] = Field(
        description="Total concepts in the code system",
        default=None,
    )
    count_ext: Optional[Element] = Field(
        description="Placeholder element for count extensions",
        default=None,
        alias="_count",
    )
    filter: Optional[List[CodeSystemFilter]] = Field(
        description="Filter that can be used in a value set",
        default=None,
    )
    property_: Optional[List[CodeSystemProperty]] = Field(
        description="Additional information supplied about each concept",
        default=None,
    )
    concept: Optional[List[CodeSystemConcept]] = Field(
        description="Concepts in the code system",
        default=None,
    )
    resourceType: Literal["CodeSystem"] = Field(
        description=None,
        default="CodeSystem",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "concept",
            "property_",
            "filter",
            "count",
            "supplements",
            "content",
            "versionNeeded",
            "compositional",
            "hierarchyMeaning",
            "valueSet",
            "caseSensitive",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_csd_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="concept.exists() implies concept.code.combine(%resource.concept.descendants().concept.code).isDistinct()",
            human="Within a code system definition, all the codes SHALL be unique",
            key="csd-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_csd_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="concept.concept.exists() implies hierarchyMeaning.exists()",
            human="If there is an explicit hierarchy, a hierarchyMeaning should be provided",
            key="csd-2",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_csd_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="concept.where(property.code = 'parent' or property.code = 'child').exists() implies hierarchyMeaning.exists()",
            human="If there is an implicit hierarchy, a hierarchyMeaning should be provided",
            key="csd-3",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_csd_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="CodeSystem.content = 'supplement' implies CodeSystem.supplements.exists()",
            human="If the code system content = supplement, it must nominate what it's a supplement for",
            key="csd-4",
            severity="error",
        )
