# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
    Canonical,
    PositiveInt,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    Period,
    BackboneElement,
    Annotation,
    RelatedArtifact,
    Reference,
    Attachment,
    DomainResource,
)


class CitationSummary(BackboneElement):
    """
    A human-readable display of key concepts to represent the citation.
    """

    style: Optional[CodeableConcept] = Field(
        description="Format for display of the citation summary",
        default=None,
    )
    text: Optional[Markdown] = Field(
        description="The human-readable display of the citation summary",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "style",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationClassification(BackboneElement):
    """
    The assignment to an organizing scheme.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of classifier (e.g. publication type, keyword)",
        default=None,
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="The specific classification value",
        default=None,
    )

    @field_validator(
        *(
            "classifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationStatusDate(BackboneElement):
    """
    The state or status of the citation record paired with an effective date or period for that state.
    """

    activity: Optional[CodeableConcept] = Field(
        description="Classification of the status",
        default=None,
    )
    actual: Optional[Boolean] = Field(
        description="Either occurred or expected",
        default=None,
    )
    actual_ext: Optional[Element] = Field(
        description="Placeholder element for actual extensions",
        default=None,
        alias="_actual",
    )
    period: Optional[Period] = Field(
        description="When the status started and/or ended",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "actual",
            "activity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactVersion(BackboneElement):
    """
    The defined version of the cited artifact.
    """

    value: Optional[String] = Field(
        description="The version number or other version identifier",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    baseCitation: Optional[Reference] = Field(
        description="Citation for the main version of the cited artifact",
        default=None,
    )

    @field_validator(
        *(
            "baseCitation",
            "value",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactStatusDate(BackboneElement):
    """
    An effective date or period, historical or future, actual or expected, for a status of the cited artifact.
    """

    activity: Optional[CodeableConcept] = Field(
        description="Classification of the status",
        default=None,
    )
    actual: Optional[Boolean] = Field(
        description="Either occurred or expected",
        default=None,
    )
    actual_ext: Optional[Element] = Field(
        description="Placeholder element for actual extensions",
        default=None,
        alias="_actual",
    )
    period: Optional[Period] = Field(
        description="When the status started and/or ended",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "actual",
            "activity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactTitle(BackboneElement):
    """
    The title details of the article or artifact.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="The kind of title",
        default=None,
    )
    language: Optional[CodeableConcept] = Field(
        description="Used to express the specific language",
        default=None,
    )
    text: Optional[Markdown] = Field(
        description="The title of the article or artifact",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "language",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactAbstract(BackboneElement):
    """
    The abstract may be used to convey article-contained abstracts, externally-created abstracts, or other descriptive summaries.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of abstract",
        default=None,
    )
    language: Optional[CodeableConcept] = Field(
        description="Used to express the specific language",
        default=None,
    )
    text: Optional[Markdown] = Field(
        description="Abstract content",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    copyright: Optional[Markdown] = Field(
        description="Copyright notice for the abstract",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )

    @field_validator(
        *(
            "copyright",
            "text",
            "language",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactPart(BackboneElement):
    """
    The component of the article or artifact.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of component",
        default=None,
    )
    value: Optional[String] = Field(
        description="The specification of the component",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    baseCitation: Optional[Reference] = Field(
        description="The citation for the full article or artifact",
        default=None,
    )

    @field_validator(
        *(
            "baseCitation",
            "value",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactRelatesTo(BackboneElement):
    """
    The artifact related to the cited artifact.
    """

    type: Optional[Code] = Field(
        description="documentation | justification | citation | predecessor | successor | derived-from | depends-on | composed-of | part-of | amends | amended-with | appends | appended-with | cites | cited-by | comments-on | comment-in | contains | contained-in | corrects | correction-in | replaces | replaced-with | retracts | retracted-by | signs | similar-to | supports | supported-with | transforms | transformed-into | transformed-with | documents | specification-of | created-with | cite-as | reprint | reprint-of",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="Additional classifiers",
        default=None,
    )
    label: Optional[String] = Field(
        description="Short label",
        default=None,
    )
    label_ext: Optional[Element] = Field(
        description="Placeholder element for label extensions",
        default=None,
        alias="_label",
    )
    display: Optional[String] = Field(
        description="Brief description of the related artifact",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    citation: Optional[Markdown] = Field(
        description="Bibliographic citation for the artifact",
        default=None,
    )
    citation_ext: Optional[Element] = Field(
        description="Placeholder element for citation extensions",
        default=None,
        alias="_citation",
    )
    document: Optional[Attachment] = Field(
        description="What document is being referenced",
        default=None,
    )
    resource: Optional[Canonical] = Field(
        description="What artifact is being referenced",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    resourceReference: Optional[Reference] = Field(
        description="What artifact, if not a conformance resource",
        default=None,
    )

    @field_validator(
        *(
            "resourceReference",
            "resource",
            "document",
            "citation",
            "display",
            "label",
            "classifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactPublicationFormPublishedIn(BackboneElement):
    """
    The collection the cited article or artifact is published in.
    """

    type: Optional[CodeableConcept] = Field(
        description="Kind of container (e.g. Periodical, database, or book)",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Journal identifiers include ISSN, ISO Abbreviation and NLMuniqueID; Book identifiers include ISBN",
        default=None,
    )
    title: Optional[String] = Field(
        description="Name of the database or title of the book or journal",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    publisher: Optional[Reference] = Field(
        description="Name of or resource describing the publisher",
        default=None,
    )
    publisherLocation: Optional[String] = Field(
        description="Geographic location of the publisher",
        default=None,
    )
    publisherLocation_ext: Optional[Element] = Field(
        description="Placeholder element for publisherLocation extensions",
        default=None,
        alias="_publisherLocation",
    )

    @field_validator(
        *(
            "publisherLocation",
            "publisher",
            "title",
            "identifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactPublicationForm(BackboneElement):
    """
    If multiple, used to represent alternative forms of the article that are not separate citations.
    """

    publishedIn: Optional[CitationCitedArtifactPublicationFormPublishedIn] = Field(
        description="The collection the cited article or artifact is published in",
        default=None,
    )
    citedMedium: Optional[CodeableConcept] = Field(
        description="Internet or Print",
        default=None,
    )
    volume: Optional[String] = Field(
        description="Volume number of journal or other collection in which the article is published",
        default=None,
    )
    volume_ext: Optional[Element] = Field(
        description="Placeholder element for volume extensions",
        default=None,
        alias="_volume",
    )
    issue: Optional[String] = Field(
        description="Issue, part or supplement of journal or other collection in which the article is published",
        default=None,
    )
    issue_ext: Optional[Element] = Field(
        description="Placeholder element for issue extensions",
        default=None,
        alias="_issue",
    )
    articleDate: Optional[DateTime] = Field(
        description="The date the article was added to the database, or the date the article was released",
        default=None,
    )
    articleDate_ext: Optional[Element] = Field(
        description="Placeholder element for articleDate extensions",
        default=None,
        alias="_articleDate",
    )
    publicationDateText: Optional[String] = Field(
        description="Text representation of the date on which the issue of the cited artifact was published",
        default=None,
    )
    publicationDateText_ext: Optional[Element] = Field(
        description="Placeholder element for publicationDateText extensions",
        default=None,
        alias="_publicationDateText",
    )
    publicationDateSeason: Optional[String] = Field(
        description="Season in which the cited artifact was published",
        default=None,
    )
    publicationDateSeason_ext: Optional[Element] = Field(
        description="Placeholder element for publicationDateSeason extensions",
        default=None,
        alias="_publicationDateSeason",
    )
    lastRevisionDate: Optional[DateTime] = Field(
        description="The date the article was last revised or updated in the database",
        default=None,
    )
    lastRevisionDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastRevisionDate extensions",
        default=None,
        alias="_lastRevisionDate",
    )
    language: Optional[List[CodeableConcept]] = Field(
        description="Language(s) in which this form of the article is published",
        default=None,
    )
    accessionNumber: Optional[String] = Field(
        description="Entry number or identifier for inclusion in a database",
        default=None,
    )
    accessionNumber_ext: Optional[Element] = Field(
        description="Placeholder element for accessionNumber extensions",
        default=None,
        alias="_accessionNumber",
    )
    pageString: Optional[String] = Field(
        description="Used for full display of pagination",
        default=None,
    )
    pageString_ext: Optional[Element] = Field(
        description="Placeholder element for pageString extensions",
        default=None,
        alias="_pageString",
    )
    firstPage: Optional[String] = Field(
        description="Used for isolated representation of first page",
        default=None,
    )
    firstPage_ext: Optional[Element] = Field(
        description="Placeholder element for firstPage extensions",
        default=None,
        alias="_firstPage",
    )
    lastPage: Optional[String] = Field(
        description="Used for isolated representation of last page",
        default=None,
    )
    lastPage_ext: Optional[Element] = Field(
        description="Placeholder element for lastPage extensions",
        default=None,
        alias="_lastPage",
    )
    pageCount: Optional[String] = Field(
        description="Number of pages or screens",
        default=None,
    )
    pageCount_ext: Optional[Element] = Field(
        description="Placeholder element for pageCount extensions",
        default=None,
        alias="_pageCount",
    )
    copyright: Optional[Markdown] = Field(
        description="Copyright notice for the full article or artifact",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )

    @field_validator(
        *(
            "copyright",
            "pageCount",
            "lastPage",
            "firstPage",
            "pageString",
            "accessionNumber",
            "language",
            "lastRevisionDate",
            "publicationDateSeason",
            "publicationDateText",
            "articleDate",
            "issue",
            "volume",
            "citedMedium",
            "publishedIn",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactWebLocation(BackboneElement):
    """
    Used for any URL for the article or artifact cited.
    """

    classifier: Optional[List[CodeableConcept]] = Field(
        description="Code the reason for different URLs, e.g. abstract and full-text",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="The specific URL",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "classifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactClassification(BackboneElement):
    """
    The assignment to an organizing scheme.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of classifier (e.g. publication type, keyword)",
        default=None,
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="The specific classification value",
        default=None,
    )
    artifactAssessment: Optional[List[Reference]] = Field(
        description="Complex or externally created classification",
        default=None,
    )

    @field_validator(
        *(
            "artifactAssessment",
            "classifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorshipEntryContributionInstance(BackboneElement):
    """
    Contributions with accounting for time or number.
    """

    type: Optional[CodeableConcept] = Field(
        description="The specific contribution",
        default=None,
    )
    time: Optional[DateTime] = Field(
        description="The time that the contribution was made",
        default=None,
    )
    time_ext: Optional[Element] = Field(
        description="Placeholder element for time extensions",
        default=None,
        alias="_time",
    )

    @field_validator(
        *(
            "time",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorshipEntry(BackboneElement):
    """
    An individual entity named as a contributor, for example in the author list or contributor list.
    """

    contributor: Optional[Reference] = Field(
        description="The identity of the individual contributor",
        default=None,
    )
    forenameInitials: Optional[String] = Field(
        description="For citation styles that use initials",
        default=None,
    )
    forenameInitials_ext: Optional[Element] = Field(
        description="Placeholder element for forenameInitials extensions",
        default=None,
        alias="_forenameInitials",
    )
    affiliation: Optional[List[Reference]] = Field(
        description="Organizational affiliation",
        default=None,
    )
    contributionType: Optional[List[CodeableConcept]] = Field(
        description="The specific contribution",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="The role of the contributor (e.g. author, editor, reviewer, funder)",
        default=None,
    )
    contributionInstance: Optional[
        List[CitationCitedArtifactContributorshipEntryContributionInstance]
    ] = Field(
        description="Contributions with accounting for time or number",
        default=None,
    )
    correspondingContact: Optional[Boolean] = Field(
        description="Whether the contributor is the corresponding contributor for the role",
        default=None,
    )
    correspondingContact_ext: Optional[Element] = Field(
        description="Placeholder element for correspondingContact extensions",
        default=None,
        alias="_correspondingContact",
    )
    rankingOrder: Optional[PositiveInt] = Field(
        description="Ranked order of contribution",
        default=None,
    )
    rankingOrder_ext: Optional[Element] = Field(
        description="Placeholder element for rankingOrder extensions",
        default=None,
        alias="_rankingOrder",
    )

    @field_validator(
        *(
            "rankingOrder",
            "correspondingContact",
            "contributionInstance",
            "role",
            "contributionType",
            "affiliation",
            "forenameInitials",
            "contributor",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorshipSummary(BackboneElement):
    """
    Used to record a display of the author/contributor list without separate data element for each list member.
    """

    type: Optional[CodeableConcept] = Field(
        description="Such as author list, contributorship statement, funding statement, acknowledgements statement, or conflicts of interest statement",
        default=None,
    )
    style: Optional[CodeableConcept] = Field(
        description="The format for the display string",
        default=None,
    )
    source: Optional[CodeableConcept] = Field(
        description="Used to code the producer or rule for creating the display string",
        default=None,
    )
    value: Optional[Markdown] = Field(
        description="The display string for the author list, contributor list, or contributorship statement",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "source",
            "style",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorship(BackboneElement):
    """
    This element is used to list authors and other contributors, their contact information, specific contributions, and summary statements.
    """

    complete: Optional[Boolean] = Field(
        description="Indicates if the list includes all authors and/or contributors",
        default=None,
    )
    complete_ext: Optional[Element] = Field(
        description="Placeholder element for complete extensions",
        default=None,
        alias="_complete",
    )
    entry: Optional[List[CitationCitedArtifactContributorshipEntry]] = Field(
        description="An individual entity named as a contributor",
        default=None,
    )
    summary: Optional[List[CitationCitedArtifactContributorshipSummary]] = Field(
        description="Used to record a display of the author/contributor list without separate data element for each list member",
        default=None,
    )

    @field_validator(
        *(
            "summary",
            "entry",
            "complete",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifact(BackboneElement):
    """
    The article or artifact being described.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="Unique identifier. May include DOI, PMID, PMCID, etc",
        default=None,
    )
    relatedIdentifier: Optional[List[Identifier]] = Field(
        description="Identifier not unique to the cited artifact. May include trial registry identifiers",
        default=None,
    )
    dateAccessed: Optional[DateTime] = Field(
        description="When the cited artifact was accessed",
        default=None,
    )
    dateAccessed_ext: Optional[Element] = Field(
        description="Placeholder element for dateAccessed extensions",
        default=None,
        alias="_dateAccessed",
    )
    version: Optional[CitationCitedArtifactVersion] = Field(
        description="The defined version of the cited artifact",
        default=None,
    )
    currentState: Optional[List[CodeableConcept]] = Field(
        description="The status of the cited artifact",
        default=None,
    )
    statusDate: Optional[List[CitationCitedArtifactStatusDate]] = Field(
        description="An effective date or period for a status of the cited artifact",
        default=None,
    )
    title: Optional[List[CitationCitedArtifactTitle]] = Field(
        description="The title details of the article or artifact",
        default=None,
    )
    abstract: Optional[List[CitationCitedArtifactAbstract]] = Field(
        description="Summary of the article or artifact",
        default=None,
    )
    part: Optional[CitationCitedArtifactPart] = Field(
        description="The component of the article or artifact",
        default=None,
    )
    relatesTo: Optional[List[CitationCitedArtifactRelatesTo]] = Field(
        description="The artifact related to the cited artifact",
        default=None,
    )
    publicationForm: Optional[List[CitationCitedArtifactPublicationForm]] = Field(
        description="If multiple, used to represent alternative forms of the article that are not separate citations",
        default=None,
    )
    webLocation: Optional[List[CitationCitedArtifactWebLocation]] = Field(
        description="Used for any URL for the article or artifact cited",
        default=None,
    )
    classification: Optional[List[CitationCitedArtifactClassification]] = Field(
        description="The assignment to an organizing scheme",
        default=None,
    )
    contributorship: Optional[CitationCitedArtifactContributorship] = Field(
        description="Attribution of authors and other contributors",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Any additional information or content for the article or artifact",
        default=None,
    )

    @field_validator(
        *(
            "note",
            "contributorship",
            "classification",
            "webLocation",
            "publicationForm",
            "relatesTo",
            "part",
            "abstract",
            "title",
            "statusDate",
            "currentState",
            "version",
            "dateAccessed",
            "relatedIdentifier",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Citation(DomainResource):
    """
    The Citation Resource enables reference to any knowledge artifact for purposes of identification and attribution. The Citation Resource supports existing reference structures and developing publication practices such as versioning, expressing complex contributorship roles, and referencing computable resources.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Citation"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this citation record, represented as a globally unique URI",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier for the citation record itself",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the citation record",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this citation record (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this citation record (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="The publisher of the citation record, not the publisher of the article or artifact being cited",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher of the citation record",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the citation",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the citation record content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for citation record (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this citation is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions for the citation record, not for the cited artifact",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s) for the ciation record, not for the cited artifact",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    approvalDate: Optional[Date] = Field(
        description="When the citation record was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the citation record was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the citation record is expected to be used",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the citation record",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the citation record",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the citation record",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the citation record",
        default=None,
    )
    summary: Optional[List[CitationSummary]] = Field(
        description="A human-readable display of key concepts to represent the citation",
        default=None,
    )
    classification: Optional[List[CitationClassification]] = Field(
        description="The assignment to an organizing scheme",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Used for general notes and annotations not coded elsewhere",
        default=None,
    )
    currentState: Optional[List[CodeableConcept]] = Field(
        description="The status of the citation record",
        default=None,
    )
    statusDate: Optional[List[CitationStatusDate]] = Field(
        description="An effective date or period for a status of the citation record",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Artifact related to the citation record",
        default=None,
    )
    citedArtifact: Optional[CitationCitedArtifact] = Field(
        description="The article or artifact being described",
        default=None,
    )
    resourceType: Literal["Citation"] = Field(
        description=None,
        default="Citation",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "citedArtifact",
            "relatedArtifact",
            "statusDate",
            "currentState",
            "note",
            "classification",
            "summary",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )
