# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Timing,
    CodeableReference,
    ContactPoint,
    Annotation,
    DomainResource,
)


class CareTeamParticipant(BackboneElement):
    """
    Identifies all people and organizations who are expected to be involved in the care team.
    """

    role: Optional[CodeableConcept] = Field(
        description="Type of involvement",
        default=None,
    )
    member: Optional[Reference] = Field(
        description="Who is involved",
        default=None,
    )
    onBehalfOf: Optional[Reference] = Field(
        description="Organization of the practitioner",
        default=None,
    )
    coveragePeriod: Optional[Period] = Field(
        description="When the member is generally available within this care team",
        default=None,
    )
    coverageTiming: Optional[Timing] = Field(
        description="When the member is generally available within this care team",
        default=None,
    )

    @property
    def coverage(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="coverage",
        )

    @field_validator(
        *(
            "onBehalfOf",
            "member",
            "role",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def coverage_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Period, Timing],
            field_name_base="coverage",
            required=False,
        )


class CareTeam(DomainResource):
    """
    The Care Team includes all the people and organizations who plan to participate in the coordination and delivery of care.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/CareTeam"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Ids for this team",
        default=None,
    )
    status: Optional[Code] = Field(
        description="proposed | active | suspended | inactive | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Type of team",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name of the team, such as crisis assessment team",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    subject: Optional[Reference] = Field(
        description="Who care team is for",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Time period team covers",
        default=None,
    )
    participant: Optional[List[CareTeamParticipant]] = Field(
        description="Members of the team",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Why the care team exists",
        default=None,
    )
    managingOrganization: Optional[List[Reference]] = Field(
        description="Organization responsible for the care team",
        default=None,
    )
    telecom: Optional[List[ContactPoint]] = Field(
        description="A contact detail for the care team (that applies to all members)",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments made about the CareTeam",
        default=None,
    )
    resourceType: Literal["CareTeam"] = Field(
        description=None,
        default="CareTeam",
    )

    @field_validator(
        *(
            "note",
            "telecom",
            "managingOrganization",
            "reason",
            "participant",
            "period",
            "subject",
            "name",
            "category",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("participant",), mode="after", check_fields=None)
    @classmethod
    def FHIR_ctm_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="onBehalfOf.exists() implies (member.resolve() is Practitioner)",
            human="CareTeam.participant.onBehalfOf can only be populated when CareTeam.participant.member is a Practitioner",
            key="ctm-1",
            severity="error",
        )

    @field_validator(*("participant",), mode="after", check_fields=None)
    @classmethod
    def FHIR_ctm_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="role.exists() or member.exists()",
            human="CareTeam.participant.role or CareTeam.participant.member exists",
            key="ctm-2",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
