# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    BackboneElement,
    CodeableReference,
    Quantity,
    Reference,
    Attachment,
    DomainResource,
)


class BodyStructureIncludedStructureBodyLandmarkOrientationDistanceFromLandmark(
    BackboneElement
):
    """
    The distance in centimeters a certain observation is made from a body landmark.
    """

    device: Optional[List[CodeableReference]] = Field(
        description="Measurement device",
        default=None,
    )
    value: Optional[List[Quantity]] = Field(
        description="Measured distance from body landmark",
        default=None,
    )

    @field_validator(
        *(
            "value",
            "device",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BodyStructureIncludedStructureBodyLandmarkOrientation(BackboneElement):
    """
    Body locations in relation to a specific body landmark (tatoo, scar, other body structure).
    """

    landmarkDescription: Optional[List[CodeableConcept]] = Field(
        description="Body ]andmark description",
        default=None,
    )
    clockFacePosition: Optional[List[CodeableConcept]] = Field(
        description="Clockface orientation",
        default=None,
    )
    distanceFromLandmark: Optional[
        List[BodyStructureIncludedStructureBodyLandmarkOrientationDistanceFromLandmark]
    ] = Field(
        description="Landmark relative location",
        default=None,
    )
    surfaceOrientation: Optional[List[CodeableConcept]] = Field(
        description="Relative landmark surface orientation",
        default=None,
    )

    @field_validator(
        *(
            "surfaceOrientation",
            "distanceFromLandmark",
            "clockFacePosition",
            "landmarkDescription",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BodyStructureIncludedStructure(BackboneElement):
    """
    The anatomical location(s) or region(s) of the specimen, lesion, or body structure.
    """

    structure: Optional[CodeableConcept] = Field(
        description="Code that represents the included structure",
        default=None,
    )
    laterality: Optional[CodeableConcept] = Field(
        description="Code that represents the included structure laterality",
        default=None,
    )
    bodyLandmarkOrientation: Optional[
        List[BodyStructureIncludedStructureBodyLandmarkOrientation]
    ] = Field(
        description="Landmark relative location",
        default=None,
    )
    spatialReference: Optional[List[Reference]] = Field(
        description="Cartesian reference for structure",
        default=None,
    )
    qualifier: Optional[List[CodeableConcept]] = Field(
        description="Code that represents the included structure qualifier",
        default=None,
    )

    @field_validator(
        *(
            "qualifier",
            "spatialReference",
            "bodyLandmarkOrientation",
            "laterality",
            "structure",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BodyStructureIncludedStructureBodyLandmarkOrientationDistanceFromLandmark(
    BackboneElement
):
    """
    The distance in centimeters a certain observation is made from a body landmark.
    """

    device: Optional[List[CodeableReference]] = Field(
        description="Measurement device",
        default=None,
    )
    value: Optional[List[Quantity]] = Field(
        description="Measured distance from body landmark",
        default=None,
    )

    @field_validator(
        *(
            "value",
            "device",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BodyStructureIncludedStructureBodyLandmarkOrientation(BackboneElement):
    """
    Body locations in relation to a specific body landmark (tatoo, scar, other body structure).
    """

    landmarkDescription: Optional[List[CodeableConcept]] = Field(
        description="Body ]andmark description",
        default=None,
    )
    clockFacePosition: Optional[List[CodeableConcept]] = Field(
        description="Clockface orientation",
        default=None,
    )
    distanceFromLandmark: Optional[
        List[BodyStructureIncludedStructureBodyLandmarkOrientationDistanceFromLandmark]
    ] = Field(
        description="Landmark relative location",
        default=None,
    )
    surfaceOrientation: Optional[List[CodeableConcept]] = Field(
        description="Relative landmark surface orientation",
        default=None,
    )

    @field_validator(
        *(
            "surfaceOrientation",
            "distanceFromLandmark",
            "clockFacePosition",
            "landmarkDescription",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BodyStructureExcludedStructure(BackboneElement):
    """
    The anatomical location(s) or region(s) not occupied or represented by the specimen, lesion, or body structure.
    """

    structure: Optional[CodeableConcept] = Field(
        description="Code that represents the included structure",
        default=None,
    )
    laterality: Optional[CodeableConcept] = Field(
        description="Code that represents the included structure laterality",
        default=None,
    )
    bodyLandmarkOrientation: Optional[
        List[BodyStructureIncludedStructureBodyLandmarkOrientation]
    ] = Field(
        description="Landmark relative location",
        default=None,
    )
    spatialReference: Optional[List[Reference]] = Field(
        description="Cartesian reference for structure",
        default=None,
    )
    qualifier: Optional[List[CodeableConcept]] = Field(
        description="Code that represents the included structure qualifier",
        default=None,
    )

    @field_validator(
        *(
            "qualifier",
            "spatialReference",
            "bodyLandmarkOrientation",
            "laterality",
            "structure",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BodyStructure(DomainResource):
    """
    Record details about an anatomical structure.  This resource may be used when a coded concept does not provide the necessary detail needed for the use case.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/BodyStructure"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Bodystructure identifier",
        default=None,
    )
    active: Optional[Boolean] = Field(
        description="Whether this record is in active use",
        default=None,
    )
    active_ext: Optional[Element] = Field(
        description="Placeholder element for active extensions",
        default=None,
        alias="_active",
    )
    morphology: Optional[CodeableConcept] = Field(
        description="Kind of Structure",
        default=None,
    )
    includedStructure: Optional[List[BodyStructureIncludedStructure]] = Field(
        description="Included anatomic location(s)",
        default=None,
    )
    excludedStructure: Optional[List[BodyStructureExcludedStructure]] = Field(
        description="Excluded anatomic locations(s)",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Text description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    image: Optional[List[Attachment]] = Field(
        description="Attached images",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="Who this is about",
        default=None,
    )
    resourceType: Literal["BodyStructure"] = Field(
        description=None,
        default="BodyStructure",
    )

    @field_validator(
        *(
            "patient",
            "image",
            "description",
            "excludedStructure",
            "includedStructure",
            "morphology",
            "active",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
