# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Boolean,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Coding,
    CodeableConcept,
    Reference,
    Identifier,
    BackboneElement,
    Period,
    Range,
    Quantity,
    Ratio,
    Attachment,
    DomainResource,
)


class BiologicallyDerivedProductCollection(BackboneElement):
    """
    How this product was collected.
    """

    collector: Optional[Reference] = Field(
        description="Individual performing collection",
        default=None,
    )
    source: Optional[Reference] = Field(
        description="The patient who underwent the medical procedure to collect the product or the organization that facilitated the collection",
        default=None,
    )
    collectedDateTime: Optional[DateTime] = Field(
        description="Time of product collection",
        default=None,
    )
    collectedDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for collectedDateTime extensions",
        default=None,
        alias="_collectedDateTime",
    )
    collectedPeriod: Optional[Period] = Field(
        description="Time of product collection",
        default=None,
    )

    @property
    def collected(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="collected",
        )

    @field_validator(
        *(
            "source",
            "collector",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def collected_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="collected",
            required=False,
        )


class BiologicallyDerivedProductProperty(BackboneElement):
    """
    A property that is specific to this BiologicallyDerviedProduct instance.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code that specifies the property",
        default=None,
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Property values",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Property values",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Property values",
        default=None,
    )
    valuePeriod: Optional[Period] = Field(
        description="Property values",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Property values",
        default=None,
    )
    valueRange: Optional[Range] = Field(
        description="Property values",
        default=None,
    )
    valueRatio: Optional[Ratio] = Field(
        description="Property values",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Property values",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="Property values",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Boolean,
                Integer,
                CodeableConcept,
                Period,
                Quantity,
                Range,
                Ratio,
                String,
                Attachment,
            ],
            field_name_base="value",
            required=True,
        )


class BiologicallyDerivedProduct(DomainResource):
    """
    A biological material originating from a biological entity intended to be transplanted or infused into another (possibly the same) biological entity.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/BiologicallyDerivedProduct"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    productCategory: Optional[Coding] = Field(
        description="organ | tissue | fluid | cells | biologicalAgent",
        default=None,
    )
    productCode: Optional[CodeableConcept] = Field(
        description="A code that identifies the kind of this biologically derived product",
        default=None,
    )
    parent: Optional[List[Reference]] = Field(
        description="The parent biologically-derived product",
        default=None,
    )
    request: Optional[List[Reference]] = Field(
        description="Request to obtain and/or infuse this product",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Instance identifier",
        default=None,
    )
    biologicalSourceEvent: Optional[Identifier] = Field(
        description="An identifier that supports traceability to the event during which material in this product from one or more biological entities was obtained or pooled",
        default=None,
    )
    processingFacility: Optional[List[Reference]] = Field(
        description="Processing facilities responsible for the labeling and distribution of this biologically derived product",
        default=None,
    )
    division: Optional[String] = Field(
        description="A unique identifier for an aliquot of a product",
        default=None,
    )
    division_ext: Optional[Element] = Field(
        description="Placeholder element for division extensions",
        default=None,
        alias="_division",
    )
    productStatus: Optional[Coding] = Field(
        description="available | unavailable",
        default=None,
    )
    expirationDate: Optional[DateTime] = Field(
        description="Date, and where relevant time, of expiration",
        default=None,
    )
    expirationDate_ext: Optional[Element] = Field(
        description="Placeholder element for expirationDate extensions",
        default=None,
        alias="_expirationDate",
    )
    collection: Optional[BiologicallyDerivedProductCollection] = Field(
        description="How this product was collected",
        default=None,
    )
    storageTempRequirements: Optional[Range] = Field(
        description="Product storage temperature requirements",
        default=None,
    )
    property_: Optional[List[BiologicallyDerivedProductProperty]] = Field(
        description="A property that is specific to this BiologicallyDerviedProduct instance",
        default=None,
    )
    resourceType: Literal["BiologicallyDerivedProduct"] = Field(
        description=None,
        default="BiologicallyDerivedProduct",
    )

    @field_validator(
        *(
            "property_",
            "storageTempRequirements",
            "collection",
            "expirationDate",
            "productStatus",
            "division",
            "processingFacility",
            "biologicalSourceEvent",
            "identifier",
            "request",
            "parent",
            "productCode",
            "productCategory",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
