# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Instant,
    Boolean,
    Base64Binary,
    Integer,
    Time,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    CodeableConcept,
    Period,
    BackboneElement,
    Coding,
    Reference,
    Quantity,
    Range,
    Ratio,
    DomainResource,
)


class AuditEventOutcome(BackboneElement):
    """
    Indicates whether the event succeeded or failed. A free text descripiton can be given in outcome.text.
    """

    code: Optional[Coding] = Field(
        description="Whether the event succeeded or failed",
        default=None,
    )
    detail: Optional[List[CodeableConcept]] = Field(
        description="Additional outcome detail",
        default=None,
    )

    @field_validator(
        *(
            "detail",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AuditEventAgent(BackboneElement):
    """
    An actor taking an active role in the event or activity that is logged.
    """

    type: Optional[CodeableConcept] = Field(
        description="How agent participated",
        default=None,
    )
    role: Optional[List[CodeableConcept]] = Field(
        description="Agent role in the event",
        default=None,
    )
    who: Optional[Reference] = Field(
        description="Identifier of who",
        default=None,
    )
    requestor: Optional[Boolean] = Field(
        description="Whether user is initiator",
        default=None,
    )
    requestor_ext: Optional[Element] = Field(
        description="Placeholder element for requestor extensions",
        default=None,
        alias="_requestor",
    )
    location: Optional[Reference] = Field(
        description="The agent location when the event occurred",
        default=None,
    )
    policy: Optional[List[Uri]] = Field(
        description="Policy that authorized the agent participation in the event",
        default=None,
    )
    policy_ext: Optional[Element] = Field(
        description="Placeholder element for policy extensions",
        default=None,
        alias="_policy",
    )
    networkReference: Optional[Reference] = Field(
        description="This agent network location for the activity",
        default=None,
    )
    networkUri: Optional[Uri] = Field(
        description="This agent network location for the activity",
        default=None,
    )
    networkUri_ext: Optional[Element] = Field(
        description="Placeholder element for networkUri extensions",
        default=None,
        alias="_networkUri",
    )
    networkString: Optional[String] = Field(
        description="This agent network location for the activity",
        default=None,
    )
    networkString_ext: Optional[Element] = Field(
        description="Placeholder element for networkString extensions",
        default=None,
        alias="_networkString",
    )
    authorization: Optional[List[CodeableConcept]] = Field(
        description="Allowable authorization for this agent",
        default=None,
    )

    @property
    def network(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="network",
        )

    @field_validator(
        *(
            "authorization",
            "policy",
            "location",
            "requestor",
            "who",
            "role",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def network_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, Uri, String],
            field_name_base="network",
            required=False,
        )


class AuditEventSource(BackboneElement):
    """
    The actor that is reporting the event.
    """

    site: Optional[Reference] = Field(
        description="Logical source location within the enterprise",
        default=None,
    )
    observer: Optional[Reference] = Field(
        description="The identity of source detecting the event",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="The type of source where event originated",
        default=None,
    )

    @field_validator(
        *(
            "type",
            "observer",
            "site",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AuditEventEntityDetail(BackboneElement):
    """
    Tagged value pairs for conveying additional information about the entity.
    """

    type: Optional[CodeableConcept] = Field(
        description="Name of the property",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Property value",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Property value",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Property value",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Property value",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Property value",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueRange: Optional[Range] = Field(
        description="Property value",
        default=None,
    )
    valueRatio: Optional[Ratio] = Field(
        description="Property value",
        default=None,
    )
    valueTime: Optional[Time] = Field(
        description="Property value",
        default=None,
    )
    valueTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueTime extensions",
        default=None,
        alias="_valueTime",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Property value",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valuePeriod: Optional[Period] = Field(
        description="Property value",
        default=None,
    )
    valueBase64Binary: Optional[Base64Binary] = Field(
        description="Property value",
        default=None,
    )
    valueBase64Binary_ext: Optional[Element] = Field(
        description="Placeholder element for valueBase64Binary extensions",
        default=None,
        alias="_valueBase64Binary",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Quantity,
                CodeableConcept,
                String,
                Boolean,
                Integer,
                Range,
                Ratio,
                Time,
                DateTime,
                Period,
                Base64Binary,
            ],
            field_name_base="value",
            required=True,
        )


class AuditEventEntityAgent(BackboneElement):
    """
    The entity is attributed to an agent to express the agent's responsibility for that entity in the activity. This is most used to indicate when persistence media (the entity) are used by an agent. For example when importing data from a device, the device would be described in an entity, and the user importing data from that media would be indicated as the entity.agent.
    """

    type: Optional[CodeableConcept] = Field(
        description="How agent participated",
        default=None,
    )
    role: Optional[List[CodeableConcept]] = Field(
        description="Agent role in the event",
        default=None,
    )
    who: Optional[Reference] = Field(
        description="Identifier of who",
        default=None,
    )
    requestor: Optional[Boolean] = Field(
        description="Whether user is initiator",
        default=None,
    )
    requestor_ext: Optional[Element] = Field(
        description="Placeholder element for requestor extensions",
        default=None,
        alias="_requestor",
    )
    location: Optional[Reference] = Field(
        description="The agent location when the event occurred",
        default=None,
    )
    policy: Optional[List[Uri]] = Field(
        description="Policy that authorized the agent participation in the event",
        default=None,
    )
    policy_ext: Optional[Element] = Field(
        description="Placeholder element for policy extensions",
        default=None,
        alias="_policy",
    )
    networkReference: Optional[Reference] = Field(
        description="This agent network location for the activity",
        default=None,
    )
    networkUri: Optional[Uri] = Field(
        description="This agent network location for the activity",
        default=None,
    )
    networkUri_ext: Optional[Element] = Field(
        description="Placeholder element for networkUri extensions",
        default=None,
        alias="_networkUri",
    )
    networkString: Optional[String] = Field(
        description="This agent network location for the activity",
        default=None,
    )
    networkString_ext: Optional[Element] = Field(
        description="Placeholder element for networkString extensions",
        default=None,
        alias="_networkString",
    )
    authorization: Optional[List[CodeableConcept]] = Field(
        description="Allowable authorization for this agent",
        default=None,
    )

    @property
    def network(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="network",
        )

    @field_validator(
        *(
            "authorization",
            "policy",
            "location",
            "requestor",
            "who",
            "role",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def network_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, Uri, String],
            field_name_base="network",
            required=False,
        )


class AuditEventEntity(BackboneElement):
    """
    Specific instances of data or objects that have been accessed.
    """

    what: Optional[Reference] = Field(
        description="Specific instance of resource",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="What role the entity played",
        default=None,
    )
    securityLabel: Optional[List[CodeableConcept]] = Field(
        description="Security labels on the entity",
        default=None,
    )
    query: Optional[Base64Binary] = Field(
        description="Query parameters",
        default=None,
    )
    query_ext: Optional[Element] = Field(
        description="Placeholder element for query extensions",
        default=None,
        alias="_query",
    )
    detail: Optional[List[AuditEventEntityDetail]] = Field(
        description="Additional Information about the entity",
        default=None,
    )
    agent: Optional[List[AuditEventEntityAgent]] = Field(
        description="Entity is attributed to this agent",
        default=None,
    )

    @field_validator(
        *(
            "agent",
            "detail",
            "query",
            "securityLabel",
            "role",
            "what",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AuditEvent(DomainResource):
    """
    A record of an event relevant for purposes such as operations, privacy, security, maintenance, and performance analysis.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/AuditEvent"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Type/identifier of event",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Specific type of event",
        default=None,
    )
    action: Optional[Code] = Field(
        description="Type of action performed during the event",
        default=None,
    )
    action_ext: Optional[Element] = Field(
        description="Placeholder element for action extensions",
        default=None,
        alias="_action",
    )
    severity: Optional[Code] = Field(
        description="emergency | alert | critical | error | warning | notice | informational | debug",
        default=None,
    )
    severity_ext: Optional[Element] = Field(
        description="Placeholder element for severity extensions",
        default=None,
        alias="_severity",
    )
    occurredPeriod: Optional[Period] = Field(
        description="When the activity occurred",
        default=None,
    )
    occurredDateTime: Optional[DateTime] = Field(
        description="When the activity occurred",
        default=None,
    )
    occurredDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurredDateTime extensions",
        default=None,
        alias="_occurredDateTime",
    )
    recorded: Optional[Instant] = Field(
        description="Time when the event was recorded",
        default=None,
    )
    recorded_ext: Optional[Element] = Field(
        description="Placeholder element for recorded extensions",
        default=None,
        alias="_recorded",
    )
    outcome: Optional[AuditEventOutcome] = Field(
        description="Whether the event succeeded or failed",
        default=None,
    )
    authorization: Optional[List[CodeableConcept]] = Field(
        description="Authorization related to the event",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Workflow authorization within which this event occurred",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="The patient is the subject of the data used/created/updated/deleted during the activity",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter within which this event occurred or which the event is tightly associated",
        default=None,
    )
    agent: Optional[List[AuditEventAgent]] = Field(
        description="Actor involved in the event",
        default=None,
    )
    source: Optional[AuditEventSource] = Field(
        description="Audit Event Reporter",
        default=None,
    )
    entity: Optional[List[AuditEventEntity]] = Field(
        description="Data or objects used",
        default=None,
    )
    resourceType: Literal["AuditEvent"] = Field(
        description=None,
        default="AuditEvent",
    )

    @property
    def occurred(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurred",
        )

    @field_validator(
        *(
            "entity",
            "source",
            "agent",
            "encounter",
            "patient",
            "basedOn",
            "authorization",
            "outcome",
            "recorded",
            "severity",
            "action",
            "code",
            "category",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurred_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Period, DateTime],
            field_name_base="occurred",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
