# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Markdown,
    DateTime,
    Date,
    Canonical,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    BackboneElement,
    CodeableConcept,
    Quantity,
    RelatedArtifact,
    DomainResource,
)


class ArtifactAssessmentContent(BackboneElement):
    """
    A component comment, classifier, or rating of the artifact.
    """

    informationType: Optional[Code] = Field(
        description="comment | classifier | rating | container | response | change-request",
        default=None,
    )
    informationType_ext: Optional[Element] = Field(
        description="Placeholder element for informationType extensions",
        default=None,
        alias="_informationType",
    )
    summary: Optional[Markdown] = Field(
        description="Brief summary of the content",
        default=None,
    )
    summary_ext: Optional[Element] = Field(
        description="Placeholder element for summary extensions",
        default=None,
        alias="_summary",
    )
    type: Optional[CodeableConcept] = Field(
        description="What type of content",
        default=None,
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="Rating, classifier, or assessment",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Quantitative rating",
        default=None,
    )
    author: Optional[Reference] = Field(
        description="Who authored the content",
        default=None,
    )
    path: Optional[List[Uri]] = Field(
        description="What the comment is directed to",
        default=None,
    )
    path_ext: Optional[Element] = Field(
        description="Placeholder element for path extensions",
        default=None,
        alias="_path",
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional information",
        default=None,
    )
    freeToShare: Optional[Boolean] = Field(
        description="Acceptable to publicly share the resource content",
        default=None,
    )
    freeToShare_ext: Optional[Element] = Field(
        description="Placeholder element for freeToShare extensions",
        default=None,
        alias="_freeToShare",
    )
    component: Optional[List["ArtifactAssessmentContent"]] = Field(
        description="Contained content",
        default=None,
    )

    @field_validator(
        *(
            "component",
            "freeToShare",
            "relatedArtifact",
            "path",
            "author",
            "quantity",
            "classifier",
            "type",
            "summary",
            "informationType",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ArtifactAssessment(DomainResource):
    """
    This Resource provides one or more comments, classifiers or ratings about a Resource and supports attribution and rights management metadata for the added content.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ArtifactAssessment"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the artifact assessment",
        default=None,
    )
    title: Optional[String] = Field(
        description="A short title for the assessment for use in displaying and selecting",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    citeAsReference: Optional[Reference] = Field(
        description="How to cite the comment or rating",
        default=None,
    )
    citeAsMarkdown: Optional[Markdown] = Field(
        description="How to cite the comment or rating",
        default=None,
    )
    citeAsMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for citeAsMarkdown extensions",
        default=None,
        alias="_citeAsMarkdown",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    approvalDate: Optional[Date] = Field(
        description="When the artifact assessment was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the artifact assessment was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    artifactReference: Optional[Reference] = Field(
        description="The artifact assessed, commented upon or rated",
        default=None,
    )
    artifactCanonical: Optional[Canonical] = Field(
        description="The artifact assessed, commented upon or rated",
        default=None,
    )
    artifactCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for artifactCanonical extensions",
        default=None,
        alias="_artifactCanonical",
    )
    artifactUri: Optional[Uri] = Field(
        description="The artifact assessed, commented upon or rated",
        default=None,
    )
    artifactUri_ext: Optional[Element] = Field(
        description="Placeholder element for artifactUri extensions",
        default=None,
        alias="_artifactUri",
    )
    content: Optional[List[ArtifactAssessmentContent]] = Field(
        description="Comment, classifier, or rating content",
        default=None,
    )
    workflowStatus: Optional[Code] = Field(
        description="submitted | triaged | waiting-for-input | resolved-no-change | resolved-change-required | deferred | duplicate | applied | published | entered-in-error",
        default=None,
    )
    workflowStatus_ext: Optional[Element] = Field(
        description="Placeholder element for workflowStatus extensions",
        default=None,
        alias="_workflowStatus",
    )
    disposition: Optional[Code] = Field(
        description="unresolved | not-persuasive | persuasive | persuasive-with-modification | not-persuasive-with-modification",
        default=None,
    )
    disposition_ext: Optional[Element] = Field(
        description="Placeholder element for disposition extensions",
        default=None,
        alias="_disposition",
    )
    resourceType: Literal["ArtifactAssessment"] = Field(
        description=None,
        default="ArtifactAssessment",
    )

    @property
    def citeAs(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="citeAs",
        )

    @property
    def artifact(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="artifact",
        )

    @field_validator(
        *(
            "disposition",
            "workflowStatus",
            "content",
            "lastReviewDate",
            "approvalDate",
            "copyright",
            "date",
            "title",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def citeAs_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, Markdown],
            field_name_base="citeAs",
            required=False,
        )

    @model_validator(mode="after")
    def artifact_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, Canonical, Uri],
            field_name_base="artifact",
            required=True,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
