# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Instant,
    PositiveInt,
    DateTime,
    Boolean,
    Date,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    CodeableReference,
    Reference,
    VirtualServiceDetail,
    Period,
    Annotation,
    BackboneElement,
    Coding,
    DomainResource,
)


class AppointmentParticipant(BackboneElement):
    """
    List of participants involved in the appointment.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="Role of participant in the appointment",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Participation period of the actor",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="The individual, device, location, or service participating in the appointment",
        default=None,
    )
    required: Optional[Boolean] = Field(
        description="The participant is required to attend (optional when false)",
        default=None,
    )
    required_ext: Optional[Element] = Field(
        description="Placeholder element for required extensions",
        default=None,
        alias="_required",
    )
    status: Optional[Code] = Field(
        description="accepted | declined | tentative | needs-action",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )

    @field_validator(
        *(
            "status",
            "required",
            "actor",
            "period",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AppointmentRecurrenceTemplateWeeklyTemplate(BackboneElement):
    """
    Information about weekly recurring appointments.
    """

    monday: Optional[Boolean] = Field(
        description="Recurs on Mondays",
        default=None,
    )
    monday_ext: Optional[Element] = Field(
        description="Placeholder element for monday extensions",
        default=None,
        alias="_monday",
    )
    tuesday: Optional[Boolean] = Field(
        description="Recurs on Tuesday",
        default=None,
    )
    tuesday_ext: Optional[Element] = Field(
        description="Placeholder element for tuesday extensions",
        default=None,
        alias="_tuesday",
    )
    wednesday: Optional[Boolean] = Field(
        description="Recurs on Wednesday",
        default=None,
    )
    wednesday_ext: Optional[Element] = Field(
        description="Placeholder element for wednesday extensions",
        default=None,
        alias="_wednesday",
    )
    thursday: Optional[Boolean] = Field(
        description="Recurs on Thursday",
        default=None,
    )
    thursday_ext: Optional[Element] = Field(
        description="Placeholder element for thursday extensions",
        default=None,
        alias="_thursday",
    )
    friday: Optional[Boolean] = Field(
        description="Recurs on Friday",
        default=None,
    )
    friday_ext: Optional[Element] = Field(
        description="Placeholder element for friday extensions",
        default=None,
        alias="_friday",
    )
    saturday: Optional[Boolean] = Field(
        description="Recurs on Saturday",
        default=None,
    )
    saturday_ext: Optional[Element] = Field(
        description="Placeholder element for saturday extensions",
        default=None,
        alias="_saturday",
    )
    sunday: Optional[Boolean] = Field(
        description="Recurs on Sunday",
        default=None,
    )
    sunday_ext: Optional[Element] = Field(
        description="Placeholder element for sunday extensions",
        default=None,
        alias="_sunday",
    )
    weekInterval: Optional[PositiveInt] = Field(
        description="Recurs every nth week",
        default=None,
    )
    weekInterval_ext: Optional[Element] = Field(
        description="Placeholder element for weekInterval extensions",
        default=None,
        alias="_weekInterval",
    )

    @field_validator(
        *(
            "weekInterval",
            "sunday",
            "saturday",
            "friday",
            "thursday",
            "wednesday",
            "tuesday",
            "monday",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AppointmentRecurrenceTemplateMonthlyTemplate(BackboneElement):
    """
    Information about monthly recurring appointments.
    """

    dayOfMonth: Optional[PositiveInt] = Field(
        description="Recurs on a specific day of the month",
        default=None,
    )
    dayOfMonth_ext: Optional[Element] = Field(
        description="Placeholder element for dayOfMonth extensions",
        default=None,
        alias="_dayOfMonth",
    )
    nthWeekOfMonth: Optional[Coding] = Field(
        description="Indicates which week of the month the appointment should occur",
        default=None,
    )
    dayOfWeek: Optional[Coding] = Field(
        description="Indicates which day of the week the appointment should occur",
        default=None,
    )
    monthInterval: Optional[PositiveInt] = Field(
        description="Recurs every nth month",
        default=None,
    )
    monthInterval_ext: Optional[Element] = Field(
        description="Placeholder element for monthInterval extensions",
        default=None,
        alias="_monthInterval",
    )

    @field_validator(
        *(
            "monthInterval",
            "dayOfWeek",
            "nthWeekOfMonth",
            "dayOfMonth",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AppointmentRecurrenceTemplateYearlyTemplate(BackboneElement):
    """
    Information about yearly recurring appointments.
    """

    yearInterval: Optional[PositiveInt] = Field(
        description="Recurs every nth year",
        default=None,
    )
    yearInterval_ext: Optional[Element] = Field(
        description="Placeholder element for yearInterval extensions",
        default=None,
        alias="_yearInterval",
    )

    @field_validator(
        *("yearInterval", "modifierExtension", "extension"),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AppointmentRecurrenceTemplate(BackboneElement):
    """
    The details of the recurrence pattern or template that is used to generate recurring appointments.
    """

    timezone: Optional[CodeableConcept] = Field(
        description="The timezone of the occurrences",
        default=None,
    )
    recurrenceType: Optional[CodeableConcept] = Field(
        description="The frequency of the recurrence",
        default=None,
    )
    lastOccurrenceDate: Optional[Date] = Field(
        description="The date when the recurrence should end",
        default=None,
    )
    lastOccurrenceDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastOccurrenceDate extensions",
        default=None,
        alias="_lastOccurrenceDate",
    )
    occurrenceCount: Optional[PositiveInt] = Field(
        description="The number of planned occurrences",
        default=None,
    )
    occurrenceCount_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceCount extensions",
        default=None,
        alias="_occurrenceCount",
    )
    occurrenceDate: Optional[List[Date]] = Field(
        description="Specific dates for a recurring set of appointments (no template)",
        default=None,
    )
    occurrenceDate_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDate extensions",
        default=None,
        alias="_occurrenceDate",
    )
    weeklyTemplate: Optional[AppointmentRecurrenceTemplateWeeklyTemplate] = Field(
        description="Information about weekly recurring appointments",
        default=None,
    )
    monthlyTemplate: Optional[AppointmentRecurrenceTemplateMonthlyTemplate] = Field(
        description="Information about monthly recurring appointments",
        default=None,
    )
    yearlyTemplate: Optional[AppointmentRecurrenceTemplateYearlyTemplate] = Field(
        description="Information about yearly recurring appointments",
        default=None,
    )
    excludingDate: Optional[List[Date]] = Field(
        description="Any dates that should be excluded from the series",
        default=None,
    )
    excludingDate_ext: Optional[Element] = Field(
        description="Placeholder element for excludingDate extensions",
        default=None,
        alias="_excludingDate",
    )
    excludingRecurrenceId: Optional[List[PositiveInt]] = Field(
        description="Any recurrence IDs that should be excluded from the recurrence",
        default=None,
    )
    excludingRecurrenceId_ext: Optional[Element] = Field(
        description="Placeholder element for excludingRecurrenceId extensions",
        default=None,
        alias="_excludingRecurrenceId",
    )

    @field_validator(
        *(
            "excludingRecurrenceId",
            "excludingDate",
            "yearlyTemplate",
            "monthlyTemplate",
            "weeklyTemplate",
            "occurrenceDate",
            "occurrenceCount",
            "lastOccurrenceDate",
            "recurrenceType",
            "timezone",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Appointment(DomainResource):
    """
    A booking of a healthcare event among patient(s), practitioner(s), related person(s) and/or device(s) for a specific date/time. This may result in one or more Encounter(s).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Appointment"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Ids for this item",
        default=None,
    )
    status: Optional[Code] = Field(
        description="proposed | pending | booked | arrived | fulfilled | cancelled | noshow | entered-in-error | checked-in | waitlist",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    cancellationReason: Optional[CodeableConcept] = Field(
        description="The coded reason for the appointment being cancelled",
        default=None,
    )
    class_: Optional[List[CodeableConcept]] = Field(
        description="Classification when becoming an encounter",
        default=None,
    )
    serviceCategory: Optional[List[CodeableConcept]] = Field(
        description="A broad categorization of the service that is to be performed during this appointment",
        default=None,
    )
    serviceType: Optional[List[CodeableReference]] = Field(
        description="The specific service that is to be performed during this appointment",
        default=None,
    )
    specialty: Optional[List[CodeableConcept]] = Field(
        description="The specialty of a practitioner that would be required to perform the service requested in this appointment",
        default=None,
    )
    appointmentType: Optional[CodeableConcept] = Field(
        description="The style of appointment or patient that has been booked in the slot (not service type)",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Reason this appointment is scheduled",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="Used to make informed decisions if needing to re-prioritize",
        default=None,
    )
    description: Optional[String] = Field(
        description="Shown on a subject line in a meeting request, or appointment list",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    replaces: Optional[List[Reference]] = Field(
        description="Appointment replaced by this Appointment",
        default=None,
    )
    virtualService: Optional[List[VirtualServiceDetail]] = Field(
        description="Connection details of a virtual service (e.g. conference call)",
        default=None,
    )
    supportingInformation: Optional[List[Reference]] = Field(
        description="Additional information to support the appointment",
        default=None,
    )
    previousAppointment: Optional[Reference] = Field(
        description="The previous appointment in a series",
        default=None,
    )
    originatingAppointment: Optional[Reference] = Field(
        description="The originating appointment in a recurring set of appointments",
        default=None,
    )
    start: Optional[Instant] = Field(
        description="When appointment is to take place",
        default=None,
    )
    start_ext: Optional[Element] = Field(
        description="Placeholder element for start extensions",
        default=None,
        alias="_start",
    )
    end: Optional[Instant] = Field(
        description="When appointment is to conclude",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )
    minutesDuration: Optional[PositiveInt] = Field(
        description="Can be less than start/end (e.g. estimate)",
        default=None,
    )
    minutesDuration_ext: Optional[Element] = Field(
        description="Placeholder element for minutesDuration extensions",
        default=None,
        alias="_minutesDuration",
    )
    requestedPeriod: Optional[List[Period]] = Field(
        description="Potential date/time interval(s) requested to allocate the appointment within",
        default=None,
    )
    slot: Optional[List[Reference]] = Field(
        description="The slots that this appointment is filling",
        default=None,
    )
    account: Optional[List[Reference]] = Field(
        description="The set of accounts that may be used for billing for this Appointment",
        default=None,
    )
    created: Optional[DateTime] = Field(
        description="The date that this appointment was initially created",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    cancellationDate: Optional[DateTime] = Field(
        description="When the appointment was cancelled",
        default=None,
    )
    cancellationDate_ext: Optional[Element] = Field(
        description="Placeholder element for cancellationDate extensions",
        default=None,
        alias="_cancellationDate",
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional comments",
        default=None,
    )
    patientInstruction: Optional[List[CodeableReference]] = Field(
        description="Detailed information and instructions for the patient",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="The request this appointment is allocated to assess",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="The patient or group associated with the appointment",
        default=None,
    )
    participant: Optional[List[AppointmentParticipant]] = Field(
        description="Participants involved in appointment",
        default=None,
    )
    recurrenceId: Optional[PositiveInt] = Field(
        description="The sequence number in the recurrence",
        default=None,
    )
    recurrenceId_ext: Optional[Element] = Field(
        description="Placeholder element for recurrenceId extensions",
        default=None,
        alias="_recurrenceId",
    )
    occurrenceChanged: Optional[Boolean] = Field(
        description="Indicates that this appointment varies from a recurrence pattern",
        default=None,
    )
    occurrenceChanged_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceChanged extensions",
        default=None,
        alias="_occurrenceChanged",
    )
    recurrenceTemplate: Optional[List[AppointmentRecurrenceTemplate]] = Field(
        description="Details of the recurrence pattern/template used to generate occurrences",
        default=None,
    )
    resourceType: Literal["Appointment"] = Field(
        description=None,
        default="Appointment",
    )

    @field_validator(
        *(
            "recurrenceTemplate",
            "occurrenceChanged",
            "recurrenceId",
            "participant",
            "subject",
            "basedOn",
            "patientInstruction",
            "note",
            "cancellationDate",
            "created",
            "account",
            "slot",
            "requestedPeriod",
            "minutesDuration",
            "end",
            "start",
            "originatingAppointment",
            "previousAppointment",
            "supportingInformation",
            "virtualService",
            "replaces",
            "description",
            "priority",
            "reason",
            "appointmentType",
            "specialty",
            "serviceType",
            "serviceCategory",
            "class_",
            "cancellationReason",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("participant",), mode="after", check_fields=None)
    @classmethod
    def FHIR_app_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type.exists() or actor.exists()",
            human="Either the type or actor on the participant SHALL be specified",
            key="app-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="start.exists() = end.exists()",
            human="Either start and end are specified, or neither",
            key="app-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(start.exists() and end.exists()) or (status in ('proposed' | 'cancelled' | 'waitlist'))",
            human="Only proposed or cancelled appointments can be missing start/end dates",
            key="app-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="cancellationReason.exists() implies (status='noshow' or status='cancelled')",
            human="Cancellation reason is only used for appointments that have been cancelled, or noshow",
            key="app-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="start.exists() implies start <= end",
            human="The start must be less than or equal to the end",
            key="app-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="originatingAppointment.exists().not() or recurrenceTemplate.exists().not()",
            human="An appointment may have an originatingAppointment or recurrenceTemplate, but not both",
            key="app-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_app_7_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="cancellationDate.exists() implies (status='noshow' or status='cancelled')",
            human="Cancellation date is only used for appointments that have been cancelled, or noshow",
            key="app-7",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
