# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    BackboneElement,
    Range,
    Annotation,
    DomainResource,
)


class RiskAssessmentPrediction(BackboneElement):
    """
    Describes the expected outcome for the subject.
    """

    outcome: Optional[CodeableConcept] = Field(
        description="Possible outcome for the subject",
        default=None,
    )
    probabilityDecimal: Optional[Decimal] = Field(
        description="Likelihood of specified outcome",
        default=None,
    )
    probabilityDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for probabilityDecimal extensions",
        default=None,
        alias="_probabilityDecimal",
    )
    probabilityRange: Optional[Range] = Field(
        description="Likelihood of specified outcome",
        default=None,
    )
    qualitativeRisk: Optional[CodeableConcept] = Field(
        description="Likelihood of specified outcome as a qualitative value",
        default=None,
    )
    relativeRisk: Optional[Decimal] = Field(
        description="Relative likelihood",
        default=None,
    )
    relativeRisk_ext: Optional[Element] = Field(
        description="Placeholder element for relativeRisk extensions",
        default=None,
        alias="_relativeRisk",
    )
    whenPeriod: Optional[Period] = Field(
        description="Timeframe or age range",
        default=None,
    )
    whenRange: Optional[Range] = Field(
        description="Timeframe or age range",
        default=None,
    )
    rationale: Optional[String] = Field(
        description="Explanation of prediction",
        default=None,
    )
    rationale_ext: Optional[Element] = Field(
        description="Placeholder element for rationale extensions",
        default=None,
        alias="_rationale",
    )

    @property
    def probability(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="probability",
        )

    @property
    def when(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="when",
        )

    @field_validator(
        *(
            "rationale",
            "relativeRisk",
            "qualitativeRisk",
            "outcome",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def probability_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Decimal, Range],
            field_name_base="probability",
            required=False,
        )

    @model_validator(mode="after")
    def when_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Period, Range],
            field_name_base="when",
            required=False,
        )


class RiskAssessment(DomainResource):
    """
    An assessment of the likely outcome(s) for a patient or other subject as well as the likelihood of each outcome.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/RiskAssessment"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Unique identifier for the assessment",
        default=None,
    )
    basedOn: Optional[Reference] = Field(
        description="Request fulfilled by this assessment",
        default=None,
    )
    parent: Optional[Reference] = Field(
        description="Part of this occurrence",
        default=None,
    )
    status: Optional[Code] = Field(
        description="registered | preliminary | final | amended +",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    method: Optional[CodeableConcept] = Field(
        description="Evaluation mechanism",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Type of assessment",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who/what does assessment apply to?",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Where was assessment performed?",
        default=None,
    )
    occurrenceDateTime: Optional[DateTime] = Field(
        description="When was assessment made?",
        default=None,
    )
    occurrenceDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDateTime extensions",
        default=None,
        alias="_occurrenceDateTime",
    )
    occurrencePeriod: Optional[Period] = Field(
        description="When was assessment made?",
        default=None,
    )
    condition: Optional[Reference] = Field(
        description="Condition assessed",
        default=None,
    )
    performer: Optional[Reference] = Field(
        description="Who did assessment?",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why the assessment was necessary?",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Why the assessment was necessary?",
        default=None,
    )
    basis: Optional[List[Reference]] = Field(
        description="Information used in assessment",
        default=None,
    )
    prediction: Optional[List[RiskAssessmentPrediction]] = Field(
        description="Outcome predicted",
        default=None,
    )
    mitigation: Optional[String] = Field(
        description="How to reduce risk",
        default=None,
    )
    mitigation_ext: Optional[Element] = Field(
        description="Placeholder element for mitigation extensions",
        default=None,
        alias="_mitigation",
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments on the risk assessment",
        default=None,
    )
    resourceType: Literal["RiskAssessment"] = Field(
        description=None,
        default="RiskAssessment",
    )

    @property
    def occurrence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurrence",
        )

    @field_validator(
        *(
            "note",
            "mitigation",
            "prediction",
            "basis",
            "reasonReference",
            "reasonCode",
            "performer",
            "condition",
            "encounter",
            "subject",
            "code",
            "method",
            "status",
            "parent",
            "basedOn",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("prediction",), mode="after", check_fields=None)
    @classmethod
    def FHIR_ras_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="probability.exists($this is decimal) implies (probability as decimal) <= 100",
            human="Must be <= 100",
            key="ras-2",
            severity="error",
        )

    @model_validator(mode="after")
    def occurrence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="occurrence",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
