# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Id,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Annotation,
    BackboneElement,
    RelatedArtifact,
    Expression,
    Duration,
    Range,
    Age,
    Period,
    Timing,
    DomainResource,
)


class RequestGroupActionCondition(BackboneElement):
    """
    An expression that describes applicability criteria, or start/stop conditions for the action.
    """

    kind: Optional[Code] = Field(
        description="applicability | start | stop",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    expression: Optional[Expression] = Field(
        description="Boolean-valued expression",
        default=None,
    )

    @field_validator(
        *(
            "expression",
            "kind",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class RequestGroupActionRelatedAction(BackboneElement):
    """
    A relationship to another action such as "before" or "30-60 minutes after start of".
    """

    actionId: Optional[Id] = Field(
        description="What action this is related to",
        default=None,
    )
    actionId_ext: Optional[Element] = Field(
        description="Placeholder element for actionId extensions",
        default=None,
        alias="_actionId",
    )
    relationship: Optional[Code] = Field(
        description="before-start | before | before-end | concurrent-with-start | concurrent | concurrent-with-end | after-start | after | after-end",
        default=None,
    )
    relationship_ext: Optional[Element] = Field(
        description="Placeholder element for relationship extensions",
        default=None,
        alias="_relationship",
    )
    offsetDuration: Optional[Duration] = Field(
        description="Time offset for the relationship",
        default=None,
    )
    offsetRange: Optional[Range] = Field(
        description="Time offset for the relationship",
        default=None,
    )

    @property
    def offset(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="offset",
        )

    @field_validator(
        *(
            "relationship",
            "actionId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def offset_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Duration, Range],
            field_name_base="offset",
            required=False,
        )


class RequestGroupAction(BackboneElement):
    """
    The actions, if any, produced by the evaluation of the artifact.
    """

    prefix: Optional[String] = Field(
        description="User-visible prefix for the action (e.g. 1. or A.)",
        default=None,
    )
    prefix_ext: Optional[Element] = Field(
        description="Placeholder element for prefix extensions",
        default=None,
        alias="_prefix",
    )
    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[String] = Field(
        description="Short description of the action",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    textEquivalent: Optional[String] = Field(
        description="Static text equivalent of the action, used if the dynamic aspects cannot be interpreted by the receiving system",
        default=None,
    )
    textEquivalent_ext: Optional[Element] = Field(
        description="Placeholder element for textEquivalent extensions",
        default=None,
        alias="_textEquivalent",
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="Code representing the meaning of the action or sub-actions",
        default=None,
    )
    documentation: Optional[List[RelatedArtifact]] = Field(
        description="Supporting documentation for the intended performer of the action",
        default=None,
    )
    condition: Optional[List[RequestGroupActionCondition]] = Field(
        description="Whether or not the action is applicable",
        default=None,
    )
    relatedAction: Optional[List[RequestGroupActionRelatedAction]] = Field(
        description="Relationship to another action",
        default=None,
    )
    timingDateTime: Optional[DateTime] = Field(
        description="When the action should take place",
        default=None,
    )
    timingDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for timingDateTime extensions",
        default=None,
        alias="_timingDateTime",
    )
    timingAge: Optional[Age] = Field(
        description="When the action should take place",
        default=None,
    )
    timingPeriod: Optional[Period] = Field(
        description="When the action should take place",
        default=None,
    )
    timingDuration: Optional[Duration] = Field(
        description="When the action should take place",
        default=None,
    )
    timingRange: Optional[Range] = Field(
        description="When the action should take place",
        default=None,
    )
    timingTiming: Optional[Timing] = Field(
        description="When the action should take place",
        default=None,
    )
    participant: Optional[List[Reference]] = Field(
        description="Who should perform the action",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="create | update | remove | fire-event",
        default=None,
    )
    groupingBehavior: Optional[Code] = Field(
        description="visual-group | logical-group | sentence-group",
        default=None,
    )
    groupingBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for groupingBehavior extensions",
        default=None,
        alias="_groupingBehavior",
    )
    selectionBehavior: Optional[Code] = Field(
        description="any | all | all-or-none | exactly-one | at-most-one | one-or-more",
        default=None,
    )
    selectionBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for selectionBehavior extensions",
        default=None,
        alias="_selectionBehavior",
    )
    requiredBehavior: Optional[Code] = Field(
        description="must | could | must-unless-documented",
        default=None,
    )
    requiredBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for requiredBehavior extensions",
        default=None,
        alias="_requiredBehavior",
    )
    precheckBehavior: Optional[Code] = Field(
        description="yes | no",
        default=None,
    )
    precheckBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for precheckBehavior extensions",
        default=None,
        alias="_precheckBehavior",
    )
    cardinalityBehavior: Optional[Code] = Field(
        description="single | multiple",
        default=None,
    )
    cardinalityBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for cardinalityBehavior extensions",
        default=None,
        alias="_cardinalityBehavior",
    )
    resource: Optional[Reference] = Field(
        description="The target of the action",
        default=None,
    )
    action: Optional[List["RequestGroupAction"]] = Field(
        description="Sub action",
        default=None,
    )

    @property
    def timing(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="timing",
        )

    @field_validator(
        *(
            "action",
            "resource",
            "cardinalityBehavior",
            "precheckBehavior",
            "requiredBehavior",
            "selectionBehavior",
            "groupingBehavior",
            "type",
            "participant",
            "relatedAction",
            "condition",
            "documentation",
            "code",
            "priority",
            "textEquivalent",
            "description",
            "title",
            "prefix",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def timing_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Age, Period, Duration, Range, Timing],
            field_name_base="timing",
            required=False,
        )


class RequestGroup(DomainResource):
    """
    A group of related requests that can be used to capture intended activities that have inter-dependencies such as "give this medication after that one".
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/RequestGroup"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Fulfills plan, proposal, or order",
        default=None,
    )
    replaces: Optional[List[Reference]] = Field(
        description="Request(s) replaced by this request",
        default=None,
    )
    groupIdentifier: Optional[Identifier] = Field(
        description="Composite request this is part of",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | on-hold | revoked | completed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    intent: Optional[Code] = Field(
        description="proposal | plan | directive | order | original-order | reflex-order | filler-order | instance-order | option",
        default=None,
    )
    intent_ext: Optional[Element] = Field(
        description="Placeholder element for intent extensions",
        default=None,
        alias="_intent",
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    code: Optional[CodeableConcept] = Field(
        description="What\u0027s being requested/ordered",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who the request group is about",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Created as part of",
        default=None,
    )
    authoredOn: Optional[DateTime] = Field(
        description="When the request group was authored",
        default=None,
    )
    authoredOn_ext: Optional[Element] = Field(
        description="Placeholder element for authoredOn extensions",
        default=None,
        alias="_authoredOn",
    )
    author: Optional[Reference] = Field(
        description="Device or practitioner that authored the request group",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why the request group is needed",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Why the request group is needed",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional notes about the response",
        default=None,
    )
    action: Optional[List[RequestGroupAction]] = Field(
        description="Proposed actions, if any",
        default=None,
    )
    resourceType: Literal["RequestGroup"] = Field(
        description=None,
        default="RequestGroup",
    )

    @field_validator(
        *(
            "action",
            "note",
            "reasonReference",
            "reasonCode",
            "author",
            "authoredOn",
            "encounter",
            "subject",
            "code",
            "priority",
            "intent",
            "status",
            "groupIdentifier",
            "replaces",
            "basedOn",
            "instantiatesUri",
            "instantiatesCanonical",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("action",), mode="after", check_fields=None)
    @classmethod
    def FHIR_rqg_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="resource.exists() != action.exists()",
            human="Must have resource or action but not both",
            key="rqg-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
