# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    Boolean,
    DateTime,
    Markdown,
    Date,
    Decimal,
    Integer,
    Time,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    Period,
    Coding,
    BackboneElement,
    Quantity,
    Reference,
    Attachment,
    DomainResource,
)


class QuestionnaireItemEnableWhen(BackboneElement):
    """
    A constraint indicating that this item should only be enabled (displayed/allow answers to be captured) when the specified condition is true.
    """

    question: Optional[String] = Field(
        description="Question that determines whether item is enabled",
        default=None,
    )
    question_ext: Optional[Element] = Field(
        description="Placeholder element for question extensions",
        default=None,
        alias="_question",
    )
    operator: Optional[Code] = Field(
        description="exists | = | != | \u003e | \u003c | \u003e= | \u003c=",
        default=None,
    )
    operator_ext: Optional[Element] = Field(
        description="Placeholder element for operator extensions",
        default=None,
        alias="_operator",
    )
    answerBoolean: Optional[Boolean] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for answerBoolean extensions",
        default=None,
        alias="_answerBoolean",
    )
    answerDecimal: Optional[Decimal] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for answerDecimal extensions",
        default=None,
        alias="_answerDecimal",
    )
    answerInteger: Optional[Integer] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerInteger_ext: Optional[Element] = Field(
        description="Placeholder element for answerInteger extensions",
        default=None,
        alias="_answerInteger",
    )
    answerDate: Optional[Date] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerDate_ext: Optional[Element] = Field(
        description="Placeholder element for answerDate extensions",
        default=None,
        alias="_answerDate",
    )
    answerDateTime: Optional[DateTime] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for answerDateTime extensions",
        default=None,
        alias="_answerDateTime",
    )
    answerTime: Optional[Time] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerTime_ext: Optional[Element] = Field(
        description="Placeholder element for answerTime extensions",
        default=None,
        alias="_answerTime",
    )
    answerString: Optional[String] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerString_ext: Optional[Element] = Field(
        description="Placeholder element for answerString extensions",
        default=None,
        alias="_answerString",
    )
    answerCoding: Optional[Coding] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerQuantity: Optional[Quantity] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )
    answerReference: Optional[Reference] = Field(
        description="Value for question comparison based on operator",
        default=None,
    )

    @property
    def answer(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="answer",
        )

    @field_validator(
        *(
            "operator",
            "question",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def answer_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Boolean,
                Decimal,
                Integer,
                Date,
                DateTime,
                Time,
                String,
                Coding,
                Quantity,
                Reference,
            ],
            field_name_base="answer",
            required=True,
        )


class QuestionnaireItemAnswerOption(BackboneElement):
    """
    One of the permitted answers for a "choice" or "open-choice" question.
    """

    valueInteger: Optional[Integer] = Field(
        description="Answer value",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDate: Optional[Date] = Field(
        description="Answer value",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueTime: Optional[Time] = Field(
        description="Answer value",
        default=None,
    )
    valueTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueTime extensions",
        default=None,
        alias="_valueTime",
    )
    valueString: Optional[String] = Field(
        description="Answer value",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueCoding: Optional[Coding] = Field(
        description="Answer value",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="Answer value",
        default=None,
    )
    initialSelected: Optional[Boolean] = Field(
        description="Whether option is selected by default",
        default=None,
    )
    initialSelected_ext: Optional[Element] = Field(
        description="Placeholder element for initialSelected extensions",
        default=None,
        alias="_initialSelected",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("initialSelected", "modifierExtension", "extension"),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Integer, Date, Time, String, Coding, Reference],
            field_name_base="value",
            required=True,
        )


class QuestionnaireItemInitial(BackboneElement):
    """
    One or more values that should be pre-populated in the answer when initially rendering the questionnaire for user input.
    """

    valueBoolean: Optional[Boolean] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )
    valueInteger: Optional[Integer] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDate: Optional[Date] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valueTime: Optional[Time] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueTime extensions",
        default=None,
        alias="_valueTime",
    )
    valueString: Optional[String] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueUri: Optional[Uri] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueUri_ext: Optional[Element] = Field(
        description="Placeholder element for valueUri extensions",
        default=None,
        alias="_valueUri",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueCoding: Optional[Coding] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Actual value for initializing the question",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="Actual value for initializing the question",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Boolean,
                Decimal,
                Integer,
                Date,
                DateTime,
                Time,
                String,
                Uri,
                Attachment,
                Coding,
                Quantity,
                Reference,
            ],
            field_name_base="value",
            required=True,
        )


class QuestionnaireItem(BackboneElement):
    """
    A particular question, question grouping or display text that is part of the questionnaire.
    """

    linkId: Optional[String] = Field(
        description="Unique id for item in questionnaire",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    definition: Optional[Uri] = Field(
        description="ElementDefinition - details for the item",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    code: Optional[List[Coding]] = Field(
        description="Corresponding concept for this item in a terminology",
        default=None,
    )
    prefix: Optional[String] = Field(
        description='E.g. "1(a)", "2.5.3"',
        default=None,
    )
    prefix_ext: Optional[Element] = Field(
        description="Placeholder element for prefix extensions",
        default=None,
        alias="_prefix",
    )
    text: Optional[String] = Field(
        description="Primary text for the item",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    type: Optional[Code] = Field(
        description="group | display | boolean | decimal | integer | date | dateTime +",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    enableWhen: Optional[List[QuestionnaireItemEnableWhen]] = Field(
        description="Only allow data when",
        default=None,
    )
    enableBehavior: Optional[Code] = Field(
        description="all | any",
        default=None,
    )
    enableBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for enableBehavior extensions",
        default=None,
        alias="_enableBehavior",
    )
    required: Optional[Boolean] = Field(
        description="Whether the item must be included in data results",
        default=None,
    )
    required_ext: Optional[Element] = Field(
        description="Placeholder element for required extensions",
        default=None,
        alias="_required",
    )
    repeats: Optional[Boolean] = Field(
        description="Whether the item may repeat",
        default=None,
    )
    repeats_ext: Optional[Element] = Field(
        description="Placeholder element for repeats extensions",
        default=None,
        alias="_repeats",
    )
    readOnly: Optional[Boolean] = Field(
        description="Don\u0027t allow human editing",
        default=None,
    )
    readOnly_ext: Optional[Element] = Field(
        description="Placeholder element for readOnly extensions",
        default=None,
        alias="_readOnly",
    )
    maxLength: Optional[Integer] = Field(
        description="No more than this many characters",
        default=None,
    )
    maxLength_ext: Optional[Element] = Field(
        description="Placeholder element for maxLength extensions",
        default=None,
        alias="_maxLength",
    )
    answerValueSet: Optional[Canonical] = Field(
        description="Valueset containing permitted answers",
        default=None,
    )
    answerValueSet_ext: Optional[Element] = Field(
        description="Placeholder element for answerValueSet extensions",
        default=None,
        alias="_answerValueSet",
    )
    answerOption: Optional[List[QuestionnaireItemAnswerOption]] = Field(
        description="Permitted answer",
        default=None,
    )
    initial: Optional[List[QuestionnaireItemInitial]] = Field(
        description="Initial value(s) when item is first rendered",
        default=None,
    )
    item: Optional[List["QuestionnaireItem"]] = Field(
        description="Nested questionnaire items",
        default=None,
    )

    @field_validator(
        *(
            "item",
            "initial",
            "answerOption",
            "answerValueSet",
            "maxLength",
            "readOnly",
            "repeats",
            "required",
            "enableBehavior",
            "enableWhen",
            "type",
            "text",
            "prefix",
            "code",
            "definition",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("enableWhen",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_7_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="operator = 'exists' implies (answer is boolean)",
            human="If the operator is 'exists', the value must be a boolean",
            key="que-7",
            severity="error",
        )


class Questionnaire(DomainResource):
    """
    A structured set of questions intended to guide the collection of answers from end-users. Questionnaires provide detailed control over order, presentation, phraseology and grouping to allow coherent, consistent data collection.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Questionnaire"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this questionnaire, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the questionnaire",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the questionnaire",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this questionnaire (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this questionnaire (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    derivedFrom: Optional[List[Canonical]] = Field(
        description="Instantiates protocol or definition",
        default=None,
    )
    derivedFrom_ext: Optional[Element] = Field(
        description="Placeholder element for derivedFrom extensions",
        default=None,
        alias="_derivedFrom",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    subjectType: Optional[List[Code]] = Field(
        description="Resource that can be subject of QuestionnaireResponse",
        default=None,
    )
    subjectType_ext: Optional[Element] = Field(
        description="Placeholder element for subjectType extensions",
        default=None,
        alias="_subjectType",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the questionnaire",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for questionnaire (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this questionnaire is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    approvalDate: Optional[Date] = Field(
        description="When the questionnaire was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the questionnaire was last reviewed",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the questionnaire is expected to be used",
        default=None,
    )
    code: Optional[List[Coding]] = Field(
        description="Concept that represents the overall questionnaire",
        default=None,
    )
    item: Optional[List[QuestionnaireItem]] = Field(
        description="Questions and sections within the Questionnaire",
        default=None,
    )
    resourceType: Literal["Questionnaire"] = Field(
        description=None,
        default="Questionnaire",
    )

    @field_validator(
        *(
            "item",
            "code",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "subjectType",
            "experimental",
            "status",
            "derivedFrom",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_1a_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(type='group' and %resource.status='complete') implies item.empty().not()",
            human="Group items must have nested items when Questionanire is complete",
            key="que-1a",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_1b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type='group' implies item.empty().not()",
            human="Groups should have items",
            key="que-1b",
            severity="warning",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_1c_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type='display' implies item.empty()",
            human="Display items cannot have child items",
            key="que-1c",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type!='display' or code.empty()",
            human='Display items cannot have a "code" asserted',
            key="que-3",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_4_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="answerOption.empty() or answerValueSet.empty()",
            human="A question cannot have both answerOption and answerValueSet",
            key="que-4",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(type ='choice' or type = 'open-choice' or type = 'decimal' or type = 'integer' or type = 'date' or type = 'dateTime' or type = 'time' or type = 'string' or type = 'quantity') or (answerValueSet.empty() and answerOption.empty())",
            human="Only 'choice' and 'open-choice' items can have answerValueSet",
            key="que-5",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_6_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type!='display' or (required.empty() and repeats.empty())",
            human="Required and repeat aren't permitted for display items",
            key="que-6",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_8_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(type!='group' and type!='display') or initial.empty()",
            human="Initial values can't be specified for groups or display items",
            key="que-8",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_9_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type!='display' or readOnly.empty()",
            human='Read-only can\'t be specified for "display" items',
            key="que-9",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_10_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(type in ('boolean' | 'decimal' | 'integer' | 'string' | 'text' | 'url' | 'open-choice')) or maxLength.empty()",
            human="Maximum length can only be declared for simple question types",
            key="que-10",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_11_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="answerOption.empty() or initial.empty()",
            human="If one or more answerOption is present, initial[x] must be missing",
            key="que-11",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_12_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="enableWhen.count() > 1 implies enableBehavior.exists()",
            human="If there are more than one enableWhen, enableBehavior must be specified",
            key="que-12",
            severity="error",
        )

    @field_validator(*("item",), mode="after", check_fields=None)
    @classmethod
    def FHIR_que_13_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="repeats=true or initial.count() <= 1",
            human="Can only have multiple initial values for repeating items",
            key="que-13",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_que_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="que-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_que_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="descendants().linkId.isDistinct()",
            human="The link ids for groups and questions must be unique within the questionnaire",
            key="que-2",
            severity="error",
        )
