# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Instant,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Reference,
    Period,
    CodeableConcept,
    BackboneElement,
    Signature,
    DomainResource,
)


class ProvenanceAgent(BackboneElement):
    """
    An actor taking a role in an activity  for which it can be assigned some degree of responsibility for the activity taking place.
    """

    type: Optional[CodeableConcept] = Field(
        description="How the agent participated",
        default=None,
    )
    role: Optional[List[CodeableConcept]] = Field(
        description="What the agents role was",
        default=None,
    )
    who: Optional[Reference] = Field(
        description="Who participated",
        default=None,
    )
    onBehalfOf: Optional[Reference] = Field(
        description="Who the agent is representing",
        default=None,
    )

    @field_validator(
        *(
            "onBehalfOf",
            "who",
            "role",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ProvenanceEntityAgent(BackboneElement):
    """
    The entity is attributed to an agent to express the agent's responsibility for that entity, possibly along with other agents. This description can be understood as shorthand for saying that the agent was responsible for the activity which generated the entity.
    """

    type: Optional[CodeableConcept] = Field(
        description="How the agent participated",
        default=None,
    )
    role: Optional[List[CodeableConcept]] = Field(
        description="What the agents role was",
        default=None,
    )
    who: Optional[Reference] = Field(
        description="Who participated",
        default=None,
    )
    onBehalfOf: Optional[Reference] = Field(
        description="Who the agent is representing",
        default=None,
    )

    @field_validator(
        *(
            "onBehalfOf",
            "who",
            "role",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ProvenanceEntity(BackboneElement):
    """
    An entity used in this activity.
    """

    role: Optional[Code] = Field(
        description="derivation | revision | quotation | source | removal",
        default=None,
    )
    role_ext: Optional[Element] = Field(
        description="Placeholder element for role extensions",
        default=None,
        alias="_role",
    )
    what: Optional[Reference] = Field(
        description="Identity of entity",
        default=None,
    )
    agent: Optional[List[ProvenanceEntityAgent]] = Field(
        description="Entity is attributed to this agent",
        default=None,
    )

    @field_validator(
        *(
            "agent",
            "what",
            "role",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Provenance(DomainResource):
    """
    Provenance of a resource is a record that describes entities and processes involved in producing and delivering or otherwise influencing that resource. Provenance provides a critical foundation for assessing authenticity, enabling trust, and allowing reproducibility. Provenance assertions are a form of contextual metadata and can themselves become important records with their own provenance. Provenance statement indicates clinical significance in terms of confidence in authenticity, reliability, and trustworthiness, integrity, and stage in lifecycle (e.g. Document Completion - has the artifact been legally authenticated), all of which may impact security, privacy, and trust policies.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Provenance"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    target: Optional[List[Reference]] = Field(
        description="Target Reference(s) (usually version specific)",
        default=None,
    )
    occurredPeriod: Optional[Period] = Field(
        description="When the activity occurred",
        default=None,
    )
    occurredDateTime: Optional[DateTime] = Field(
        description="When the activity occurred",
        default=None,
    )
    occurredDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurredDateTime extensions",
        default=None,
        alias="_occurredDateTime",
    )
    recorded: Optional[Instant] = Field(
        description="When the activity was recorded / updated",
        default=None,
    )
    recorded_ext: Optional[Element] = Field(
        description="Placeholder element for recorded extensions",
        default=None,
        alias="_recorded",
    )
    policy: Optional[List[Uri]] = Field(
        description="Policy or plan the activity was defined by",
        default=None,
    )
    policy_ext: Optional[Element] = Field(
        description="Placeholder element for policy extensions",
        default=None,
        alias="_policy",
    )
    location: Optional[Reference] = Field(
        description="Where the activity occurred, if relevant",
        default=None,
    )
    reason: Optional[List[CodeableConcept]] = Field(
        description="Reason the activity is occurring",
        default=None,
    )
    activity: Optional[CodeableConcept] = Field(
        description="Activity that occurred",
        default=None,
    )
    agent: Optional[List[ProvenanceAgent]] = Field(
        description="Actor involved",
        default=None,
    )
    entity: Optional[List[ProvenanceEntity]] = Field(
        description="An entity used in this activity",
        default=None,
    )
    signature: Optional[List[Signature]] = Field(
        description="Signature on target",
        default=None,
    )
    resourceType: Literal["Provenance"] = Field(
        description=None,
        default="Provenance",
    )

    @property
    def occurred(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurred",
        )

    @field_validator(
        *(
            "signature",
            "entity",
            "agent",
            "activity",
            "reason",
            "location",
            "policy",
            "recorded",
            "target",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurred_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Period, DateTime],
            field_name_base="occurred",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
