# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Time,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Period,
    Reference,
    CodeableConcept,
    ContactPoint,
    BackboneElement,
    DomainResource,
)


class PractitionerRoleAvailableTime(BackboneElement):
    """
    A collection of times the practitioner is available or performing this role at the location and/or healthcareservice.
    """

    daysOfWeek: Optional[List[Code]] = Field(
        description="mon | tue | wed | thu | fri | sat | sun",
        default=None,
    )
    daysOfWeek_ext: Optional[Element] = Field(
        description="Placeholder element for daysOfWeek extensions",
        default=None,
        alias="_daysOfWeek",
    )
    allDay: Optional[Boolean] = Field(
        description="Always available? e.g. 24 hour service",
        default=None,
    )
    allDay_ext: Optional[Element] = Field(
        description="Placeholder element for allDay extensions",
        default=None,
        alias="_allDay",
    )
    availableStartTime: Optional[Time] = Field(
        description="Opening time of day (ignored if allDay = true)",
        default=None,
    )
    availableStartTime_ext: Optional[Element] = Field(
        description="Placeholder element for availableStartTime extensions",
        default=None,
        alias="_availableStartTime",
    )
    availableEndTime: Optional[Time] = Field(
        description="Closing time of day (ignored if allDay = true)",
        default=None,
    )
    availableEndTime_ext: Optional[Element] = Field(
        description="Placeholder element for availableEndTime extensions",
        default=None,
        alias="_availableEndTime",
    )

    @field_validator(
        *(
            "availableEndTime",
            "availableStartTime",
            "allDay",
            "daysOfWeek",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PractitionerRoleNotAvailable(BackboneElement):
    """
    The practitioner is not available or performing this role during this period of time due to the provided reason.
    """

    description: Optional[String] = Field(
        description="Reason presented to the user explaining why time not available",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    during: Optional[Period] = Field(
        description="Service not available from this date",
        default=None,
    )

    @field_validator(
        *(
            "during",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PractitionerRole(DomainResource):
    """
    A specific set of Roles/Locations/specialties/services that a practitioner may perform at an organization for a period of time.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/PractitionerRole"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifiers that are specific to a role/location",
        default=None,
    )
    active: Optional[Boolean] = Field(
        description="Whether this practitioner role record is in active use",
        default=None,
    )
    active_ext: Optional[Element] = Field(
        description="Placeholder element for active extensions",
        default=None,
        alias="_active",
    )
    period: Optional[Period] = Field(
        description="The period during which the practitioner is authorized to perform in these role(s)",
        default=None,
    )
    practitioner: Optional[Reference] = Field(
        description="Practitioner that is able to provide the defined services for the organization",
        default=None,
    )
    organization: Optional[Reference] = Field(
        description="Organization where the roles are available",
        default=None,
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="Roles which this practitioner may perform",
        default=None,
    )
    specialty: Optional[List[CodeableConcept]] = Field(
        description="Specific specialty of the practitioner",
        default=None,
    )
    location: Optional[List[Reference]] = Field(
        description="The location(s) at which this practitioner provides care",
        default=None,
    )
    healthcareService: Optional[List[Reference]] = Field(
        description="The list of healthcare services that this worker provides for this role\u0027s Organization/Location(s)",
        default=None,
    )
    telecom: Optional[List[ContactPoint]] = Field(
        description="Contact details that are specific to the role/location/service",
        default=None,
    )
    availableTime: Optional[List[PractitionerRoleAvailableTime]] = Field(
        description="Times the Service Site is available",
        default=None,
    )
    notAvailable: Optional[List[PractitionerRoleNotAvailable]] = Field(
        description="Not available during this time due to provided reason",
        default=None,
    )
    availabilityExceptions: Optional[String] = Field(
        description="Description of availability exceptions",
        default=None,
    )
    availabilityExceptions_ext: Optional[Element] = Field(
        description="Placeholder element for availabilityExceptions extensions",
        default=None,
        alias="_availabilityExceptions",
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Technical endpoints providing access to services operated for the practitioner with this role",
        default=None,
    )
    resourceType: Literal["PractitionerRole"] = Field(
        description=None,
        default="PractitionerRole",
    )

    @field_validator(
        *(
            "endpoint",
            "availabilityExceptions",
            "notAvailable",
            "availableTime",
            "telecom",
            "healthcareService",
            "location",
            "specialty",
            "code",
            "organization",
            "practitioner",
            "period",
            "active",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
